<?php
if (! defined('ABSPATH')) exit; // Exit if accessed directly

/**
 * Centralized logging functionality using a trait.
 *
 * @package Altomatic
 */
trait AltomaticLoggerTrait
{
	/**
	 * Logs a message using error_log, prefixed with the class name.
	 * Checks for a WordPress option 'altomatic_enable_debug_logging' or if WP_DEBUG is true.
	 *
	 * @param string $message The message to log.
	 */
	protected static function log(string $message): void
	{
		// Static variable to store the debug setting. Initialized only once per request.
		static $debug_enabled_global = null;

		// If the debug setting hasn't been checked yet in this request...
		if (null === $debug_enabled_global) {
			// Check a global setting or WP_DEBUG and store the result.
			$debug_enabled_global = (bool) get_option('altomatic_enable_debug_logging', false) || (defined('WP_DEBUG') && WP_DEBUG);
		}

		// Log only if debug is enabled.
		if ($debug_enabled_global) {
			// phpcs:disable WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log(static::class . ': ' . $message); // Use static::class to get the name of the class using the trait
			// phpcs:enable WordPress.PHP.DevelopmentFunctions.error_log_error_log
		}
	}
}
