<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

// Ensure the trait is loaded
require_once plugin_dir_path(__FILE__) . 'trait-logger.php';

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * @since      1.0.0
 * @package    Altomatic
 * @subpackage Altomatic/includes
 */
class Altomatic {
	use AltomaticLoggerTrait;


    /**
     * The unique identifier of this plugin.
     */
    protected $plugin_name;

    /**
     * The current version of the plugin.
     */
    protected $version;

    /**
     * The admin class instance.
     */
    protected $admin;

    /**
     * The image optimizer class instance.
     */
    protected $image_optimizer;

    /**
     * The media library class instance.
     */
    protected $media_library;

    /**
     * Initialize the class and set its properties.
     */
    public function __construct() {
        $this->plugin_name = 'altomatic';
        $this->version = ALTOMATIC_VERSION;

        $this->load_dependencies();
        $this->define_core_hooks();

        $this->admin = new Altomatic_Admin();
        $this->image_optimizer = new Altomatic_Image_Optimizer();

        $this->initialize_srcset_optimizer();

        // Schedule credit check
        if (!wp_next_scheduled('altomatic_check_credits')) {
            wp_schedule_event(time(), 'hourly', 'altomatic_check_credits');
        }
        add_action('altomatic_check_credits', array($this, 'check_credits'));
        $this->initialize_media_library();
    }

    /**
     * Load the required dependencies for this plugin.
     */
    private function load_dependencies() {
        $includes_path = plugin_dir_path(__FILE__);

        // Admin class
        require_once $includes_path . 'class-altomatic-admin.php';

        // Srcset Optimization Class
        require_once $includes_path . 'class-altomatic-srcset.php';

        // Media Library Class
        require_once $includes_path . 'class-altomatic-media-library.php';

        // Image Optimizer Class
        require_once $includes_path . 'class-altomatic-image-optimizer.php';
    }

    /**
     * Register all core hooks for the plugin.
     */
    private function define_core_hooks() {
        // WordPress core hooks
        add_action('rest_api_init', array($this, 'register_rest_routes'));

        // Media Library processing hooks
        add_filter('wp_generate_attachment_metadata', array($this, 'process_image_metadata'), 10, 2);
        add_action('delete_attachment', array($this, 'cleanup_optimized_files'), 10, 1);

        // Add credit limit warning to media library page
        add_action('admin_notices', array($this, 'display_credit_limit_warning'));
    }

    /**
    * Initializes the Srcset Optimizer hooks.
    */
    private function initialize_srcset_optimizer() {
        // Check if the feature is enabled in settings (default to true)
        $enable_srcset_optimization = get_option('altomatic_enable_srcset_optimization', true);

        if ($enable_srcset_optimization) {
            if (class_exists('Altomatic_Optimized_Srcset')) {
                Altomatic_Optimized_Srcset::init();
                $this->log("Srcset Optimization enabled and Altomatic_Optimized_Srcset initialized.");
            } else {
                $this->log("Error: Srcset Optimization is enabled, but Altomatic_Optimized_Srcset class not found after include.");
            }
        } else {
             $this->log("Srcset Optimization is disabled in settings.");
        }
    }

    /**
     * Initialize the Media Library hooks
     */
    private function initialize_media_library() {
        if (class_exists('Altomatic_Media_Library')) {
            $this->media_library = new Altomatic_Media_Library();
            $this->media_library->init();
        }
    }

    //------------------------------------------------------------------
    // Media Processing Methods
    //------------------------------------------------------------------

    /**
     * Process image metadata and optimize additional sizes
     */
    public function process_image_metadata($metadata, $attachment_id) {
        $auto_generate_alt = get_option('altomatic_auto_generate_alt', true);
        $auto_optimize_on_upload = get_option('altomatic_format_on_upload', true);

        try {
            return $this->image_optimizer->optimize_image($metadata, $attachment_id, $auto_generate_alt, $auto_optimize_on_upload);
        } catch (Exception $e) {
            $this->log("Error optimizing image: " . $e->getMessage());
            return $metadata;
        }
    }

    //------------------------------------------------------------------
    // REST API Methods
    //------------------------------------------------------------------

    /**
     * Register REST API routes
     */
    public function register_rest_routes() {


        // Add validate API key endpoint
        register_rest_route('altomatic/v1', '/validate-api', array(
            'methods' => 'POST',
            'callback' => array($this, 'validate_api_key'),
            'permission_callback' => function() {
                return current_user_can('manage_options');
            }
        ));

        // Add get credit info endpoint
        register_rest_route('altomatic/v1', '/credits', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_credit_info'),
            'permission_callback' => function() {
                return current_user_can('manage_options');
            }
        ));
    }

    /**
     * Validate API key endpoint
     */
    public function validate_api_key($request) {
        $api_key = $request->get_param('api_key');
        $api = new Altomatic_API();
        $result = $api->validate_key($api_key);

        if (is_wp_error($result)) {
            return rest_ensure_response(array(
                'success' => false,
                'message' => $result->get_error_message()
            ));
        }

        // Store credit usage information
        if (isset($result['data']['credits'])) {
            update_option('altomatic_credit_usage', $result['data']['credits']['used']);
            update_option('altomatic_credit_limit', $result['data']['credits']['total']);
        }

        return rest_ensure_response(array(
            'success' => true,
            'message' => __('API key is valid', 'altomatic'),
            'data' => $result['data']
        ));
    }

    /**
     * Get credit info endpoint
     */
    public function get_credit_info() {
        $api = new Altomatic_API();
        $result = $api->get_credit_info();

        if (is_wp_error($result)) {
            return rest_ensure_response(array(
                'success' => false,
                'message' => $result->get_error_message()
            ));
        }

        // Store credit usage information
        if (isset($result['data'])) {
            update_option('altomatic_credit_usage', $result['data']['used_credits']);
            update_option('altomatic_credit_limit', $result['data']['total_credits']);
        }

        return rest_ensure_response(array(
            'success' => true,
            'data' => $result['data']
        ));
    }


    //------------------------------------------------------------------
    // Helper Methods
    //------------------------------------------------------------------

    /**
     * The name of the plugin used to uniquely identify it within the context of
     * WordPress and to define internationalization functionality.
     *
     * @return    string    The name of the plugin.
     */
    public function get_plugin_name() {
        return $this->plugin_name;
    }

    /**
     * Retrieve the version number of the plugin.
     *
     * @return    string    The version number of the plugin.
     */
    public function get_version() {
        return $this->version;
    }

    /**
     * Clean up optimized files when an attachment is deleted
     */
    public function cleanup_optimized_files($attachment_id) {
        $this->log("Cleaning up optimized files for attachment {$attachment_id}");

        // Get attachment metadata
        $metadata = wp_get_attachment_metadata($attachment_id);
        if (empty($metadata) || empty($metadata['_altomatic_optimized_sizes'])) {
            return;
        }
        $size_data = $metadata['_altomatic_optimized_sizes'];
        $formats = array('jpeg', 'webp', 'avif');
        $sizes = array_merge(array('full'), array_keys(isset($metadata['sizes']) ? $metadata['sizes'] : array()));

        $original_file = get_attached_file($attachment_id);
        $this->log("Original file: {$original_file}");

        // Loop through each format and size
        foreach ($formats as $format) {
            if (empty($size_data[$format])) {
                continue;
            }

            foreach ($sizes as $size) {
                if (empty($size_data[$format][$size])) {
                    continue;
                }

                $file_path = $size_data[$format][$size]['file'];
                // Don't delete the original file
                if ($file_path === $original_file) {
                    continue;
                }

                $this->log("Deleting optimized file: {$file_path}");

                if (file_exists($file_path)) {
                    $this->log("Deleting optimized file: {$file_path}");
                    wp_delete_file($file_path);
                }
            }
        }

        $this->log("Cleanup completed for attachment {$attachment_id}");
    }

    /**
     * Check credits and update stored values
     */
    public function check_credits() {
        $api = new Altomatic_API();
        $result = $api->get_credit_info();

        if (!is_wp_error($result) && isset($result['data'])) {
            update_option('altomatic_credit_usage', $result['data']['used_credits']);
            update_option('altomatic_credit_limit', $result['data']['total_credits']);
        }
    }

    /**
     * Display credit limit warning on media pages
     */
    public function display_credit_limit_warning() {
        // Only show on media library and bulk optimization pages
        $screen = get_current_screen();
        if (!$screen || !in_array($screen->base, array('upload', 'media_page_altomatic-bulk'))) {
            return;
        }

        // Check if credits are exceeded
        $credit_usage = get_option('altomatic_credit_usage', 0);
        $credit_limit = get_option('altomatic_credit_limit', 50); // Default to free tier limit

        if ($credit_usage > $credit_limit) {
            echo '<div class="notice notice-warning is-dismissible" style="border-left-color: #dba617;">';
            echo '<p><strong>' . esc_html__('Altomatic: Credit Limit Exceeded', 'altomatic') . '</strong></p>';
            echo '<p>' . sprintf(
                /* translators: 1: Current credit usage 2: Credit limit 3: Upgrade link HTML */
                esc_html__('Image optimization and AI alt text generation have been paused because you\'ve used %1$d of %2$d credits. To resume services, please %3$s or wait until your credits reset next month.', 'altomatic'),
                esc_html($credit_usage),
                esc_html($credit_limit),
                '<a href="https://altomatic.ai/profile" target="_blank">' . esc_html__('upgrade your plan', 'altomatic') . '</a>'
            ) . '</p>';
            echo '</div>';
        }
    }
}
