<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

// Ensure the trait is loaded
require_once plugin_dir_path(__FILE__) . 'trait-logger.php';

/**
 * Class to handle replacing srcset URLs with optimized formats.
 */
class Altomatic_Optimized_Srcset {
	use AltomaticLoggerTrait;


    /**
     * Initialize the hooks.
     */
    public static function init() {
        self::log('Initializing Altomatic_Optimized_Srcset');
        // Hook into WordPress's srcset calculation
        add_filter('wp_calculate_image_srcset', [self::class, 'set_optimized_srcset'], 20, 5);
    }

    /**
     * Filters an image's 'srcset' sources to use optimized formats (AVIF > WebP > Keep > Original).
     *
     * @param array  $sources       An array of image sources generated by WordPress.
     * @param array  $size_array    An array [width, height] for the requested context size.
     * @param string $image_src     The 'src' of the image (often corresponds to one of the sources).
     * @param array  $image_meta    The image meta data (contains _altomatic_optimized_sizes).
     * @param int    $attachment_id Image attachment ID.
     * @return array The filtered sources array with URLs replaced by optimized versions.
     */
    public static function set_optimized_srcset($sources, $size_array, $image_src, $image_meta, $attachment_id) {
        // 1. Initial Checks & Get Optimized Data
        if (empty($sources) || !is_array($sources) || empty($image_meta['_altomatic_optimized_sizes'])) {
            self::log("Skipping srcset optimization for attachment {$attachment_id}: No sources or optimization data.", $attachment_id);
            return $sources;
        }
        $optimized_data = $image_meta['_altomatic_optimized_sizes'];
        self::log("Processing srcset for attachment {$attachment_id}.", $attachment_id);

        // 2. Process Each Source URL
        $new_sources = []; // Build a new array to preserve original keys if needed
        foreach ($sources as $key => $source_data) {
            if (empty($source_data['url'])) {
                $new_sources[$key] = $source_data; // Keep original if URL is missing
                continue;
            }

            $original_url = $source_data['url'];

            // 3. Determine the WordPress Size Name for this source
            $size_name = self::determine_size_name_from_url($original_url, $image_meta);

            if (!$size_name) {
                self::log("Could not determine size name for URL: {$original_url}. Keeping original.", $attachment_id);
                $new_sources[$key] = $source_data; // Keep original if size cannot be determined
                continue;
            }

            // 4. Find the Best Optimized URL for this size
            $optimized_url = self::get_best_optimized_url($size_name, $original_url, $optimized_data, $attachment_id);

            // 5. Update the Source Data
            $updated_source_data = $source_data;
            if ($optimized_url !== $original_url) {
                $updated_source_data['url'] = $optimized_url;
                self::log("Replaced '{$size_name}' URL: {$original_url} -> {$optimized_url}", $attachment_id);
            } else {
                 self::log("Kept original '{$size_name}' URL: {$original_url}", $attachment_id);
            }
            $new_sources[$key] = $updated_source_data;
        }

        self::log("Finished processing srcset for attachment {$attachment_id}.", $attachment_id);
        // self::log("Final sources: " . print_r($new_sources, true), $attachment_id); // Uncomment for detailed debugging

        return $new_sources;
    }

    /**
     * Determines the best available optimized URL for a given size.
     * Priority: AVIF > WebP > Optimized Keep > Original.
     *
     * @param string $size_name       The WordPress image size name (e.g., 'medium', 'full').
     * @param string $original_url    The original URL for this size from WordPress.
     * @param array  $optimized_data  The _altomatic_optimized_sizes metadata array.
     * @param int    $attachment_id   Attachment ID for logging.
     * @return string The URL of the best available format.
     */
    private static function get_best_optimized_url($size_name, $original_url, $optimized_data, $attachment_id) {
        $formats_to_check = ['avif', 'webp', 'jpeg'];
        $format_extensions = ['avif' => 'avif', 'webp' => 'webp', 'jpeg' => 'jpeg']; // Keep uses original extension

        foreach ($formats_to_check as $format) {
            if (!empty($optimized_data[$format][$size_name]['file'])) {
                $optimized_file_path = $optimized_data[$format][$size_name]['file'];

                // IMPORTANT ASSUMPTION: The 'file' path stored in metadata MUST be an absolute server path
                // If it's relative, the 'process_image_metadata' function needs to store absolute paths.
                if (file_exists($optimized_file_path)) {
                    // For AVIF/WebP, generate the new URL
                    $new_extension = $format_extensions[$format];
                    $optimized_url = self::replace_url_extension($original_url, $new_extension);
                    self::log("Found '{$format}' for size '{$size_name}'. Using URL: {$optimized_url}", $attachment_id);
                    return $optimized_url;
                } else {
                     self::log("Metadata exists for '{$format}' size '{$size_name}', but file not found at: {$optimized_file_path}", $attachment_id);
                }
            }
        }

        // Fallback if no existing optimized files found
        self::log("No existing optimized file found for size '{$size_name}'. Falling back to original URL.", $attachment_id);
        return $original_url;
    }

    /**
     * Determines the WordPress image size name based on the URL filename and metadata.
     *
     * @param string $url        The URL of the image source.
     * @param array $image_meta The attachment metadata.
     * @return string|null The name of the size ('full', 'medium', etc.) or null.
     */
    private static function determine_size_name_from_url($url, $image_meta) {
        $filename_from_url = basename(strtok($url, '?')); // Get filename, remove query string

        if (empty($filename_from_url)) return null;

        // Check against the main file ('full' size)
        if (!empty($image_meta['file']) && $filename_from_url === basename($image_meta['file'])) {
            return 'full';
        }

        // Check against intermediate sizes
        if (!empty($image_meta['sizes']) && is_array($image_meta['sizes'])) {
            foreach ($image_meta['sizes'] as $size_name => $size_data) {
                if (!empty($size_data['file']) && $filename_from_url === $size_data['file']) {
                    return $size_name;
                }
            }
        }

        return null; // Size name couldn't be reliably determined
    }

    /**
     * Replaces the file extension in a URL, preserving query strings.
     *
     * @param string $url           The original URL.
     * @param string $new_extension The desired new extension (e.g., 'webp', 'avif').
     * @return string The URL with the replaced extension, or original URL on failure.
     */
    private static function replace_url_extension($url, $new_extension) {
         // Handle cases where new_extension might be null (e.g., for 'keep') - should not happen based on calling logic
         if (!$new_extension) return $url;

         // Separate URL base from query string
         $query_string = '';
         $url_base = $url;
         $query_pos = strpos($url, '?');
         if ($query_pos !== false) {
             $url_base = substr($url, 0, $query_pos);
             $query_string = substr($url, $query_pos);
         }

         // Find the last dot for the extension
         $last_dot_pos = strrpos($url_base, '.');
         if ($last_dot_pos === false) {
             return $url; // No extension found
         }

         // Check if the part after the dot looks like a common image extension
         $current_extension = substr($url_base, $last_dot_pos + 1);
         if (!preg_match('/^(jpe?g|png|gif|webp|avif)$/i', $current_extension)) {
             return $url; // Doesn't look like a standard image extension we handle
         }

         // Rebuild the URL
         $new_url_base = substr($url_base, 0, $last_dot_pos) . '.' . $new_extension;
         return $new_url_base . $query_string;
    }
}
