<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

// Ensure the trait is loaded
require_once plugin_dir_path(__FILE__) . 'trait-logger.php';

/**
 * Class to handle media library hooks
 */
class Altomatic_Media_Library {
	use AltomaticLoggerTrait;

    /**
     * Define a constant for enabling debug logging. Set to true to enable.
     */
    const DEBUG_LOGGING = true;

    /**
     * Initialize the hooks.
     */
    public function init() {
        self::log('Initializing Altomatic_Media_Library');

       // Media Library UI hooks
       add_filter('manage_media_columns', [$this, 'add_status_column']);
       add_action('manage_media_custom_column', [$this, 'display_status'], 10, 2);
       add_filter('attachment_fields_to_edit', [$this, 'add_alternative_format_urls'], 10, 2);

       add_action('attachment_submitbox_misc_actions', [$this, 'show_media_info'], 100); // Higher priority to show below WP sections
    }


    /**
     * Add status column to media library
     */
    public function add_status_column($columns) {
        $columns['optimization'] = __('Optimization', 'altomatic');
        return $columns;
    }

    /**
     * Display status in media library column
     */
    public function display_status($column_name, $post_id) {
        if ($column_name !== 'optimization') {
            return;
        }

        // Include the column display template
        $path = plugin_dir_path(dirname(__FILE__)) . 'admin/partials/column-display.php';
        if (file_exists($path)) {
            include $path;
        }
    }

    /**
     * Show optimization information in the media edit sidebar
     */
    public function show_media_info() {
        global $post;

        if (!wp_attachment_is_image($post->ID)) {
            return;
        }

        // Set variables for the template
        $attachment_id = $post->ID;

        // Include the sidebar template
        include plugin_dir_path(dirname(__FILE__)) . 'admin/partials/sidebar-optimization.php';
    }

    /**
     * Add WebP and AVIF URLs to the attachment edit screen
     */
    public function add_alternative_format_urls($form_fields, $post) {
        if (!wp_attachment_is_image($post->ID)) {
            return $form_fields;
        }

        $metadata = wp_get_attachment_metadata($post->ID);

        // Add alt text section
        $form_fields = $this->add_alt_text_field($form_fields, $post);

        // If its already optimized we'll show the optimization summary in the sidebar
        if (!empty($metadata['_altomatic_optimized_sizes'])) {
            // Add the optimization summary to the form fields
            ob_start();
            include plugin_dir_path(dirname(__FILE__)) . 'admin/partials/modal-optimization-summary.php';
            $html = ob_get_clean();

            $form_fields['altomatic_optimization'] = array(
                'label' => __('Altomatic Image Optimizations', 'altomatic'),
                'input' => 'html',
                'html' => $html,
                'helps' => '',
                'show_in_edit' => false, // Don't show in edit screen since we're using the sidebar
                'show_in_modal' => true,
                'required' => false
            );

            // Add WebP and AVIF URL fields
            $form_fields = $this->add_format_url_fields($form_fields, $post, $metadata['_altomatic_optimized_sizes']);
        } else {
            // Add A button to optimize the image
            $form_fields = $this->add_optimize_button($form_fields, $post);
        }

        return $form_fields;
    }

    /**
     * Add WebP and AVIF URL fields to form fields
     */
    private function add_format_url_fields($form_fields, $post, $size_data) {
        // Get attachment metadata for original file info
        $attachment_metadata = wp_get_attachment_metadata($post->ID);
        if (empty($attachment_metadata)) {
            return $form_fields;
        }

        // Get original file extension
        $original_extension = strtolower(pathinfo($attachment_metadata['file'], PATHINFO_EXTENSION));

        // Check if we have any optimized files
        $has_jpeg = !empty($size_data['jpeg']['full']['file']) && file_exists($size_data['jpeg']['full']['file']);
        $has_webp = !empty($size_data['webp']['full']['file']) && file_exists($size_data['webp']['full']['file']);
        $has_avif = (!empty($size_data['avif']['full']['file']) && file_exists($size_data['avif']['full']['file'])) ||
                   (!empty($size_data['avif']['medium']['file']) && file_exists($size_data['avif']['medium']['file']));

        // Add jpeg url if available
        if ($has_jpeg) {
            $form_fields['altomatic_jpeg_url'] = array(
                'label' => __('JPEG URL', 'altomatic'),
                'input' => 'html',
                'html' => $this->get_format_url_field_html($post->ID, 'jpeg', $size_data),
                'helps' => '',
                'show_in_edit' => false,
                'show_in_modal' => true,
                'required' => false
            );
        }

        // WebP URL field
        if ($has_webp) {
            $form_fields['altomatic_webp_url'] = array(
                'label' => __('WebP URL', 'altomatic'),
                'input' => 'html',
                'html' => $this->get_format_url_field_html($post->ID, 'webp', $size_data),
                'helps' => '',
                'show_in_edit' => false,
                'show_in_modal' => true,
                'required' => false
            );
        }

        // AVIF URL field
        if ($has_avif) {
            $avif_size = null;
            if (!empty($size_data['avif']['full']['file']) && file_exists($size_data['avif']['full']['file'])) {
                $avif_size = 'full';
            } else if (!empty($size_data['avif']['medium']['file']) && file_exists($size_data['avif']['medium']['file'])) {
                $avif_size = 'medium';
            }

            $form_fields['altomatic_avif_url'] = array(
                'label' => __('AVIF URL', 'altomatic') . ($avif_size === 'medium' ? ' (medium)' : ''),
                'input' => 'html',
                'html' => $this->get_format_url_field_html($post->ID, 'avif', $size_data),
                'helps' => '',
                'show_in_edit' => false,
                'show_in_modal' => true,
                'required' => false
            );
        }

        return $form_fields;
    }

    /**
     * Get HTML for format URL field using template
     */
    private function get_format_url_field_html($post_id, $format, $size_data) {
        // Set up variables for the template
        ob_start();

        // Include the template with the appropriate variables
        include plugin_dir_path(dirname(__FILE__)) . 'admin/partials/format-url-fields.php';
        return ob_get_clean();
    }


    /**
     * Add alt text field to the media modal
     */
    private function add_alt_text_field($form_fields, $post) {
        // Get alt text information
        $alt_text = get_post_meta($post->ID, '_wp_attachment_image_alt', true);
        $alt_generated_date = get_post_meta($post->ID, '_altomatic_alt_generated', true);
        $metadata = wp_get_attachment_metadata($post->ID);
        $is_processed = !empty($metadata['_altomatic_api_image_id']);
        $is_our_alt_text = !empty($alt_generated_date);

        // Create HTML for alt text section
        ob_start();
        ?>
        <div class="altomatic-alt-text">
            <?php if (!empty($alt_text)): ?>
                <p class="alt-text-preview"><?php echo esc_html($alt_text); ?></p>
                <?php if ($is_our_alt_text): ?>
                    <p class="alt-text-date">
                        <?php
                        printf(
                            /* translators: %s: formatted date */
                            esc_html__('Generated on %s', 'altomatic'),
                            esc_html(date_i18n(get_option('date_format'), strtotime($alt_generated_date)))
                        );
                        ?>
                    </p>
                <?php else: ?>
                    <p class="alt-text-note">
                        <?php esc_html_e('This alt text was manually entered', 'altomatic'); ?>
                    </p>
                <?php endif; ?>
            <?php else: ?>
                <p class="alt-text-preview alt-text-empty"><?php esc_html_e('No alt text available', 'altomatic'); ?></p>
            <?php endif; ?>

            <button type="button" class="button altomatic-regenerate-alt" data-id="<?php echo esc_attr($post->ID); ?>">
                <?php
                if (!$is_processed && empty($alt_text)) {
                    esc_html_e('Generate Alt Text', 'altomatic');
                } else {
                    esc_html_e('Regenerate Alt Text', 'altomatic');
                }
                ?>
            </button>

            <?php if (!$is_processed && !empty($alt_text)): ?>
                <p class="description">
                    <?php esc_html_e('This will replace the existing manual alt text with AI-generated alt text.', 'altomatic'); ?>
                </p>
            <?php endif; ?>
        </div>
        <?php
        $alt_text_html = ob_get_clean();

        // Add alt text section to form fields
        $form_fields['altomatic_alt_text'] = array(
            'label' => esc_html__('Altomatic AI Alt Text', 'altomatic'),
            'input' => 'html',
            'html' => $alt_text_html,
            'helps' => '',
            'show_in_edit' => false,
            'show_in_modal' => true,
            'required' => false
        );

        return $form_fields;
    }


    /**
     * Add a button to optimize the image
     */
    private function add_optimize_button($form_fields, $post) {
        // Create HTML for optimize button section
        ob_start();
        ?>
        <div class="altomatic-optimize-section">
            <button type="button" class="button altomatic-optimize" data-id="<?php echo esc_attr($post->ID); ?>">
                <?php esc_html_e('Optimize Image', 'altomatic'); ?>
            </button>
        </div>
        <?php
        $optimize_html = ob_get_clean();

        $form_fields['altomatic_optimize'] = array(
            'label' => esc_html__('Altomatic Optimize Image', 'altomatic'),
            'input' => 'html',
            'html' => $optimize_html,
            'helps' => '',
            'show_in_edit' => false,
            'show_in_modal' => true,
            'required' => false
        );

        return $form_fields;
    }

    /**
     * Get optimization statistics for an attachment
     */
    public function get_optimization_stats($attachment_id) {
        $stats = array(
            'status' => 'not_optimized',
            'bytes_saved' => 0,
            'percent_saved' => 0,
            'formats_count' => 0,
            'size_data' => array()
        );

        // Get attachment metadata
        $metadata = wp_get_attachment_metadata($attachment_id);
        if (empty($metadata)) {
            return $stats;
        }

        // Check if image is being processed
        $processing = get_transient('altomatic_processing_' . $attachment_id);
        if ($processing) {
            $stats['status'] = 'processing';
            return $stats;
        }

        // Check if this image was processed through our API
        if (empty($metadata['_altomatic_api_image_id'])) {
            return $stats;
        }

        // Check for optimization data
        $size_data = !empty($metadata['_altomatic_optimized_sizes']) ? $metadata['_altomatic_optimized_sizes'] : array();
        if (empty($size_data)) {
            // Check if credits are exceeded
            $credit_usage = get_option('altomatic_credit_usage', 0);
            $credit_limit = get_option('altomatic_credit_limit', 50);
            if ($credit_usage >= $credit_limit) {
                $stats['status'] = 'credits_exceeded';
            }
            return $stats;
        }

        // Calculate total sizes and format count
        $total_original = 0;
        $total_optimized = 0;
        $counted_formats = array();

        // Get original sizes
        if (!empty($metadata['filesize'])) {
            $total_original += $metadata['filesize'];
        }
        if (!empty($metadata['sizes'])) {
            foreach ($metadata['sizes'] as $size_info) {
                if (!empty($size_info['filesize'])) {
                    $total_original += $size_info['filesize'];
                }
            }
        }

        // Check each format
        foreach (['jpeg', 'webp', 'avif'] as $format) {
            if (!empty($size_data[$format])) {
                $format_exists = false;
                foreach ($size_data[$format] as $size_name => $size_info) {
                    // Only count if file exists and has a size
                    if (!empty($size_info['file']) && file_exists($size_info['file']) && !empty($size_info['filesize'])) {
                        $total_optimized += $size_info['filesize'];
                        $format_exists = true;
                    }
                }
                // Only count format if at least one size exists
                if ($format_exists) {
                    $counted_formats[$format] = true;
                }
            }
        }

        // Update stats if any formats were optimized
        $formats_count = count($counted_formats);
        if ($formats_count > 0) {
            $stats['status'] = 'optimized';
            $stats['formats_count'] = $formats_count;
            $stats['bytes_saved'] = $total_original - $total_optimized;
            if ($total_original > 0) {
                $stats['percent_saved'] = ($stats['bytes_saved'] / $total_original) * 100;
            }
            $stats['size_data'] = $size_data;
        }

        // Check for errors
        $error = get_post_meta($attachment_id, '_altomatic_error', true);
        if (!empty($error)) {
            $stats['status'] = 'error';
            $stats['error'] = $error;
        }

        return $stats;
    }
}
