<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Class to handle media library hooks
 */
class Altomatic_Image_Optimizer {
	use AltomaticLoggerTrait;


    /**
     * The API class instance.
     */
    protected Altomatic_API $api;

    /**
     * Constructor
     */
    public function __construct() {
        $this->log('Initializing Altomatic_Image_Optimizer');

        $this->api = new Altomatic_API();
    }

    /**
     * @param mixed $metadata
     * @param mixed $attachment_id
     * @param mixed $generate_alt_text
     * @param mixed $generate_optimized_images
     * @return mixed
     */
    public function optimize_image($metadata, $attachment_id, $generate_alt_text, $generate_optimized_images) {
        if (!wp_attachment_is_image($attachment_id)) {
            return $metadata;
        }

        // Get the original file path
        $original_file = get_attached_file($attachment_id);
        if (!$original_file) {
            return $metadata;
        }

        // Get settings
        $selected_sizes = (array) get_option('altomatic_image_sizes', get_intermediate_image_sizes());
        $selected_formats = (array) get_option('altomatic_format_conversion', array('jpeg', 'webp', 'avif'));
        $seo_keywords = $this->merge_seo_keywords(
            get_option('altomatic_seo_keywords', ''),
            $attachment_id
        );

        $credits_needed = $this->determine_credits_needed($selected_formats, $selected_sizes, $generate_alt_text, $generate_optimized_images);
        if ($credits_needed == 0) {
            // Nothing to do
            $this->log('No operations needed');
            return $metadata;
        }

        if (!$this->has_enough_credits($credits_needed)) {
            $this->log('Not enough credits. Skipping optimization');
            throw new Exception('Not enough credits');
        }

        // Log key optimization details
        $this->log('Processing image ' . $attachment_id . ': ' . basename($original_file));
        $this->log('Optimizing image: ' . ($generate_optimized_images ? 'Yes' : 'No'));
        $this->log('Generate alt text: ' . ($generate_alt_text ? 'Yes' : 'No'));
        $this->log('Target formats: ' . implode(', ', $selected_formats));
        $this->log('Target sizes: ' . implode(', ', $selected_sizes));
        $this->log('Credits required: ' . $credits_needed);

        // Upload to API
        $upload_result = $this->upload_to_api($original_file);
        if (!$upload_result) {
            return $metadata;
        }

        $links = $upload_result['links'];
        $api_image_id = $upload_result['id'];

        // Save the API image ID to the attachment metadata
        $metadata['_altomatic_api_image_id'] = $api_image_id;

        // Handle alt text generation
        if ($generate_alt_text) {
            $this->handle_alt_text($links, $attachment_id, $seo_keywords);
        }

        // Process each format
        $optimized_sizes = array();

        if ($generate_optimized_images) {
            foreach ($selected_formats as $format) {
                $result = $this->process_format($attachment_id, $original_file, $metadata, $format, $selected_sizes, $links);
                if (!empty($result)) {
                    $optimized_sizes[$format] = $result;
                }
            }
        }

        // Save the optimized sizes to the attachment metadata
        if (!empty($optimized_sizes)) {
            $metadata['_altomatic_optimized_sizes'] = $optimized_sizes;
        }

        // Track total optimizations for review prompts
        $total_optimized = get_option('altomatic_total_optimized', 0);
        update_option('altomatic_total_optimized', $total_optimized + 1);

        return $metadata;
    }

    /**
    * Determine how many credits are needed
    *
    * @param array $selected_formats The selected formats for optimization
    * @param array $selected_sizes The selected sizes for optimization
    * @param bool $generate_alt_text If true, add credits for generating alt text
    * @param bool $generate_optimized_images If true, add credits for generating optimized images
    * @return int The number of credits needed
    */
    private function determine_credits_needed($selected_formats, $selected_sizes, $generate_alt_text, $generate_optimized_images): int {
        $credits_needed = 0;

        if ($generate_optimized_images) {
            foreach ($selected_formats as $format) {
                foreach ($selected_sizes as $size) {
                    // Skip AVIF for sizes other than thumbnail and medium
                    if ($format === 'avif' && !in_array($size, array('thumbnail', 'medium'))) {
                        continue;
                    }
                    $credits_needed += 1;
                }
            }
        }

        if ($generate_alt_text) {
            $credits_needed += 3;
        }

        return $credits_needed;
    }

    /**
     * Check if we have enough credits to process an image
     *
     * @param int $credits_needed The number of credits needed to process the image
     * @return bool True if we have enough credits, false otherwise
     */
    private function has_enough_credits($credits_needed): bool {
        // Check credits
        $credits = $this->api->get_credit_info();
        if (is_wp_error($credits)) {
            $this->log("Error getting credit info: " . $credits->get_error_message());
            return false;
        }

        // Check if we have remaining credits
        if ($credits['data']['remaining_credits'] < $credits_needed) {
            $this->log("Not enough credits, skipping image. Need {$credits_needed} credits. Remaining: {$credits['data']['remaining_credits']}");
            return false;
        }

        return true;
    }


    /**
     * Process a specific format (keep/original, webp, avif)
     * @return array
     * @param mixed $attachment_id
     * @param mixed $original_file
     * @param mixed $metadata
     * @param mixed $format
     * @param mixed $selected_sizes
     * @param mixed $links
     */
    private function process_format($attachment_id, $original_file, $metadata, $format, $selected_sizes, $links): array {
        $this->log("Processing format: {$format}");
        $optimized_sizes = array();

        foreach ($selected_sizes as $size_name) {
            // Skip if webp and size is original and original size is greater than 2,560px (longest side).
            if ($format === 'webp' && $size_name === 'full' && ($metadata['width'] > 2560 || $metadata['height'] > 2560)) {
                $this->log("Skipping webp {$size_name} because it's too large");
                continue;
            }

            // Skip AVIF for sizes other than thumbnail and medium
            if ($format === 'avif' && !in_array($size_name, array('thumbnail', 'medium'))) {
                continue;
            }

            // Get the size file path
            if ($size_name === 'full') {
                $size_file = $original_file;
            } else {
                // Add directory path to the size file
                $size_file = path_join(dirname($original_file), $metadata['sizes'][$size_name]['file']);
            }

            $processed_size = $this->process_size($attachment_id, $size_file, $metadata, $format, $size_name, $links);
            if ($processed_size) {
                $optimized_sizes[$size_name] = $processed_size;
            }
        }

        return $optimized_sizes;
    }

    /**
     * Process a specific size for a format
     * @return bool|array<string,mixed>
     * @param mixed $attachment_id
     * @param mixed $original_file
     * @param mixed $metadata
     * @param mixed $format
     * @param mixed $size_name
     * @param mixed $links
     */
    private function process_size($attachment_id, $original_file, $metadata, $format, $size_name, $links): bool|array {
        $this->log("Processing size: {$size_name} for format: {$format}");


        $api_format = $format;
        $info = pathinfo($original_file);
        $output_file = $info['dirname'] . '/' . $info['filename'] . '.' . $format;


        // Check if API has this format available
        if (empty($links[$api_format])) {
            $this->log("No {$api_format} format available for {$original_file}");
            return false;
        }

        // Set resize parameters for the API
        $query_params = null;
        if ($size_name !== 'full') {
            $query_params = array(
                'width' => $metadata['sizes'][$size_name]['width'],
                'height' => $metadata['sizes'][$size_name]['height'],
                'fit' => 'crop'
            );
        }

        // Process the image
        $result = $this->api->download_and_save_image($links[$api_format], $output_file, $query_params);

        if ($result === true) {
            // Get the filesize AFTER the API call has written the new file
            clearstatcache(true, $output_file); // Clear stat cache for accurate filesize
            $new_filesize = filesize($output_file);

            // Save the new file details to the metadata
            $result = array(
                'file' => $output_file,
                'filesize' => $new_filesize
            );

            update_post_meta($attachment_id, '_altomatic_size_data_' . $size_name . '_' . $format, $result);

            return $result;
        }

        return false;
    }


    /**
     * Upload image to API and get optimization links
     * @return array|WP_Error
     * @param mixed $file
     */
    private function upload_to_api($file): array|WP_Error {
        $this->log("Uploading image to get optimization URLs");
        $upload_result = $this->api->upload_image($file);

        if (is_wp_error($upload_result) || empty($upload_result['links'])) {
            $this->log("Failed to upload image: " . (is_wp_error($upload_result) ? $upload_result->get_error_message() : 'No links returned'));
            return new WP_Error('upload_failed', 'Failed to upload image');
        }

        return $upload_result;
    }

    /**
     * Get focus keywords from active SEO plugin for the parent post of an attachment
     * Supports Yoast SEO, Rank Math, and AIOSEO
     *
     * @param int $attachment_id The attachment ID
     * @return string Comma-separated focus keywords, or empty string
     */
    private function get_seo_plugin_keywords($attachment_id): string {
        $post_id = (int) get_post_field('post_parent', $attachment_id);
        if ($post_id === 0) {
            return '';
        }

        // Yoast SEO
        if (defined('WPSEO_VERSION')) {
            $focus_kw = get_post_meta($post_id, '_yoast_wpseo_focuskw', true);
            if (!empty($focus_kw)) {
                $this->log("Yoast focus keyword for post {$post_id}: {$focus_kw}");
                return $focus_kw;
            }
        }

        // Rank Math
        if (class_exists('RankMath')) {
            $focus_kw = get_post_meta($post_id, 'rank_math_focus_keyword', true);
            if (!empty($focus_kw)) {
                $this->log("Rank Math focus keyword for post {$post_id}: {$focus_kw}");
                return $focus_kw;
            }
        }

        // AIOSEO
        if (defined('AIOSEO_VERSION')) {
            $keyphrases_json = get_post_meta($post_id, '_aioseo_keyphrases', true);
            if (!empty($keyphrases_json)) {
                $keyphrases = json_decode($keyphrases_json, true);
                if (!empty($keyphrases['focus']['keyphrase'])) {
                    $focus_kw = $keyphrases['focus']['keyphrase'];
                    $this->log("AIOSEO focus keyphrase for post {$post_id}: {$focus_kw}");
                    return $focus_kw;
                }
            }
        }

        return '';
    }

    /**
     * Merge global SEO keywords with per-post SEO plugin keywords
     *
     * @param string $global_keywords Global keywords from plugin settings
     * @param int $attachment_id The attachment ID
     * @return string Combined comma-separated keywords
     */
    private function merge_seo_keywords($global_keywords, $attachment_id): string {
        $plugin_keywords = $this->get_seo_plugin_keywords($attachment_id);

        if (empty($global_keywords)) {
            return $plugin_keywords;
        }
        if (empty($plugin_keywords)) {
            return $global_keywords;
        }

        return $global_keywords . ', ' . $plugin_keywords;
    }

    /**
     * Handle alt text generation
     * @return string|null
     * @param mixed $links
     * @param mixed $attachment_id
     */
    private function handle_alt_text($links, $attachment_id, $seo_keywords): string|null {
        if (empty($links['alt_text'])) {
            return null;
        }

        $this->log("Getting alt text from: " . $links['alt_text']);
        $alt_result = $this->api->get_alt_text($links['alt_text'], $seo_keywords);

        if (!is_wp_error($alt_result) && !empty($alt_result['alt_text'])) {
            $alt_text = $alt_result['alt_text'];
            update_post_meta($attachment_id, '_wp_attachment_image_alt', $alt_text);
            update_post_meta($attachment_id, '_altomatic_alt_generated', current_time('mysql'));
            $this->log("Alt text generated successfully: {$alt_text}");
            return $alt_text;
        } else {
            $this->log("Failed to get alt text: " . (is_wp_error($alt_result) ? $alt_result->get_error_message() : 'Unknown error'));
            return null;
        }
    }

    /**
     * Regenerates the alt text for a given attachment. // Optional: Add a brief description of the method's purpose.
     *
     * @param mixed $attachment_id The ID of the attachment.
     * @return array|string The generated alt text (string) or an array containing status/results/error information. // Added @return tag
     */
    public function regenerate_alt_text($attachment_id) {
        $this->log("Regenerating alt text for image: {$attachment_id}");

        // Get metdata
        $metadata = wp_get_attachment_metadata($attachment_id);
        if (!$metadata) {
            $this->log("No metadata found for image: {$attachment_id}");
            return;
        }

        $original_file = get_attached_file($attachment_id);
        if (!$original_file) {
            $this->log("No original file found for image: {$attachment_id}");
            return;
        }

        $upload_result = $this->upload_to_api($original_file);
        if (!$upload_result) {
            return $metadata;
        }

        $seo_keywords = $this->merge_seo_keywords(
            get_option('altomatic_seo_keywords', ''),
            $attachment_id
        );

        $links = $upload_result['links'];

        $alt_text = $this->handle_alt_text($links, $attachment_id, $seo_keywords);
        if (!$alt_text) {
            $this->log("Failed to get alt text: {$attachment_id}");
            return;
       }

        return $alt_text;
    }
}
