<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

// Ensure the trait is loaded
require_once plugin_dir_path(__FILE__) . 'trait-logger.php';

/**
 * Class responsible for interacting with the Altomatic API
 */
class Altomatic_API {
	use AltomaticLoggerTrait;


    /**
     * API endpoint
     */
    private $api_endpoint = 'https://api.altomatic.ai';

    /**
     * API key
     */
    private $api_key;

    /**
     * Maximum retries for API calls
     */
    private $max_retries = 2;

    /**
     * Initialize the class
     */
    public function __construct() {
        $this->api_key = get_option('altomatic_api_key');

        if (empty($this->api_key)) {
            self::log('No API key configured');
            add_action('admin_notices', function() {
                echo '<div class="notice notice-error"><p>' .
                     esc_html__('Altomatic: API key is not configured. Please enter your API key in the', 'altomatic') .
                     ' <a href="' . esc_url(admin_url('options-general.php?page=altomatic')) . '">' .
                     esc_html__('settings', 'altomatic') . '</a>.' .
                     '</p></div>';
            });
        } else {
            self::log('Using API key: ' . substr($this->api_key, 0, 8) . '...');
        }
    }

    /**
     * Get client identification headers for API requests
     */
    private function get_client_headers() {
        return array(
            'X-Altomatic-Client' => 'wordpress_plugin',
            'X-Altomatic-Client-Version' => defined('ALTOMATIC_VERSION') ? ALTOMATIC_VERSION : 'unknown',
        );
    }

    /**
     * Map file extension to API format
     * @param mixed $format
     */
    private function map_format($format) {
        $format = strtolower($format);
        switch ($format) {
            case 'jpg':
                return 'jpeg';
            case 'jpeg':
                return 'jpeg';
            default:
                return $format;
        }
    }

    /**
     * Map API format to file extension
     * @param mixed $format
     */
    private function map_extension($format) {
        $format = strtolower($format);
        switch ($format) {
            case 'jpeg':
                return 'jpg';
            default:
                return $format;
        }
    }

    /**
     * Upload an image to get optimization URLs
     * @param mixed $file_path
     */
    public function upload_image($file_path) {
        $this->log("Starting image optimization for file: $file_path");

        // Get file info for logging
        $file_size = filesize($file_path);
        $image_info = getimagesize($file_path);
        $mime_type = $image_info['mime'];
        $width = $image_info[0];
        $height = $image_info[1];

        $this->log("File size: " . round($file_size / 1024) . " KB");
        $this->log("MIME type: $mime_type");
        $this->log("Image dimensions: {$width}x{$height}");

        // Try binary upload first
        $this->log("Upload URL: " . $this->get_api_url('v1/image'));

        $response = wp_remote_post($this->get_api_url('v1/image'), array(
            'headers' => array_merge(array(
                'Authorization' => 'Bearer ' . $this->api_key,
                'Content-Type' => $mime_type
            ), $this->get_client_headers()),
            'body' => file_get_contents($file_path),
            'timeout' => 30,
            'sslverify' => false
        ));

        if (is_wp_error($response)) {
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);
        if ($code !== 200) {
            return new WP_Error(
                'api_error',
                'Failed to upload image. Status code: ' . $code
            );
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);
        if (!is_array($body) || !isset($body['links'])) {
            return new WP_Error(
                'invalid_response',
                'Invalid response from API'
            );
        }

        // Map the format in links if needed
        $format = strtolower(pathinfo($file_path, PATHINFO_EXTENSION));
        $api_format = $this->map_format($format);

        if ($format !== $api_format && isset($body['links'][$api_format])) {
            // If we have a jpg file but API returns jpeg format, add it as both
            $body['links'][$format] = $body['links'][$api_format];
        }

        return $body;
    }

    /**
     * Validate API key
     * @param mixed $api_key
     */
    public function validate_key($api_key = null) {
        $key_to_validate = $api_key ?? $this->api_key;

        if (empty($key_to_validate)) {
            $this->log('❌ API: No API key provided for validation');
            return new WP_Error('no_api_key', __('No API key provided', 'altomatic'));
        }

        // Use stats endpoint for validation
        $url = $this->get_api_url('v1/stats');
        self::log("Validating key at: {$url}");
        self::log("Using key: " . substr($key_to_validate, 0, 8) . '...');

        $response = wp_remote_get($url, array(
            'headers' => array_merge(array(
                'Authorization' => 'Bearer ' . $key_to_validate,
                'Accept' => 'application/json'
            ), $this->get_client_headers()),
            'timeout' => 15,
            'sslverify' => false,
            'user-agent' => 'WordPress/' . get_bloginfo('version') . '; ' . get_bloginfo('url')
        ));

        if (is_wp_error($response)) {
            $this->log('❌ API: Validation request failed - ' . $response->get_error_message());
            return $response;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        self::log("Validation response - Status: {$status_code}");
        self::log("Validation body: {$body}");

        if ($status_code === 200 && !empty($data)) {
            $this->log('✅ API: Key validation successful');
            return array(
                'valid' => true,
                'message' => __('API key is valid', 'altomatic'),
                'data' => array(
                    'credits' => $data['credits'],
                    'optimized_images' => $data['optimized_images'],
                    'alt_text_generations' => $data['alt_text_generations']
                )
            );
        }

        $this->log('❌ API: Key validation failed - Status: ' . $status_code);

        return new WP_Error(
            'invalid_api_key',
            __('Invalid API key', 'altomatic'),
            array('status' => $status_code)
        );
    }

    /**
     * Constructs a full API URL from a given endpoint
     * @param mixed $endpoint
     */
    private function get_api_url($endpoint) {
        // Make sure to add v1 prefix if not already present
        if (strpos($endpoint, 'v1/') !== 0) {
            $endpoint = 'v1/' . $endpoint;
        }
        return $this->api_endpoint . '/' . ltrim($endpoint, '/');
    }

    /**
     * Download and save an image from a URL
     * @param mixed $url
     * @param mixed $file_path
     * @param mixed $query_params
     */
    public function download_and_save_image($url, $file_path, $query_params = array()) {
        if (!empty($query_params)) {
            $url = add_query_arg($query_params, $url);
        }

        self::log("Downloading image from {$url} to {$file_path}");

        $response = wp_remote_get($url, array(
            'headers' => array_merge(array(
                'Authorization' => 'Bearer ' . $this->api_key,
                'Accept' => 'image/*'
            ), $this->get_client_headers()),
            'timeout' => 30,
            'sslverify' => false
        ));

        if (is_wp_error($response)) {
            self::log("Download failed: " . $response->get_error_message());
            return $response;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code !== 200) {
            self::log("Download failed - HTTP {$status_code} - {$url}");
            return new WP_Error('download_failed', "HTTP error {$status_code}");
        }

        $content = wp_remote_retrieve_body($response);
        if (empty($content)) {
            self::log("Empty response body");
            return new WP_Error('empty_response', 'Empty response body');
        }

        // Create directory if it doesn't exist
        $dir = dirname($file_path);
        if (!file_exists($dir)) {
            wp_mkdir_p($dir);
        }

        // Save the file
        $result = file_put_contents($file_path, $content);
        if ($result === false) {
            self::log("Failed to save file");
            return new WP_Error('save_failed', 'Failed to save file');
        }

        self::log("Successfully downloaded and saved image");
        return true;
    }

    /**
     * Get alt text for an image using the provided URL
     * @param mixed $url
     */
    public function get_alt_text($url, $seo_keywords) {
        // If seo_keywords isn't empty, set it as a query parameter
        if (!empty($seo_keywords)) {
            $query_params = array();
            $query_params['keywords'] = $seo_keywords;
            $url = add_query_arg($query_params, $url);
        }

        $response = wp_remote_get($url, array(
            'headers' => array_merge(array(
                'Authorization' => 'Bearer ' . $this->api_key,
                'Accept' => 'application/json'
            ), $this->get_client_headers()),
            'timeout' => 30,
            'sslverify' => false
        ));

        if (is_wp_error($response)) {
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);
        if ($code !== 200) {
            return new WP_Error(
                'api_error',
                'Failed to get alt text. Status code: ' . $code
            );
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);
        if (!is_array($body) || !isset($body['alt_text'])) {
            return new WP_Error(
                'invalid_response',
                'Invalid response from alt text API'
            );
        }

        return $body;
    }

    /**
     * Get credit information
     */
    public function get_credit_info() {
        // We can just reuse validate_key since it now returns the stats data
        $result = $this->validate_key();

        if (is_wp_error($result)) {
            return $result;
        }

        return array(
            'success' => true,
            'data' => array(
                'remaining_credits' => $result['data']['credits']['remaining'],
                'total_credits' => $result['data']['credits']['total'],
                'used_credits' => $result['data']['credits']['used'],
                'monthly_optimizations' => $result['data']['optimized_images']['monthly'],
                'monthly_alt_text' => $result['data']['alt_text_generations']['monthly']
            )
        );
    }
}
