<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * The admin-specific functionality of the plugin.
 *
 * @package    Altomatic
 * @subpackage Altomatic/admin
 */
class Altomatic_Admin {
    use AltomaticLoggerTrait;

    /**
     * The version of this plugin.
     *
     * @var      string    $version    The current version of this plugin.
     */
    protected $version;

    /**
     * Initialize the class and set its properties.
     */
    public function __construct() {
        // Get version from global ALTOMATIC_VERSION
        $this->version = ALTOMATIC_VERSION;

        add_action('admin_menu', array($this, 'add_plugin_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        add_action('enqueue_block_editor_assets', array($this, 'enqueue_editor_assets'));
        add_action('rest_api_init', array($this, 'register_rest_routes'));

        // Register AJAX handlers
        add_action('wp_ajax_altomatic_get_unoptimized_count', array($this, 'ajax_get_unoptimized_count'));
        add_action('wp_ajax_altomatic_get_missing_alt_count', array($this, 'ajax_get_missing_alt_count'));
        add_action('wp_ajax_altomatic_bulk_optimize', array($this, 'ajax_bulk_optimize'));

        // Login code handlers
        add_action('wp_ajax_altomatic_request_login_code', array($this, 'ajax_request_login_code'));
        add_action('wp_ajax_altomatic_verify_login_code', array($this, 'ajax_verify_login_code'));

        // Review prompt handler
        add_action('wp_ajax_altomatic_review_action', array($this, 'ajax_review_action'));
        add_action('admin_notices', array($this, 'maybe_show_review_notice'));
    }

    /**
     * Register the administration menu for this plugin.
     */
    public function add_plugin_admin_menu() {
        // Add settings page under Settings
        add_options_page(
            'Altomatic Settings',
            'Altomatic',
            'manage_options',
            'altomatic',
            array($this, 'display_plugin_admin_page')
        );

        // Add bulk optimization page under Media
        add_submenu_page(
            'upload.php', // Parent slug (upload.php is Media)
            'Altomatic Bulk', // Page title
            'Altomatic Bulk', // Menu title
            'manage_options', // Capability
            'altomatic-bulk', // Menu slug
            array($this, 'display_bulk_optimization_page') // Function to display the page
        );
    }

    /**
     * Register plugin settings
     */
    public function register_settings() {
        // Remove default WordPress settings notices
        remove_action('admin_notices', 'update_nag', 3);
        add_action('admin_notices', array($this, 'display_settings_errors'));

        // Register settings
        register_setting(
            'altomatic_settings',
            'altomatic_api_key',
            array(
                'type' => 'string',
                'sanitize_callback' => 'sanitize_text_field',
                'default' => '',
            )
        );

        register_setting(
            'altomatic_settings',
            'altomatic_auto_generate_alt',
            array(
                'type' => 'boolean',
                'sanitize_callback' => array($this, 'sanitize_boolean'),
                'default' => true,
            )
        );

        register_setting(
            'altomatic_settings',
            'altomatic_seo_keywords',
            array(
                'type' => 'string',
                'sanitize_callback' => 'sanitize_text_field',
                'default' => '',
            )
        );

        register_setting(
            'altomatic_settings',
            'altomatic_format_on_upload',
            array(
                'type' => 'boolean',
                'sanitize_callback' => array($this, 'sanitize_boolean'),
                'default' => true,
            )
        );

        register_setting(
            'altomatic_settings',
            'altomatic_format_conversion',
            array(
                'type' => 'array',
                'sanitize_callback' => array($this, 'sanitize_format_conversion'),
                'default' => array(), // Remove default to respect empty arrays
            )
        );

        register_setting(
            'altomatic_settings',
            'altomatic_image_sizes',
            array(
                'type' => 'array',
                'sanitize_callback' => array($this, 'sanitize_image_sizes'),
                'default' => array(), // Remove default to respect empty arrays
            )
        );

        register_setting(
            'altomatic_settings',
            'altomatic_enable_srcset_optimization',
            array(
                'type' => 'boolean',
                'sanitize_callback' => array($this, 'sanitize_boolean'),
                'default' => true,
            )
        );

        register_setting(
            'altomatic_settings',
            'altomatic_enable_debug_logging',
            array(
                'type' => 'boolean',
                'sanitize_callback' => array($this, 'sanitize_boolean'),
                'default' => false,
            )
        );
    }

    /**
     * Display settings errors
     */
    public function display_settings_errors() {
        settings_errors('altomatic_messages');
    }

    /**
     * Sanitize format conversion setting
     */
    public function sanitize_format_conversion($value) {
        if (!is_array($value)) {
            return array();
        }

        // If empty array is submitted (all unchecked), respect that choice
        if (empty($value)) {
            return array();
        }

        $allowed = array('jpeg', 'webp', 'avif');
        return array_intersect($value, $allowed);
    }

    /**
     * Sanitize image sizes setting
     */
    public function sanitize_image_sizes($value) {
        if (!is_array($value)) {
            return array();
        }

        // If empty array is submitted (all unchecked), respect that choice
        if (empty($value)) {
            return array();
        }

        $allowed_sizes = array_merge(array('full'), get_intermediate_image_sizes());
        return array_intersect($value, $allowed_sizes);
    }

    /**
     * Sanitize boolean settings
     */
    public function sanitize_boolean($value) {
        return filter_var($value, FILTER_VALIDATE_BOOLEAN);
    }

    /**
     * Enqueue admin scripts and styles
     */
    public function enqueue_admin_scripts($hook) {
        // Enqueue admin styles on all admin pages
        wp_enqueue_style(
            'altomatic-admin',
            plugin_dir_url(dirname(__FILE__)) . 'admin/css/altomatic-admin.css',
            array('thickbox'),
            $this->version
        );

        // Enqueue thickbox
        wp_enqueue_style('thickbox');
        wp_enqueue_script('thickbox');

        // Only load scripts on our bulk optimization page or media pages
        if ($hook == 'media_page_altomatic-bulk' || $hook == 'settings_page_altomatic' || $hook == 'post.php' || $hook == 'upload.php') {
            // Enqueue admin script with wp-api-fetch dependency
            wp_enqueue_script(
                'altomatic-admin',
                plugin_dir_url(dirname(__FILE__)) . 'admin/js/altomatic-admin.js',
                array('jquery', 'wp-api-fetch', 'thickbox'),
                $this->version,
                true
            );


            // Localize script with translations
            wp_localize_script(
                'altomatic-admin',
                'altomaticL10n',
                array(
                    'regenerating' => __('Regenerating...', 'altomatic'),
                    'generatedOn' => __('Generated on', 'altomatic'),
                    'altTextUpdated' => __('Alt text updated successfully', 'altomatic'),
                    'errorRegenerating' => __('Error regenerating alt text', 'altomatic'),
                    'generateAltText' => __('Generate Alt Text', 'altomatic'),
                    'regenerateAltText' => __('Regenerate Alt Text', 'altomatic')
                )
            );
        }

        // If we're on the bulk optimization page, add the bulk-specific data
        if ($hook === 'media_page_altomatic-bulk') {
            wp_enqueue_script(
                'altomatic-bulk',
                plugin_dir_url(dirname(__FILE__)) . 'admin/js/altomatic-bulk.js',
                array('jquery'),
                $this->version,
                true
            );

            wp_localize_script('altomatic-admin', 'altomaticAdmin', array(
                'ajaxurl' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('altomatic_ajax')
            ));
        }

        // On settings page add settings script
        if ($hook === 'settings_page_altomatic') {
            wp_enqueue_script(
                'altomatic-settings',
                plugin_dir_url(dirname(__FILE__)) . 'admin/js/altomatic-settings.js',
                array('jquery'),
                $this->version,
                true
            );

            wp_localize_script('altomatic-settings', 'altomaticSettings', array(
                'ajaxurl' => admin_url('admin-ajax.php'),
                'nonce'   => wp_create_nonce('altomatic_login_code'),
            ));
        }
    }

    /**
     * Enqueue block editor assets
    */
    public function enqueue_editor_assets() {
        // Register and enqueue the script
        wp_register_script(
            'altomatic-format-selector',
            plugin_dir_url(dirname(__FILE__)) . 'js/format-selector.js',
            array(
                'wp-blocks',
                'wp-element',
                'wp-editor',
                'wp-components',
                'wp-compose',
                'wp-hooks',
                'wp-data',
                'wp-block-editor',
                'wp-api-fetch'  // Add wp-api-fetch as a dependency
            ),
            $this->version,
            true
        );

        wp_enqueue_script('altomatic-format-selector');
    }

    /**
     * AJAX handler for getting unoptimized image count
     */
    public function ajax_get_unoptimized_count() {
        check_ajax_referer('altomatic_ajax');

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
            return;
        }

        // Try to get cached count first
        $count = wp_cache_get('altomatic_unoptimized_count');

        if (false === $count) {
            $args = array(
                'post_type' => 'attachment',
                'post_mime_type' => 'image',
                'post_status' => 'inherit',
                'posts_per_page' => -1,
                'fields' => 'ids',
                'meta_query' => array(
                    'relation' => 'OR',
                    array(
                        'key' => '_wp_attachment_metadata',
                        'value' => '_altomatic_optimized_sizes',
                        'compare' => 'NOT LIKE'
                    ),
                    array(
                        'key' => '_wp_attachment_metadata',
                        'compare' => 'NOT EXISTS'
                    )
                ),
                'no_found_rows' => false,
                'update_post_meta_cache' => false,
                'update_post_term_cache' => false
            );

            $query = new WP_Query($args);
            $count = $query->found_posts;

            // Cache for 1 hour
            $this->log('Cache miss for unoptimized image count: ' . $count);
            wp_cache_set('altomatic_unoptimized_count', $count, '', HOUR_IN_SECONDS);
        } else {
            $this->log('Cache hit for unoptimized image count');
        }

        wp_send_json_success(array(
            'count' => (int) $count
        ));
    }

    /**
     * AJAX handler for getting missing alt text count
     */
    public function ajax_get_missing_alt_count() {
        check_ajax_referer('altomatic_ajax');

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
            return;
        }

        // Try to get cached count first
        $count = wp_cache_get('altomatic_missing_alt_count');

        if (false === $count) {
            $args = array(
                'post_type' => 'attachment',
                'post_mime_type' => 'image',
                'post_status' => 'inherit',
                'posts_per_page' => -1,
                'fields' => 'ids',
                'meta_query' => array(
                    array(
                        'key' => '_altomatic_alt_generated',
                        'compare' => 'NOT EXISTS'
                    )
                ),
                'no_found_rows' => false,
                'update_post_meta_cache' => false,
                'update_post_term_cache' => false
            );

            $query = new WP_Query($args);
            $count = $query->found_posts;

            // Cache for 1 hour
            $this->log('Cache miss for missing alt text count: ' . $count);
            wp_cache_set('altomatic_missing_alt_count', $count, '', HOUR_IN_SECONDS);
        } else {
            $this->log('Cache hit for missing alt text count');
        }

        wp_send_json_success(array(
            'count' => (int) $count
        ));
    }

    /**
     * AJAX handler for bulk optimization
     */
    public function ajax_bulk_optimize() {
        $this->log('Starting bulk optimization');
        check_ajax_referer('altomatic_ajax');

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
            return;
        }

        $batch_size = isset($_POST['batch_size']) ? intval($_POST['batch_size']) : 1;
        $this->log('Fetching batch of unoptimized images. batch_size ' . $batch_size);

        $args = array(
            'post_type' => 'attachment',
            'post_mime_type' => 'image',
            'post_status' => 'inherit',
            'posts_per_page' => $batch_size,
            'meta_query' => array(
                'relation' => 'OR',
                array(
                    'key' => '_wp_attachment_metadata',
                    'value' => '_altomatic_optimized_sizes',
                    'compare' => 'NOT LIKE'
                ),
                array(
                    'key' => '_wp_attachment_metadata',
                    'compare' => 'NOT EXISTS'
                )
            ),
            'orderby' => 'ID',
            'order' => 'DESC',
            'update_post_meta_cache' => true,
            'update_post_term_cache' => false
        );

        $query = new WP_Query($args);
        $images = $query->posts;

        $results = array(
            'processed' => 0,
            'successful' => 0,
            'failed' => 0,
            'log' => array()
        );

        $this->log('Processing batch of images: ' . count($images));

        $image_optimizer = new Altomatic_Image_Optimizer();
        $cache_needs_refresh = false;

        foreach ($images as $image) {
            $results['processed']++;

            try {
                // Get current metadata
                $metadata = wp_get_attachment_metadata($image->ID);

                // Check if we already have Altomatic Alt text
                $alt_generated = get_post_meta($image->ID, '_altomatic_alt_generated', true);
                $generate_alt_text = empty($alt_generated);

                // Process the image using our main optimization function
                // Always optimize images, query returns just the images not optimized
                $new_metadata = $image_optimizer->optimize_image($metadata, $image->ID, $generate_alt_text, true);

                if ($new_metadata !== $metadata) {
                    // Update the attachment metadata
                    wp_update_attachment_metadata($image->ID, $new_metadata);
                    $results['successful']++;
                    $results['log'][] = array(
                        'id' => $image->ID,
                        'title' => $image->post_title,
                        'status' => 'success'
                    );
                    $cache_needs_refresh = true;
                } else {
                    $results['failed']++;
                    $results['log'][] = array(
                        'id' => $image->ID,
                        'title' => $image->post_title,
                        'status' => 'error',
                        'message' => 'Failed to process image'
                    );
                }
            } catch (Exception $e) {
                // If exception was not enough credits then end processing
                if ($e->getMessage() === 'Not enough credits') {
                    wp_send_json_error('Not enough credits');
                    return;
                }
                $results['failed']++;
                $results['log'][] = array(
                    'id' => $image->ID,
                    'title' => $image->post_title,
                    'status' => 'error',
                    'message' => $e->getMessage()
                );
            }
        }

        // If we processed any images successfully, invalidate the cache
        if ($cache_needs_refresh) {
            wp_cache_delete('altomatic_unoptimized_count');
            wp_cache_delete('altomatic_missing_alt_count');
        }

        wp_send_json_success($results);
    }

    /**
     * Render the bulk optimization page
     */
    public function display_bulk_optimization_page() {
        if (!current_user_can('manage_options')) {
            return;
        }

        require_once plugin_dir_path(dirname(__FILE__)) . 'admin/partials/bulk-optimization-page.php';
    }

    /**
     * Render the settings page
     */
    public function display_plugin_admin_page() {
        if (!current_user_can('manage_options')) {
            return;
        }

        require_once plugin_dir_path(dirname(__FILE__)) . 'admin/partials/settings-page.php';
    }

    /**
     * Register REST API routes
     */
    public function register_rest_routes() {
        // Add REST API namespace
        $namespace = 'altomatic/v1';

        // Register regenerate alt text endpoint
        register_rest_route($namespace, '/regenerate-alt-text/(?P<id>\d+)', array(
            'methods' => WP_REST_Server::CREATABLE,
            'callback' => array($this, 'regenerate_alt_text'),
            'permission_callback' => function() {
                return current_user_can('upload_files');
            },
            'args' => array(
                'id' => array(
                    'required' => true,
                    'type' => 'integer',
                    'validate_callback' => function($param) {
                        return is_numeric($param) && $param > 0;
                    }
                )
            )
        ));

        // Register optimize image endpoint
        register_rest_route($namespace, '/optimize-image/(?P<id>\d+)', array(
            'methods' => WP_REST_Server::CREATABLE,
            'callback' => array($this, 'optimize_image'),
            'permission_callback' => function() {
                return current_user_can('upload_files');
            },
            'args' => array(
                'id' => array(
                    'required' => true,
                    'type' => 'integer',
                    'validate_callback' => function($param) {
                        return is_numeric($param) && $param > 0;
                    }
                )
            )
        ));
    }

    /**
     * REST API endpoint for regenerating alt text
     */
    public function regenerate_alt_text($request) {
        $attachment_id = $request['id'];

        // Verify this is an image
        if (!wp_attachment_is_image($attachment_id)) {
            return new WP_Error('invalid_image', __('Not a valid image attachment', 'altomatic'), array('status' => 400));
        }

        // Create API instance with error handling
        try {
            $image_optimizer = new Altomatic_Image_Optimizer();
            $alt_text = $image_optimizer->regenerate_alt_text($attachment_id);

            if ($alt_text) {
                return rest_ensure_response(array(
                    'success' => true,
                    'alt_text' => $alt_text,
                    'generated_date' => get_post_meta($attachment_id, '_altomatic_alt_generated', true)
                ));
            } else {
                return new WP_Error('alt_text_error', __('Failed to regenerate alt text', 'altomatic'), array('status' => 500));
            }
        } catch (Exception $e) {
            $this->log('Altomatic: Exception - ' . $e->getMessage());
            return new WP_Error('alt_text_error', $e->getMessage(), array('status' => 500));
        }
    }

    /**
     * REST API endpoint for optimizing an image
     */
    public function optimize_image($request) {
        $attachment_id = $request['id'];

        // Verify this is an image
        if (!wp_attachment_is_image($attachment_id)) {
            return new WP_Error('invalid_image', __('Not a valid image attachment', 'altomatic'), array('status' => 400));
        }

        // Create API instance with error handling
        try {
            $image_optimizer = new Altomatic_Image_Optimizer();
            // Get the current metadata
            $metadata = wp_get_attachment_metadata($attachment_id);
            // Optimize the image but don't generate alt text
            $new_metadata = $image_optimizer->optimize_image($metadata, $attachment_id, false, true);

            if ($new_metadata !== $metadata) {
                // Update the attachment metadata
                wp_update_attachment_metadata($attachment_id, $new_metadata);
                return rest_ensure_response(array(
                    'success' => true,
                    'message' => 'Image optimized successfully'
                ));
            } else {
                return new WP_Error('optimization_error', __('Failed to optimize image', 'altomatic'), array('status' => 500));
            }
        } catch (Exception $e) {
            $this->log('Altomatic: Exception - ' . $e->getMessage());
            return new WP_Error('optimization_error', $e->getMessage(), array('status' => 500));
        }
    }

    /**
     * Maybe show a review notice based on optimization count
     */
    public function maybe_show_review_notice() {
        // Only show on Altomatic pages
        $screen = get_current_screen();
        if (!$screen || !in_array($screen->id, array('settings_page_altomatic', 'media_page_altomatic-bulk'))) {
            return;
        }

        $total = get_option('altomatic_total_optimized', 0);
        $next_prompt_at = get_option('altomatic_review_next_prompt_at', 25);

        // Show if they've reached the threshold
        if ($total >= $next_prompt_at) {
            $this->show_review_notice($total);
        }
    }

    /**
     * Display the review notice
     */
    private function show_review_notice($total) {
        $review_url = 'https://wordpress.org/support/plugin/altomatic/reviews/#new-post';
        $nonce = wp_create_nonce('altomatic_review');
        ?>
        <div class="notice notice-info is-dismissible altomatic-review-notice">
            <p>
                <strong><?php esc_html_e('Altomatic', 'altomatic'); ?>:</strong>
                <?php
                printf(
                    esc_html__("You've optimized %d images! If Altomatic is saving you time, would you mind leaving a quick review?", 'altomatic'),
                    $total
                );
                ?>
                <a href="<?php echo esc_url($review_url); ?>"
                   target="_blank"
                   class="button button-primary button-small altomatic-review-btn"
                   style="margin-left: 10px;">
                    <?php esc_html_e('Leave a Review', 'altomatic'); ?>
                </a>
                <a href="#" class="altomatic-review-dismiss" style="margin-left: 10px;">
                    <?php esc_html_e('Maybe Later', 'altomatic'); ?>
                </a>
            </p>
        </div>
        <script>
        jQuery(document).ready(function($) {
            var reviewNonce = '<?php echo esc_js($nonce); ?>';

            // Click "Leave a Review" - next prompt in 100 more optimizations
            $('.altomatic-review-btn').on('click', function() {
                $.post(ajaxurl, {
                    action: 'altomatic_review_action',
                    type: 'reviewed',
                    nonce: reviewNonce
                });
            });

            // Click "Maybe Later" - next prompt in 25 more optimizations
            $('.altomatic-review-dismiss').on('click', function(e) {
                e.preventDefault();
                $.post(ajaxurl, {
                    action: 'altomatic_review_action',
                    type: 'later',
                    nonce: reviewNonce
                });
                $(this).closest('.notice').fadeOut();
            });

            // Click X button - same as Maybe Later
            $('.altomatic-review-notice').on('click', '.notice-dismiss', function() {
                $.post(ajaxurl, {
                    action: 'altomatic_review_action',
                    type: 'later',
                    nonce: reviewNonce
                });
            });
        });
        </script>
        <?php
    }

    /**
     * AJAX handler for requesting a login code via email.
     */
    public function ajax_request_login_code() {
        check_ajax_referer('altomatic_login_code', 'nonce');

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Unauthorized.', 'altomatic' ) ) );
            return;
        }

        $email = isset( $_POST['email'] ) ? sanitize_email( $_POST['email'] ) : '';
        if ( empty( $email ) || ! is_email( $email ) ) {
            wp_send_json_error( array( 'message' => __( 'Please enter a valid email address.', 'altomatic' ) ) );
            return;
        }

        $user_id = get_current_user_id();

        // Resend cooldown: 30 seconds per user.
        $cooldown_key = 'altomatic_login_cooldown_' . $user_id;
        if ( get_transient( $cooldown_key ) ) {
            wp_send_json_error( array( 'message' => __( 'Please wait before requesting another code.', 'altomatic' ) ) );
            return;
        }

        $response = wp_remote_post( 'https://altomatic.ai/api/auth/code/request', array(
            'body'    => wp_json_encode( array( 'email' => $email ) ),
            'headers' => array( 'Content-Type' => 'application/json' ),
            'timeout' => 15,
        ) );

        if ( is_wp_error( $response ) ) {
            wp_send_json_error( array( 'message' => __( 'Could not reach the Altomatic server. Please try again.', 'altomatic' ) ) );
            return;
        }

        $status_code = wp_remote_retrieve_response_code( $response );
        $body        = json_decode( wp_remote_retrieve_body( $response ), true );

        if ( $status_code < 200 || $status_code >= 300 || empty( $body['challengeToken'] ) ) {
            $error_msg = ! empty( $body['message'] ) ? $body['message'] : __( 'Failed to send verification code.', 'altomatic' );
            wp_send_json_error( array( 'message' => $error_msg ) );
            return;
        }

        // Store challenge data in a transient keyed by user ID (15 min TTL).
        $challenge_key = 'altomatic_login_challenge_' . $user_id;
        set_transient( $challenge_key, array(
            'challengeToken' => $body['challengeToken'],
            'email'          => $email,
            'attempts'       => 0,
        ), 15 * MINUTE_IN_SECONDS );

        // Set resend cooldown (30 seconds).
        set_transient( $cooldown_key, true, 30 );

        wp_send_json_success( array(
            'message' => ! empty( $body['message'] ) ? $body['message'] : __( 'Verification code sent.', 'altomatic' ),
        ) );
    }

    /**
     * AJAX handler for verifying a login code and retrieving the API key.
     */
    public function ajax_verify_login_code() {
        check_ajax_referer('altomatic_login_code', 'nonce');

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => __( 'Unauthorized.', 'altomatic' ) ) );
            return;
        }

        $code  = isset( $_POST['code'] ) ? sanitize_text_field( $_POST['code'] ) : '';
        $email = isset( $_POST['email'] ) ? sanitize_email( $_POST['email'] ) : '';

        if ( empty( $code ) || empty( $email ) ) {
            wp_send_json_error( array( 'message' => __( 'Email and code are required.', 'altomatic' ) ) );
            return;
        }

        $user_id       = get_current_user_id();
        $challenge_key = 'altomatic_login_challenge_' . $user_id;
        $challenge     = get_transient( $challenge_key );

        if ( empty( $challenge ) || empty( $challenge['challengeToken'] ) ) {
            wp_send_json_error( array( 'message' => __( 'Verification session expired. Please request a new code.', 'altomatic' ) ) );
            return;
        }

        // Check email matches.
        if ( $challenge['email'] !== $email ) {
            wp_send_json_error( array( 'message' => __( 'Email does not match the original request.', 'altomatic' ) ) );
            return;
        }

        // Max 5 verify attempts per challenge.
        if ( $challenge['attempts'] >= 5 ) {
            delete_transient( $challenge_key );
            wp_send_json_error( array( 'message' => __( 'Too many attempts. Please request a new code.', 'altomatic' ) ) );
            return;
        }

        // Increment attempt count.
        $challenge['attempts']++;
        set_transient( $challenge_key, $challenge, 15 * MINUTE_IN_SECONDS );

        $response = wp_remote_post( 'https://altomatic.ai/api/auth/code/verify', array(
            'body'    => wp_json_encode( array(
                'email'          => $email,
                'code'           => $code,
                'challengeToken' => $challenge['challengeToken'],
            ) ),
            'headers' => array( 'Content-Type' => 'application/json' ),
            'timeout' => 15,
        ) );

        if ( is_wp_error( $response ) ) {
            wp_send_json_error( array( 'message' => __( 'Could not reach the Altomatic server. Please try again.', 'altomatic' ) ) );
            return;
        }

        $status_code = wp_remote_retrieve_response_code( $response );
        $body        = json_decode( wp_remote_retrieve_body( $response ), true );

        if ( $status_code < 200 || $status_code >= 300 || empty( $body['apiKey'] ) ) {
            $error_msg = ! empty( $body['message'] ) ? $body['message'] : __( 'Invalid verification code.', 'altomatic' );
            wp_send_json_error( array( 'message' => $error_msg ) );
            return;
        }

        // Success: save the API key and clean up.
        update_option( 'altomatic_api_key', sanitize_text_field( $body['apiKey'] ) );
        delete_transient( $challenge_key );

        wp_send_json_success( array(
            'message' => __( 'API key saved successfully!', 'altomatic' ),
            'apiKey'  => $body['apiKey'],
        ) );
    }

    /**
     * AJAX handler for review actions
     */
    public function ajax_review_action() {
        check_ajax_referer('altomatic_review', 'nonce');

        $type = isset($_POST['type']) ? sanitize_text_field($_POST['type']) : 'later';
        $total = get_option('altomatic_total_optimized', 0);

        if ($type === 'reviewed') {
            // They clicked review - give them a longer break (100 more optimizations)
            update_option('altomatic_review_next_prompt_at', $total + 100);
        } else {
            // Maybe Later or X - show again after 25 more optimizations
            update_option('altomatic_review_next_prompt_at', $total + 25);
        }

        wp_send_json_success();
    }
}
