<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Template for the optimization information in the attachment edit sidebar
 *
 * @link       https://altomatic.io
 * @since      1.0.0
 *
 * @package    Altomatic
 * @subpackage Altomatic/admin/partials
 */

// Get credit usage information
$credit_usage = get_option('altomatic_credit_usage', 0);
$credit_limit = get_option('altomatic_credit_limit', 50);
$credits_exceeded = $credit_usage >= $credit_limit;

// Expected variables: $attachment_id
$metadata = wp_get_attachment_metadata($attachment_id);
$size_data = !empty($metadata['_altomatic_optimized_sizes']) ? $metadata['_altomatic_optimized_sizes'] : array();

// Get optimization stats
$stats = $this->get_optimization_stats($attachment_id);

// Display optimization status
?>

<div class="misc-pub-section altomatic-optimization">
    <h4 class="altomatic-section-heading"><?php esc_html_e('Altomatic Image Optimizations', 'altomatic'); ?></h4>

    <?php if ($credits_exceeded): ?>
        <div class="notice notice-warning" style="margin: 5px 0;">
            <p>
                <?php esc_html_e('Credit limit exceeded. Image optimization is paused.', 'altomatic'); ?>
                <a href="https://altomatic.ai/profile" target="_blank"><?php esc_html_e('Upgrade Plan', 'altomatic'); ?></a>
            </p>
        </div>
    <?php endif; ?>

    <?php if ($stats['status'] === 'optimized'): ?>
        <div class="altomatic-status optimized">
            <span class="dashicons dashicons-yes-alt"></span>
            <?php esc_html_e('Optimized', 'altomatic'); ?>
        </div>
        <div class="altomatic-stats">
            <p>
                <?php
                printf(
                    /* translators: %d: Number of formats optimized */
                    esc_html(_n('%d format optimized', '%d formats optimized', $stats['formats_count'], 'altomatic')),
                    esc_html($stats['formats_count'])
                );
                ?>
                <?php if ($stats['bytes_saved'] > 0): ?>
                    <br>
                    <?php
                    printf(
                        /* translators: %s: The size of the image saved */
                        esc_html__('Reduced by %s', 'altomatic'),
                        esc_html(size_format($stats['bytes_saved'], 1))
                    );
                    ?>
                <?php endif; ?>
                <br>
                <a href="#TB_inline?width=600&height=550&inlineId=altomatic-modal-<?php echo esc_attr($attachment_id); ?>"
                   class="thickbox"
                   title="<?php esc_attr_e('Altomatic Compression Details', 'altomatic'); ?>">
                    <?php esc_html_e('Details', 'altomatic'); ?>
                </a>
            </p>
        </div>
    <?php elseif ($stats['status'] === 'processing'): ?>
        <div class="altomatic-status processing">
            <span class="dashicons dashicons-update"></span>
            <?php esc_html_e('Processing', 'altomatic'); ?>
        </div>
    <?php elseif ($stats['status'] === 'error'): ?>
        <div class="altomatic-status error">
            <span class="dashicons dashicons-warning"></span>
            <?php esc_html_e('Error', 'altomatic'); ?>
            <?php if (!empty($stats['error'])): ?>
                <p class="description"><?php echo esc_html($stats['error']); ?></p>
            <?php endif; ?>
        </div>
    <?php else: ?>
        <div class="altomatic-status unoptimized">
            <span class="dashicons dashicons-marker"></span>
            <?php esc_html_e('Not optimized', 'altomatic'); ?>
            <?php if ($credits_exceeded): ?>
                <p class="description"><?php esc_html_e('Optimization unavailable - credit limit exceeded', 'altomatic'); ?></p>
            <?php endif; ?>
        </div>
    <?php endif; ?>
</div>



<?php
// Display alternative format URLs if they exist
if (!empty($size_data)) {
    $has_jpeg = !empty($size_data['jpeg']['full']['file']) && file_exists($size_data['jpeg']['full']['file']);
    $has_webp = !empty($size_data['webp']['full']['file']) && file_exists($size_data['webp']['full']['file']);
    $has_avif = (!empty($size_data['avif']['full']['file']) && file_exists($size_data['avif']['full']['file'])) ||
               (!empty($size_data['avif']['medium']['file']) && file_exists($size_data['avif']['medium']['file']));

    if ($has_jpeg):
        $format = 'jpeg';
        $post_id = $attachment_id;
    ?>
        <div class="misc-pub-section altomatic-format-url">
            <h4><?php esc_html_e('JPEG URL', 'altomatic'); ?></h4>
            <?php include plugin_dir_path(dirname(dirname(__FILE__))) . 'admin/partials/format-url-fields.php'; ?>
        </div>
    <?php endif;

    if ($has_webp):
        $format = 'webp';
        $post_id = $attachment_id;
    ?>
        <div class="misc-pub-section altomatic-format-url">
            <h4><?php esc_html_e('WebP URL', 'altomatic'); ?></h4>
            <?php include plugin_dir_path(dirname(dirname(__FILE__))) . 'admin/partials/format-url-fields.php'; ?>
        </div>
    <?php endif;

    if ($has_avif):
        $avif_size = null;
        $format = 'avif';
        $post_id = $attachment_id;
        if (!empty($size_data['avif']['full']['file']) && file_exists($size_data['avif']['full']['file'])) {
            $avif_size = 'full';
        } else if (!empty($size_data['avif']['medium']['file']) && file_exists($size_data['avif']['medium']['file'])) {
            $avif_size = 'medium';
        }
    ?>
        <div class="misc-pub-section altomatic-format-url">
            <h4>
                <?php esc_html_e('AVIF URL', 'altomatic'); ?>
                <?php if ($avif_size === 'medium'): ?>
                    <span class="altomatic-size-note"><?php esc_html_e('(medium)', 'altomatic'); ?></span>
                <?php endif; ?>
            </h4>
            <?php include plugin_dir_path(dirname(dirname(__FILE__))) . 'admin/partials/format-url-fields.php'; ?>
        </div>
    <?php endif;
} ?>


<?php
// Add alt text section for all images
$alt_text = get_post_meta($attachment_id, '_wp_attachment_image_alt', true);
$alt_generated_date = get_post_meta($attachment_id, '_altomatic_alt_generated', true);
$is_optimized = !empty($metadata['_altomatic_optimized_sizes']);
$is_processed = !empty($metadata['_altomatic_api_image_id']);
$is_our_alt_text = !empty($alt_generated_date);
?>
<div class="misc-pub-section altomatic-alt-text">
    <h4><?php esc_html_e('Altomatic AI Alt Text', 'altomatic'); ?></h4>

    <?php if ($credits_exceeded): ?>
        <div class="notice notice-warning" style="margin: 5px 0;">
            <p>
                <?php esc_html_e('Credit limit exceeded. Alt text generation is paused.', 'altomatic'); ?>
                <a href="https://altomatic.ai/profile" target="_blank"><?php esc_html_e('Upgrade Plan', 'altomatic'); ?></a>
            </p>
        </div>
    <?php endif; ?>

    <?php if (!empty($alt_text)): ?>
        <p class="alt-text-preview"><?php echo esc_html($alt_text); ?></p>
        <?php if ($is_our_alt_text): ?>
            <p class="alt-text-date">
                <?php
                printf(
                    /* translators: %s: The formatted date when the alt text was generated */
                    esc_html__('Generated on %s', 'altomatic'),
                    esc_html(date_i18n(get_option('date_format'), strtotime($alt_generated_date)))
                );
                ?>
            </p>
        <?php else: ?>
            <p class="alt-text-note">
                <?php esc_html_e('This alt text was manually entered', 'altomatic'); ?>
            </p>
        <?php endif; ?>
    <?php else: ?>
        <p class="alt-text-preview alt-text-empty"><?php esc_html_e('No alt text available', 'altomatic'); ?></p>
    <?php endif; ?>

    <button type="button" class="button altomatic-regenerate-alt" data-id="<?php echo esc_attr($attachment_id); ?>" <?php echo $credits_exceeded ? 'disabled' : ''; ?>>
        <?php
        if ($credits_exceeded) {
            esc_html_e('Credits Exceeded', 'altomatic');
        } else if (!$is_processed && empty($alt_text)) {
            esc_html_e('Generate Alt Text', 'altomatic');
        } else {
            esc_html_e('Regenerate Alt Text', 'altomatic');
        }
        ?>
    </button>

    <?php if (!$is_processed && !empty($alt_text)): ?>
        <p class="description">
            <?php esc_html_e('This will replace the existing manual alt text with AI-generated alt text.', 'altomatic'); ?>
        </p>
    <?php endif; ?>
</div>

<?php
// Add the hidden modal for details
if (!empty($size_data)): ?>
    <div id="altomatic-modal-<?php echo esc_attr($attachment_id); ?>" style="display:none;">
        <?php include plugin_dir_path(dirname(dirname(__FILE__))) . 'admin/partials/compression-details.php'; ?>
    </div>
<?php endif;
?>
