<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Settings page template
 *
 * @package Altomatic
 */

// Get registered image sizes
$image_sizes = get_intermediate_image_sizes();

// Add 'full' size to the list of available sizes
$all_sizes = array_merge(['full'], $image_sizes);

// Get selected sizes from options, with no default fallback
$selected_sizes = get_option('altomatic_image_sizes');
if ($selected_sizes === false) {
    // Only use default on first install
    $selected_sizes = $all_sizes;
}

// Handle format conversion option (might be string from previous version)
$format_option = get_option('altomatic_format_conversion');
if ($format_option === false) {
    // Only use default on first install
    $format_option = array('jpeg', 'webp');
}
$selected_formats = is_array($format_option) ? $format_option : array($format_option);
?>
<?php
    // Check if credits are exceeded
    $credit_usage = get_option('altomatic_credit_usage', 0);
    $credit_limit = get_option('altomatic_credit_limit', 50); // Default to free tier limit

    if ($credit_usage > $credit_limit) {
        echo '<div class="notice notice-warning is-dismissible">';
        echo '<p><strong>' . esc_html__('Altomatic: Credit Limit Exceeded', 'altomatic') . '</strong></p>';
        echo '<p>' . sprintf(
            /* translators: 1: Current credit usage 2: Credit limit 3: Upgrade link HTML */
            esc_html__('Image optimization and AI alt text generation have been paused because you\'ve used %1$d of %2$d credits. To resume services, please %3$s or wait until your credits reset next month.', 'altomatic'),
            esc_html($credit_usage),
            esc_html($credit_limit),
            '<a href="https://altomatic.ai/profile" target="_blank">' . esc_html__('upgrade your plan', 'altomatic') . '</a>'
        ) . '</p>';
        echo '</div>';
    }
    ?>
<div class="altomatic-header">
    <div class="altomatic-header-inner">
        <div class="altomatic-header-left">
            <img src="https://altomatic.ai/images/logo_light.svg" alt="Altomatic" class="altomatic-header-logo">
        </div>
        <div class="altomatic-quick-links">
            <a href="<?php echo esc_url(admin_url('upload.php?page=altomatic-bulk')); ?>" class="altomatic-header-link">
                <span class="dashicons dashicons-images-alt2"></span>
                <?php esc_html_e('Bulk Optimization', 'altomatic'); ?>
            </a>
            <a href="https://altomatic.ai/profile" target="_blank" class="altomatic-header-link">
                <span class="dashicons dashicons-admin-users"></span>
                <?php esc_html_e('Manage Account', 'altomatic'); ?>
            </a>
            <a href="mailto:support@altomatic.ai" class="altomatic-header-link">
                <span class="dashicons dashicons-email"></span>
                <?php esc_html_e('Contact Support', 'altomatic'); ?>
            </a>
        </div>
    </div>
</div>

<div class="wrap altomatic">
    <h1 class="screen-reader-text"><?php esc_html_e('Altomatic Settings', 'altomatic'); ?></h1>

    <!-- Info Tabs -->
    <div class="altomatic-info-tabs">
        <div class="altomatic-tab-nav">
            <button class="altomatic-tab active" data-tab="getting-started"><?php esc_html_e('Getting Started', 'altomatic'); ?></button>
            <button class="altomatic-tab" data-tab="how-it-works"><?php esc_html_e('How It Works', 'altomatic'); ?></button>
            <button class="altomatic-tab" data-tab="tips"><?php esc_html_e('Tips & Best Practices', 'altomatic'); ?></button>
        </div>

        <div class="altomatic-tab-content">
            <!-- Getting Started -->
            <div class="tab-pane active" id="getting-started">
                <div class="altomatic-guide-grid">
                    <div class="altomatic-guide-step">
                        <span class="altomatic-step-num">1</span>
                        <div>
                            <strong><?php esc_html_e('Get your API key', 'altomatic'); ?></strong>
                            <p><?php esc_html_e('Click the "Get API Key" button below, enter your email, and we\'ll send you a verification code. Enter the code and your API key is automatically saved. Or paste an existing key from', 'altomatic'); ?> <a href="https://altomatic.ai/profile" target="_blank">altomatic.ai</a>.</p>
                        </div>
                    </div>
                    <div class="altomatic-guide-step">
                        <span class="altomatic-step-num">2</span>
                        <div>
                            <strong><?php esc_html_e('Configure your settings', 'altomatic'); ?></strong>
                            <p><?php esc_html_e('Choose image sizes, select output formats (WebP recommended), and enable AI alt text generation.', 'altomatic'); ?></p>
                        </div>
                    </div>
                    <div class="altomatic-guide-step">
                        <span class="altomatic-step-num">3</span>
                        <div>
                            <strong><?php esc_html_e('Start optimizing', 'altomatic'); ?></strong>
                            <p><?php esc_html_e('New uploads are optimized automatically. Use', 'altomatic'); ?> <a href="<?php echo esc_url(admin_url('upload.php?page=altomatic-bulk')); ?>"><?php esc_html_e('Bulk Optimization', 'altomatic'); ?></a> <?php esc_html_e('for existing images.', 'altomatic'); ?></p>
                        </div>
                    </div>
                </div>
            </div>

            <!-- How It Works -->
            <div class="tab-pane" id="how-it-works" style="display: none;">
                <div class="altomatic-guide-grid">
                    <div class="altomatic-guide-step">
                        <span class="altomatic-step-icon dashicons dashicons-format-image"></span>
                        <div>
                            <strong><?php esc_html_e('Image optimization', 'altomatic'); ?></strong>
                            <p><?php esc_html_e('1 credit per image size. Each format (JPEG, WebP, AVIF) uses 1 credit per size.', 'altomatic'); ?></p>
                        </div>
                    </div>
                    <div class="altomatic-guide-step">
                        <span class="altomatic-step-icon dashicons dashicons-editor-textcolor"></span>
                        <div>
                            <strong><?php esc_html_e('AI alt text', 'altomatic'); ?></strong>
                            <p><?php esc_html_e('3 credits per image for AI-powered alt text generation.', 'altomatic'); ?></p>
                        </div>
                    </div>
                    <div class="altomatic-guide-step">
                        <span class="altomatic-step-icon dashicons dashicons-chart-bar"></span>
                        <div>
                            <strong><?php esc_html_e('Track usage', 'altomatic'); ?></strong>
                            <p><?php esc_html_e('Monitor credits and manage your plan in your', 'altomatic'); ?> <a href="https://altomatic.ai/profile" target="_blank"><?php esc_html_e('Altomatic dashboard', 'altomatic'); ?></a>.</p>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Tips -->
            <div class="tab-pane" id="tips" style="display: none;">
                <div class="altomatic-guide-grid">
                    <div class="altomatic-guide-step">
                        <span class="altomatic-step-icon dashicons dashicons-yes-alt"></span>
                        <div>
                            <strong><?php esc_html_e('Use WebP', 'altomatic'); ?></strong>
                            <p><?php esc_html_e('Best compression with wide browser support. Recommended for all sites.', 'altomatic'); ?></p>
                        </div>
                    </div>
                    <div class="altomatic-guide-step">
                        <span class="altomatic-step-icon dashicons dashicons-warning"></span>
                        <div>
                            <strong><?php esc_html_e('AVIF sparingly', 'altomatic'); ?></strong>
                            <p><?php esc_html_e('Best for thumbnail and medium sizes. Not all browsers support AVIF yet.', 'altomatic'); ?></p>
                        </div>
                    </div>
                    <div class="altomatic-guide-step">
                        <span class="altomatic-step-icon dashicons dashicons-performance"></span>
                        <div>
                            <strong><?php esc_html_e('Enable srcset', 'altomatic'); ?></strong>
                            <p><?php esc_html_e('Keep srcset optimization enabled for seamless responsive image loading.', 'altomatic'); ?></p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Upgrade CTA -->
    <div class="altomatic-upgrade-bar">
        <div class="altomatic-upgrade-text">
            <span class="dashicons dashicons-star-filled"></span>
            <?php esc_html_e('Free plan includes 50 credits/month.', 'altomatic'); ?>
            <a href="https://altomatic.ai/profile" target="_blank"><?php esc_html_e('Upgrade your plan', 'altomatic'); ?> &rarr;</a>
        </div>
    </div>

    <form method="post" action="options.php">
        <?php settings_fields('altomatic_settings'); ?>

        <!-- Hidden fields to ensure empty arrays are submitted -->
        <input type="hidden" name="altomatic_format_conversion[]" value="" disabled>
        <input type="hidden" name="altomatic_image_sizes[]" value="" disabled>

        <table class="form-table" role="presentation">
            <!-- API Key -->
            <tr>
                <th scope="row">
                    <label for="altomatic_api_key"><?php esc_html_e('API Key', 'altomatic'); ?></label>
                </th>
                <td>
                    <div class="api-key-wrapper" style="display: flex; align-items: center; gap: 8px;">
                        <input type="password"
                               id="altomatic_api_key"
                               name="altomatic_api_key"
                               value="<?php echo esc_attr(get_option('altomatic_api_key')); ?>"
                               class="regular-text"
                               autocomplete="off">
                        <span class="status-icon"></span>
                        <button type="button" class="button button-secondary toggle-api-key" aria-label="<?php esc_attr_e('Toggle API key visibility', 'altomatic'); ?>">
                            <span class="dashicons dashicons-visibility"></span>
                        </button>
                        <button type="button" class="button button-primary" id="altomatic-get-api-key">
                            <?php esc_html_e('Get API Key', 'altomatic'); ?>
                        </button>
                    </div>
                    <p class="description"><?php esc_html_e('Enter your Altomatic API key or click "Get API Key" to sign in with your email.', 'altomatic'); ?></p>

                    <!-- Email Code Login Modal -->
                    <div id="altomatic-login-modal" class="altomatic-modal" style="display: none;">
                        <div class="altomatic-modal-backdrop"></div>
                        <div class="altomatic-modal-content">
                            <button type="button" class="altomatic-modal-close" aria-label="<?php esc_attr_e('Close', 'altomatic'); ?>">&times;</button>
                            <h2><?php esc_html_e('Get Your API Key', 'altomatic'); ?></h2>

                            <!-- Step 1: Email -->
                            <div id="altomatic-login-step-email" class="altomatic-login-step">
                                <p><?php esc_html_e('Enter the email address associated with your Altomatic account. We\'ll send you a verification code.', 'altomatic'); ?></p>
                                <label for="altomatic-login-email"><?php esc_html_e('Email address', 'altomatic'); ?></label>
                                <input type="email" id="altomatic-login-email" class="regular-text" placeholder="you@example.com" autocomplete="email">
                                <div class="altomatic-modal-actions">
                                    <button type="button" class="button button-primary" id="altomatic-send-code">
                                        <?php esc_html_e('Send Code', 'altomatic'); ?>
                                    </button>
                                </div>
                            </div>

                            <!-- Step 2: Code -->
                            <div id="altomatic-login-step-code" class="altomatic-login-step" style="display: none;">
                                <p><?php esc_html_e('Enter the 6-digit verification code sent to your email.', 'altomatic'); ?></p>
                                <label for="altomatic-login-code"><?php esc_html_e('Verification code', 'altomatic'); ?></label>
                                <input type="text" id="altomatic-login-code" class="regular-text" placeholder="000000" maxlength="6" autocomplete="one-time-code" inputmode="numeric" pattern="[0-9]*">
                                <div class="altomatic-modal-actions">
                                    <button type="button" class="button button-primary" id="altomatic-verify-code">
                                        <?php esc_html_e('Verify', 'altomatic'); ?>
                                    </button>
                                    <button type="button" class="button button-secondary" id="altomatic-resend-code">
                                        <?php esc_html_e('Resend Code', 'altomatic'); ?>
                                    </button>
                                </div>
                            </div>

                            <!-- Status message area -->
                            <div id="altomatic-login-status" class="altomatic-login-status"></div>
                        </div>
                    </div>
                    <!-- Nonce passed via wp_localize_script in altomaticSettings.nonce -->

                    <!-- API Status Section -->
                    <div class="altomatic-api-status" style="display: none;">
                        <div class="api-credit-info">
                            <div class="credit-usage">
                                <span class="credit-count"></span>
                            </div>
                            <div class="credit-period">
                                <span class="period-text"></span>
                            </div>
                        </div>
                    </div>
                </td>
            </tr>

            <!-- Alt Text Generation -->
            <tr>
                <th scope="row">
                    <label for="altomatic_auto_generate_alt"><?php esc_html_e('Alt Text Generation', 'altomatic'); ?></label>
                </th>
                <td>
                    <label>
                        <input type="checkbox"
                               id="altomatic_auto_generate_alt"
                               name="altomatic_auto_generate_alt"
                               value="1"
                               <?php checked(get_option('altomatic_auto_generate_alt', true), true); ?>>
                        <?php esc_html_e('Automatically generate alt text for images', 'altomatic'); ?>
                        <br><br>
                        <p class="" style="padding-bottom: 5px;">Custom SEO keywords (Optional)</p>
                        <input type="text"
                               id="altomatic_seo_keywords"
                               name="altomatic_seo_keywords"
                               value="<?php echo esc_attr(get_option('altomatic_seo_keywords')); ?>"
                               class="regular-text"
                               style="padding: 4px 8px;"
                               placeholder="Separate each keyword with a comma (max of 6)"
                               autocomplete="off">
                               <p class="description"><i>Only added to images that match the keywords</i></p>

                    </label>
                </td>
            </tr>

            <!-- Format on Upload -->
            <tr>
                <th scope="row">
                    <?php esc_html_e('Format on Upload', 'altomatic'); ?>
                </th>
                <td>
                    <label>
                        <input type="checkbox"
                               id="altomatic_format_on_upload"
                               name="altomatic_format_on_upload"
                               value="1"
                               <?php checked(get_option('altomatic_format_on_upload', true), true); ?>>
                        <?php esc_html_e('Optimize images on upload', 'altomatic'); ?>
                    </label>
                    <p class="description"><?php esc_html_e('Automatically optimize images when they are uploaded to your site.', 'altomatic'); ?></p>
                </td>
            </tr>


            <!-- Format Conversion -->
            <tr>
                <th scope="row">
                    <?php esc_html_e('Format Conversion', 'altomatic'); ?>
                </th>
                <td>
                    <fieldset class="altomatic-format-conversion">
                        <!-- Add a hidden field that will be disabled when any checkbox is checked -->
                        <input type="hidden" name="altomatic_format_conversion[]" value="" class="empty-value">
                        <label style="display: block; margin-bottom: 12px;">
                            <input type="checkbox"
                                   name="altomatic_format_conversion[]"
                                   value="jpeg"
                                   <?php checked(in_array('jpeg', $selected_formats), true); ?>>
                            <?php esc_html_e('Create JPEG', 'altomatic'); ?>
                        </label>
                        <label style="display: block; margin-bottom: 12px;">
                            <input type="checkbox"
                                   name="altomatic_format_conversion[]"
                                   value="webp"
                                   <?php checked(in_array('webp', $selected_formats), true); ?>>
                            <?php esc_html_e('Create WebP', 'altomatic'); ?>
                        </label>
                        <label style="display: block; margin-bottom: 12px;">
                            <input type="checkbox"
                                   name="altomatic_format_conversion[]"
                                   value="avif"
                                   <?php checked(in_array('avif', $selected_formats), true); ?>>
                            <?php esc_html_e('Create AVIF', 'altomatic'); ?>
                        </label>
                    </fieldset>
                    <p class="description">
                        <?php esc_html_e('Choose which format(s) should be created and optimized if needed.', 'altomatic'); ?>
                    </p>
                </td>
            </tr>

            <!-- Image Sizes -->
            <tr>
                <th scope="row">
                    <?php esc_html_e('Image Sizes', 'altomatic'); ?>
                </th>
                <td>
                    <fieldset class="altomatic-image-sizes">
                        <!-- Add a hidden field that will be disabled when any checkbox is checked -->
                        <input type="hidden" name="altomatic_image_sizes[]" value="" class="empty-value">
                        <!-- Original size -->
                        <label style="display: block; margin-bottom: 12px;">
                            <input type="checkbox"
                                   name="altomatic_image_sizes[]"
                                   value="full"
                                   <?php checked(in_array('full', $selected_sizes), true); ?>>
                            <?php esc_html_e('Original size', 'altomatic'); ?>
                        </label>

                        <?php
                        // Intermediate sizes
                        foreach ($image_sizes as $size) {
                            if ($size === 'full') continue;

                            $label = ucfirst(str_replace('_', ' ', $size));
                            $dimensions = '';

                            // Get dimensions for core sizes
                            if ($size === 'thumbnail') {
                                $width = get_option('thumbnail_size_w');
                                $height = get_option('thumbnail_size_h');
                                $dimensions = " ({$width}x{$height})";
                            } elseif ($size === 'medium') {
                                $width = get_option('medium_size_w');
                                $height = get_option('medium_size_h');
                                $dimensions = " ({$width}x{$height})";
                            } elseif ($size === 'medium_large') {
                                $width = get_option('medium_large_size_w', 768);
                                $dimensions = " ({$width}x?)";
                            } elseif ($size === 'large') {
                                $width = get_option('large_size_w');
                                $height = get_option('large_size_h');
                                $dimensions = " ({$width}x{$height})";
                            }
                            ?>
                            <label style="display: block; margin-bottom: 12px;">
                                <input type="checkbox"
                                       name="altomatic_image_sizes[]"
                                       value="<?php echo esc_attr($size); ?>"
                                       <?php checked(in_array($size, $selected_sizes), true); ?>>
                                <?php echo esc_html($label . $dimensions); ?>
                            </label>
                            <?php
                        }
                        ?>
                    </fieldset>
                    <p class="description">
                        <?php esc_html_e('Choose which image sizes should be optimized. Note: each image size uses 1 credit.', 'altomatic'); ?>
                        <br>
                        <em><?php esc_html_e('If AVIF is selected, only thumbnail and medium sizes will be created.', 'altomatic'); ?></em>
                        <br>
                        <em><?php esc_html_e('WebP Original Size will be optimized up to 2,560px (longest side).', 'altomatic'); ?></em>
                    </p>
                </td>
            </tr>
            <!-- Srcset Optimization Setting -->
            <tr>
                <th scope="row">
                <label for="altomatic_enable_srcset_optimization"><?php esc_html_e('Srcset Optimization', 'altomatic'); ?></label>
                </th>
                <td>
                <label>
                    <input type="checkbox"
                        id="altomatic_enable_srcset_optimization"
                        name="altomatic_enable_srcset_optimization"
                        value="1"
                        <?php checked(get_option('altomatic_enable_srcset_optimization', true), true); ?>>
                    <?php esc_html_e('Optimize image URLs in srcset attributes (Recommended)', 'altomatic'); ?>
                </label>
                <p class="description">
                    <?php esc_html_e('Replaces standard image URLs in srcset with optimized versions (AVIF > WebP > Optimized Original) where available.', 'altomatic'); ?>
                </p>
                </td>
            </tr>
            <!-- Debug Logging -->
                <tr>
                    <th scope="row">
                        <label for="altomatic_enable_debug_logging"><?php esc_html_e('Debug Logging', 'altomatic'); ?></label>
                    </th>
                    <td>
                        <label>
                            <input type="checkbox"
                                id="altomatic_enable_debug_logging"
                                name="altomatic_enable_debug_logging"
                                value="1"
                                <?php checked(get_option('altomatic_enable_debug_logging', false), true); ?>>
                            <?php esc_html_e('Enable debug logging', 'altomatic'); ?>
                        </label>
                        <p class="description">
                            <?php esc_html_e('Enables logging of debug information to the WordPress error log.', 'altomatic'); ?>
                        </p>
                    </td>
                </tr>
        </table>

        <?php submit_button(); ?>
    </form>
</div>
