<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Template for displaying optimization summary in the media modal view
 *
 * @link       https://altomatic.io
 * @since      1.0.0
 *
 * @package    Altomatic
 * @subpackage Altomatic/admin/partials
 */

// Expected variables: $metadata (containing _altomatic_optimized_sizes)
$size_data = $metadata['_altomatic_optimized_sizes'];

// Calculate optimization stats
$stats = array(
    'status' => 'unoptimized',
    'size_count' => 0,
    'bytes_saved' => 0,
    'percent_saved' => 0
);

if (!empty($size_data)) {
    $total_original = 0;
    $total_optimized = 0;
    $optimized_count = 0;
    $counted_formats = array();

    // Get original sizes from attachment metadata
    if (!empty($metadata['filesize'])) {
        $total_original += $metadata['filesize'];
    }
    if (!empty($metadata['sizes'])) {
        foreach ($metadata['sizes'] as $size_info) {
            if (!empty($size_info['filesize'])) {
                $total_original += $size_info['filesize'];
            }
        }
    }

    // Get optimized sizes
    foreach (['jpeg', 'webp', 'avif'] as $format) {
        if (!empty($size_data[$format])) {
            foreach ($size_data[$format] as $size_name => $size_info) {
                // Only count if file exists and has a size
                if (!empty($size_info['file']) && file_exists($size_info['file']) && !empty($size_info['filesize'])) {
                    $total_optimized += $size_info['filesize'];

                    // For formats count, only increment once per format (not per size)
                    if (!isset($counted_formats[$format])) {
                        $optimized_count++;
                        $counted_formats[$format] = true;
                    }
                }
            }
        }
    }

    if ($optimized_count > 0) {
        $stats['status'] = 'optimized';
        $stats['size_count'] = $optimized_count;
        $stats['bytes_saved'] = $total_original - $total_optimized;
        if ($total_original > 0) {
            $stats['percent_saved'] = ($stats['bytes_saved'] / $total_original) * 100;
        }
    }
}

// Get optimization stats
$stats = $this->get_optimization_stats($post->ID);

// Display optimization status
?>

<div class="altomatic-optimization-summary">
    <?php if ($stats['status'] === 'optimized'): ?>
        <div class="altomatic-status optimized">
            <span class="dashicons dashicons-yes-alt"></span>
            <?php
            printf(
                /* translators: %d: Number of formats optimized */
                esc_html(_n('%d format optimized', '%d formats optimized', $stats['formats_count'], 'altomatic')),
                esc_html($stats['formats_count'])
            );

            if ($stats['percent_saved'] > 0) {
                echo ' &ndash; ';
                printf(
                    /* translators: %s: The percentage of bytes saved */
                    esc_html__('Saved %s%%', 'altomatic'),
                    esc_html(number_format($stats['percent_saved'], 1))
                );
            }

            // Show bytes saved directly
            if ($stats['bytes_saved'] > 0) {
                echo '<div class="bytes-saved">';
                printf(
                    /* translators: %s: The size of the image saved */
                    esc_html__('Reduced by %s', 'altomatic'),
                    esc_html(size_format($stats['bytes_saved'], 1))
                );
                echo '</div>';
            }
            ?>
        </div>
    <?php elseif ($stats['status'] === 'processing'): ?>
        <div class="altomatic-status processing">
            <span class="dashicons dashicons-update"></span>
            <?php esc_html_e('Processing', 'altomatic'); ?>
        </div>
    <?php elseif ($stats['status'] === 'error'): ?>
        <div class="altomatic-status error">
            <span class="dashicons dashicons-warning"></span>
            <?php esc_html_e('Error', 'altomatic'); ?>
            <?php if (!empty($stats['error'])): ?>
                <p class="description"><?php echo esc_html($stats['error']); ?></p>
            <?php endif; ?>
        </div>
    <?php elseif ($stats['status'] === 'credits_exceeded'): ?>
        <div class="altomatic-status unoptimized">
            <span class="dashicons dashicons-marker"></span>
            <?php esc_html_e('Not optimized', 'altomatic'); ?>
            <p class="description"><?php esc_html_e('Credit limit exceeded. Please upgrade your plan.', 'altomatic'); ?></p>
        </div>
    <?php else: ?>
        <div class="altomatic-status unoptimized">
            <span class="dashicons dashicons-marker"></span>
            <?php esc_html_e('Not optimized', 'altomatic'); ?>
        </div>
    <?php endif; ?>
</div>
