<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Template for displaying detailed compression information
 *
 * @link       https://altomatic.io
 * @since      1.0.0
 *
 * @package    Altomatic
 * @subpackage Altomatic/admin/partials
 */


// Expected variables: $attachment_id

// Get attachment metadata
$attachment_metadata = wp_get_attachment_metadata($attachment_id);
if (empty($attachment_metadata)) {
    return;
}

// Get optimization data
$size_data = !empty($attachment_metadata['_altomatic_optimized_sizes']) ? $attachment_metadata['_altomatic_optimized_sizes'] : array();

// Get original file extension
$original_extension = strtolower(pathinfo($attachment_metadata['file'], PATHINFO_EXTENSION));

// Get original sizes
$original_sizes = array();
$original_sizes['full'] = !empty($attachment_metadata['filesize']) ? $attachment_metadata['filesize'] : 0;
if (!empty($attachment_metadata['sizes'])) {
    foreach ($attachment_metadata['sizes'] as $size_name => $size_info) {
        $original_sizes[$size_name] = !empty($size_info['filesize']) ? $size_info['filesize'] : 0;
    }
}

// Get optimization date
$date = !empty($attachment_metadata['_altomatic_date']) ?
    date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($attachment_metadata['_altomatic_date'])) : '';

// Remove debug output
// echo '<pre>'; print_r($attachment_metadata); echo '</pre>';
// echo '<pre>'; print_r(get_post_meta($attachment_id)); echo '</pre>';
?>

<div class="altomatic-compression-details">
    <h3><?php esc_html_e('Image Optimization Details', 'altomatic'); ?></h3>

    <table>
        <thead>
            <tr>
                <th class="size-column"><?php esc_html_e('Size', 'altomatic'); ?></th>
                <th class="original-column"><?php esc_html_e('Original', 'altomatic'); ?> (<?php echo esc_html(strtoupper($original_extension)); ?>)</th>
                <th class="jpeg-column"><?php esc_html_e('JPEG', 'altomatic'); ?></th>
                <th class="webp-column"><?php esc_html_e('WebP', 'altomatic'); ?></th>
                <th class="avif-column"><?php esc_html_e('AVIF', 'altomatic'); ?></th>
                <th class="savings-column"><?php esc_html_e('Savings', 'altomatic'); ?></th>
            </tr>
        </thead>
        <tbody>
            <?php
            $total_original = 0;
            $total_jpeg = 0;
            $total_webp = 0;
            $total_avif = 0;
            $total_savings = 0;

            // Track sizes available for each format for accurate percentage calculations
            $sizes_with_jpeg = array();
            $sizes_with_webp = array();
            $sizes_with_avif = array();

            // Process each size
            foreach ($original_sizes as $size_name => $original_size):
                $total_original += $original_size;

                // Get jpeg size
                $jpeg_size = !empty($size_data['jpeg'][$size_name]['filesize']) ? $size_data['jpeg'][$size_name]['filesize'] : '-';
                if ($jpeg_size !== '-') {
                    $total_jpeg += $jpeg_size;
                    $sizes_with_jpeg[$size_name] = $original_size;
                }

                // Get WebP size
                $webp_size = !empty($size_data['webp'][$size_name]['filesize']) ? $size_data['webp'][$size_name]['filesize'] : '-';
                if ($webp_size !== '-') {
                    $total_webp += $webp_size;
                    $sizes_with_webp[$size_name] = $original_size;
                }

                // Get AVIF size
                $avif_size = !empty($size_data['avif'][$size_name]['filesize']) ? $size_data['avif'][$size_name]['filesize'] : '-';
                if ($avif_size !== '-') {
                    $total_avif += $avif_size;
                    $sizes_with_avif[$size_name] = $original_size;
                }

                // Calculate savings based on smallest available format for this size
                $sizes = array($original_size);
                if ($jpeg_size !== '-') $sizes[] = $jpeg_size;
                if ($webp_size !== '-') $sizes[] = $webp_size;
                if ($avif_size !== '-') $sizes[] = $avif_size;

                $smallest_size = min($sizes);
                $savings = $original_size - $smallest_size;
                $savings_percent = $original_size > 0 ? round(($savings / $original_size) * 100, 1) : 0;
                $total_savings += $savings;
            ?>
                <tr>
                    <td class="size-name"><?php echo esc_html(ucfirst($size_name)); ?></td>
                    <td><?php echo esc_html(size_format($original_size, 1)); ?></td>
                    <td><?php echo $jpeg_size !== '-' ? esc_html(size_format($jpeg_size, 1)) : '-'; ?></td>
                    <td><?php echo $webp_size !== '-' ? esc_html(size_format($webp_size, 1)) : '-'; ?></td>
                    <td><?php echo $avif_size !== '-' ? esc_html(size_format($avif_size, 1)) : '-'; ?></td>
                    <td class="savings">
                        <?php if ($savings > 0): ?>
                            <?php echo esc_html(size_format($savings, 1)); ?>
                            (<?php echo esc_html($savings_percent); ?>%)
                        <?php else: ?>
                            -
                        <?php endif; ?>
                    </td>
                </tr>
            <?php endforeach; ?>
        </tbody>
        <tfoot>
            <tr>
                <td class="size-name"><strong><?php esc_html_e('Total', 'altomatic'); ?></strong></td>
                <td><strong><?php echo esc_html(size_format($total_original, 1)); ?></strong></td>
                <td>
                    <?php if ($total_jpeg > 0):
                        $original_for_jpeg = array_sum($sizes_with_jpeg);
                        $jpeg_savings_percent = round((($original_for_jpeg - $total_jpeg) / $original_for_jpeg) * 100, 1);
                    ?>
                        <strong>
                            <?php echo esc_html(size_format($total_jpeg, 1)); ?>
                            (<?php echo esc_html($jpeg_savings_percent); ?>%)
                        </strong>
                    <?php else: ?>
                        -
                    <?php endif; ?>
                </td>
                <td>
                    <?php if ($total_webp > 0):
                        $original_for_webp = array_sum($sizes_with_webp);
                        $webp_savings_percent = round((($original_for_webp - $total_webp) / $original_for_webp) * 100, 1);
                    ?>
                        <strong>
                            <?php echo esc_html(size_format($total_webp, 1)); ?>
                            (<?php echo esc_html($webp_savings_percent); ?>%)
                        </strong>
                    <?php else: ?>
                        -
                    <?php endif; ?>
                </td>
                <td>
                    <?php if ($total_avif > 0):
                        $original_for_avif = array_sum($sizes_with_avif);
                        $avif_savings_percent = round((($original_for_avif - $total_avif) / $original_for_avif) * 100, 1);
                    ?>
                        <strong>
                            <?php echo esc_html(size_format($total_avif, 1)); ?>
                            (<?php echo esc_html($avif_savings_percent); ?>%)
                        </strong>
                    <?php else: ?>
                        -
                    <?php endif; ?>
                </td>
                <td class="savings">
                    <?php if ($total_savings > 0): ?>
                        <strong>
                            <?php echo esc_html(size_format($total_savings, 1)); ?>
                            (<?php echo esc_html(round(($total_savings / $total_original) * 100, 1)); ?>%)
                        </strong>
                    <?php else: ?>
                        -
                    <?php endif; ?>
                </td>
            </tr>
        </tfoot>
    </table>

    <div class="optimization-meta">
        <?php if (!empty($date)): ?>
            <p><strong><?php esc_html_e('Optimized on:', 'altomatic'); ?></strong> <?php echo esc_html($date); ?></p>
        <?php endif; ?>
    </div>
</div>
