/**
 * Altomatic Settings JavaScript
 */

(function($) {
    "use strict";
    $(document).ready(function() {
        // Tab switching
        $(".altomatic-tab").on("click", function() {
            $(".altomatic-tab").removeClass("active");
            $(this).addClass("active");

            $(".tab-pane").hide().removeClass("active");
            $("#" + $(this).data("tab")).show().addClass("active");
        });

        // Handle format conversion checkboxes
        const formatCheckboxes = document.querySelectorAll(
            'input[name="altomatic_format_conversion[]"][type="checkbox"]'
        );
        const formatEmptyValue = document.querySelector(
            'input[name="altomatic_format_conversion[]"].empty-value'
        );

        formatCheckboxes.forEach((checkbox) => {
            checkbox.addEventListener("change", function() {
                const anyChecked = Array.from(formatCheckboxes).some(
                    (cb) => cb.checked
                );
                formatEmptyValue.disabled = anyChecked;
            });
        });

        // Handle image sizes checkboxes
        const sizeCheckboxes = document.querySelectorAll(
            'input[name="altomatic_image_sizes[]"][type="checkbox"]'
        );
        const sizeEmptyValue = document.querySelector(
            'input[name="altomatic_image_sizes[]"].empty-value'
        );

        sizeCheckboxes.forEach((checkbox) => {
            checkbox.addEventListener("change", function() {
                const anyChecked = Array.from(sizeCheckboxes).some((cb) => cb.checked);
                sizeEmptyValue.disabled = anyChecked;
            });
        });

        // Initial state
        formatEmptyValue.disabled = Array.from(formatCheckboxes).some(
            (cb) => cb.checked
        );
        sizeEmptyValue.disabled = Array.from(sizeCheckboxes).some(
            (cb) => cb.checked
        );

        // ── Email Code Login Modal ──────────────────────────────────

        var $modal        = $("#altomatic-login-modal");
        var $stepEmail    = $("#altomatic-login-step-email");
        var $stepCode     = $("#altomatic-login-step-code");
        var $emailInput   = $("#altomatic-login-email");
        var $codeInput    = $("#altomatic-login-code");
        var $sendBtn      = $("#altomatic-send-code");
        var $verifyBtn    = $("#altomatic-verify-code");
        var $resendBtn    = $("#altomatic-resend-code");
        var $statusArea   = $("#altomatic-login-status");
        var resendTimer   = null;

        function openModal() {
            $modal.show();
            resetModal();
            $emailInput.focus();
        }

        function closeModal() {
            $modal.hide();
            resetModal();
        }

        function resetModal() {
            $stepEmail.show();
            $stepCode.hide();
            $emailInput.val("");
            $codeInput.val("");
            $statusArea.html("").removeClass("success error");
            $sendBtn.prop("disabled", false).text(
                altomaticSettings.i18n ? altomaticSettings.i18n.sendCode : "Send Code"
            );
            $verifyBtn.prop("disabled", false);
            $resendBtn.prop("disabled", false);
            clearResendTimer();
        }

        function setStatus(message, type) {
            $statusArea
                .html(message)
                .removeClass("success error")
                .addClass(type || "");
        }

        function clearResendTimer() {
            if (resendTimer) {
                clearInterval(resendTimer);
                resendTimer = null;
            }
        }

        function startResendCooldown(seconds) {
            var remaining = seconds;
            $resendBtn.prop("disabled", true);
            $resendBtn.text("Resend Code (" + remaining + "s)");

            resendTimer = setInterval(function() {
                remaining--;
                if (remaining <= 0) {
                    clearResendTimer();
                    $resendBtn.prop("disabled", false).text("Resend Code");
                } else {
                    $resendBtn.text("Resend Code (" + remaining + "s)");
                }
            }, 1000);
        }

        // Open modal
        $("#altomatic-get-api-key").on("click", function(e) {
            e.preventDefault();
            openModal();
        });

        // Close modal via X button or backdrop click
        $modal.on("click", ".altomatic-modal-close, .altomatic-modal-backdrop", function(e) {
            e.preventDefault();
            closeModal();
        });

        // Close modal on Escape key
        $(document).on("keydown", function(e) {
            if (e.key === "Escape" && $modal.is(":visible")) {
                closeModal();
            }
        });

        // Step 1: Send code
        $sendBtn.on("click", function() {
            sendCode();
        });

        // Allow Enter key on email input
        $emailInput.on("keydown", function(e) {
            if (e.key === "Enter") {
                e.preventDefault();
                sendCode();
            }
        });

        function sendCode() {
            var email = $.trim($emailInput.val());
            if (!email) {
                setStatus("Please enter your email address.", "error");
                $emailInput.focus();
                return;
            }

            $sendBtn.prop("disabled", true).text("Sending...");
            setStatus("");

            $.post(altomaticSettings.ajaxurl, {
                action: "altomatic_request_login_code",
                nonce:  altomaticSettings.nonce,
                email:  email
            }, function(response) {
                if (response.success) {
                    // Move to code step
                    $stepEmail.hide();
                    $stepCode.show();
                    setStatus(response.data.message, "success");
                    startResendCooldown(30);
                    $codeInput.val("").focus();
                } else {
                    setStatus(response.data.message || "An error occurred.", "error");
                    $sendBtn.prop("disabled", false).text("Send Code");
                }
            }).fail(function() {
                setStatus("Request failed. Please try again.", "error");
                $sendBtn.prop("disabled", false).text("Send Code");
            });
        }

        // Step 2: Verify code
        $verifyBtn.on("click", function() {
            verifyCode();
        });

        // Allow Enter key on code input
        $codeInput.on("keydown", function(e) {
            if (e.key === "Enter") {
                e.preventDefault();
                verifyCode();
            }
        });

        function verifyCode() {
            var code  = $.trim($codeInput.val());
            var email = $.trim($emailInput.val());

            if (!code || code.length < 6) {
                setStatus("Please enter the 6-digit code.", "error");
                $codeInput.focus();
                return;
            }

            $verifyBtn.prop("disabled", true).text("Verifying...");
            setStatus("");

            $.post(altomaticSettings.ajaxurl, {
                action: "altomatic_verify_login_code",
                nonce:  altomaticSettings.nonce,
                email:  email,
                code:   code
            }, function(response) {
                if (response.success) {
                    setStatus(response.data.message, "success");

                    // Auto-fill the API key field
                    var $apiKeyInput = $("#altomatic_api_key");
                    $apiKeyInput.val(response.data.apiKey).trigger("change");

                    // Close modal after short delay
                    setTimeout(function() {
                        closeModal();
                    }, 1500);
                } else {
                    setStatus(response.data.message || "Verification failed.", "error");
                    $verifyBtn.prop("disabled", false).text("Verify");
                }
            }).fail(function() {
                setStatus("Request failed. Please try again.", "error");
                $verifyBtn.prop("disabled", false).text("Verify");
            });
        }

        // Resend code
        $resendBtn.on("click", function() {
            var email = $.trim($emailInput.val());
            if (!email) {
                return;
            }

            $resendBtn.prop("disabled", true);
            setStatus("");

            $.post(altomaticSettings.ajaxurl, {
                action: "altomatic_request_login_code",
                nonce:  altomaticSettings.nonce,
                email:  email
            }, function(response) {
                if (response.success) {
                    setStatus(response.data.message, "success");
                    startResendCooldown(30);
                    $codeInput.val("").focus();
                } else {
                    setStatus(response.data.message || "Could not resend code.", "error");
                    $resendBtn.prop("disabled", false).text("Resend Code");
                }
            }).fail(function() {
                setStatus("Request failed. Please try again.", "error");
                $resendBtn.prop("disabled", false).text("Resend Code");
            });
        });
    });
})(jQuery);
