/**
 * Altomatic Admin JavaScript
 */
(function($) {
    'use strict';

    // Wait for document to be ready
    $(document).ready(function() {
        // Initialize thickbox
        if (typeof tb_init === 'function') {
            tb_init('a.thickbox');
        }

        // Initialize copy buttons
        initializeCopyButtons();

        // Re-initialize Thickbox on AJAX requests (for media library lazy loading)
        $(document).ajaxComplete(function() {
            if (typeof tb_init === 'function') {
                tb_init('a.thickbox');
            }
            // Also reinitialize copy buttons after AJAX
            initializeCopyButtons();
        });

        // Add optimize button handler
        $(document).on('click', '.altomatic-optimize', function(e) {
            e.preventDefault();

            const button = $(this);
            const imageId = button.data('id');
            const originalText = button.text();

            if (!imageId) {
                console.error('No image ID found');
                return;
            }

            // Show loading state
            button.text('Optimizing...').prop('disabled', true);

            // Make the REST API request using wp.apiFetch
            wp.apiFetch({
                path: '/altomatic/v1/optimize-image/' + imageId,
                method: 'POST'
            }).then(response => {
                if (response.success) {
                    button.text('Success!');

                    // Reload the page after showing success message
                    setTimeout(() => {
                        window.location.reload();
                    }, 1000);
                } else {
                    throw new Error(response.message || 'Failed to optimize image');
                }
            }).catch(error => {
                console.error('Error:', error);

                // Check if error is due to credit limit
                if (error.message && error.message.includes('credit')) {
                    button.text('Credits Exceeded').prop('disabled', true);

                    // Add error message below button
                    const errorMsg = $('<p class="description error">').text('Credit limit exceeded. Please upgrade your plan to optimize more images.');
                    button.after(errorMsg);
                } else {
                    button.text('Error!');

                    // Reset button after delay
                    setTimeout(() => {
                        button.text(originalText).prop('disabled', false);
                    }, 2000);
                }
            });
        });

        // Re-initialize when thickbox modal is opened
        $(document).on('thickbox:iframe:loaded', function() {
            initializeCopyButtons();
        });

        // Monitor for any dynamic content changes
        // This helps with WordPress admin which often loads things dynamically
        var observer = new MutationObserver(function(mutations) {
            mutations.forEach(function(mutation) {
                if (mutation.addedNodes && mutation.addedNodes.length > 0) {
                    // Check if any of the added nodes contain our buttons
                    for (var i = 0; i < mutation.addedNodes.length; i++) {
                        var node = mutation.addedNodes[i];
                        if (node.nodeType === 1 &&
                            ($(node).find('.altomatic-url-field').length > 0 ||
                                $(node).hasClass('altomatic-url-field'))) {
                            initializeCopyButtons();
                            break;
                        }
                    }
                }
            });
        });

        // Start observing the body for changes
        observer.observe(document.body, { childList: true, subtree: true });

        // Handle regenerate alt text button click
        $(document).on('click', '.altomatic-regenerate-alt', function(e) {
            e.preventDefault();

            const button = $(this);
            const imageId = button.data('id');
            const originalText = button.text();

            if (!imageId) {
                console.error('No image ID found');
                return;
            }

            // Show loading state
            button.text('Generating...').prop('disabled', true);

            // Make the REST API request using wp.apiFetch
            wp.apiFetch({
                path: '/altomatic/v1/regenerate-alt-text/' + imageId,
                method: 'POST'
            }).then(response => {
                if (response.success) {
                    // Update the preview
                    button.closest('.altomatic-alt-text').find('.alt-text-preview').text(response.alt_text);

                    // Save the alt text via WordPress REST API
                    wp.apiFetch({
                        path: `/wp/v2/media/${imageId}`,
                        method: 'POST',
                        data: {
                            alt_text: response.alt_text
                        }
                    }).then(() => {
                        // Update the wordpress input field
                        // #attachment-details-two-column-alt-text for modal
                        // #attachment_alt for edit page
                        $('#attachment-details-two-column-alt-text').val(response.alt_text);
                        $('#attachment_alt').val(response.alt_text);

                        // Update the date if it exists
                        if (response.generated_date) {
                            const date = new Date(response.generated_date);
                            const formattedDate = date.toLocaleDateString(
                                document.documentElement.lang || 'en-US', { year: 'numeric', month: 'long', day: 'numeric' }
                            );
                            const dateText = altomaticL10n.generatedOn + ' ' + formattedDate;

                            let dateElement = button.closest('.altomatic-alt-text').find('.alt-text-date');
                            if (dateElement.length) {
                                dateElement.text(dateText);
                            } else {
                                button.closest('.altomatic-alt-text').find('.alt-text-preview')
                                    .after('<p class="alt-text-date">' + dateText + '</p>');
                            }
                        }

                        button.text('Success!');

                        // Reset button after delay
                        setTimeout(() => {
                            button.text(originalText).prop('disabled', false);
                        }, 2000);
                    });
                } else {
                    throw new Error(response.message || 'Failed to generate alt text');
                }
            }).catch(error => {
                console.error('Error:', error);

                // Check if error is due to credit limit
                if (error.message && error.message.includes('credit')) {
                    button.text('Credits Exceeded').prop('disabled', true);

                    // Add error message below button
                    const errorMsg = $('<p class="description error">').text('Credit limit exceeded. Please upgrade your plan to generate more alt text.');
                    button.after(errorMsg);
                } else {
                    button.text('Error!');

                    // Reset button after delay
                    setTimeout(() => {
                        button.text(originalText).prop('disabled', false);
                    }, 2000);
                }
            });
        });

        // Initialize API key toggle
        $('.toggle-api-key').on('click', function() {
            const $input = $('#altomatic_api_key');
            const $icon = $(this).find('.dashicons');

            if ($input.attr('type') === 'password') {
                $input.attr('type', 'text');
                $icon.removeClass('dashicons-visibility').addClass('dashicons-hidden');
            } else {
                $input.attr('type', 'password');
                $icon.removeClass('dashicons-hidden').addClass('dashicons-visibility');
            }
        });

        // Initialize API status check if we're on the settings page
        if ($('.altomatic-api-status').length) {
            checkApiStatus();

            // Check status when API key changes
            $('#altomatic_api_key').on('change', function() {
                checkApiStatus();
            });
        }



        // Initialize copy buttons
        function initializeCopyButtons() {
            $('.altomatic-url-field .copy-button').each(function() {
                var $button = $(this);

                // Skip if already initialized
                if ($button.data('initialized')) {
                    return;
                }

                $button.data('initialized', true);

                $button.on('click', function(e) {
                    e.preventDefault();

                    var $input = $(this).siblings('input[type="text"]');

                    // Copy URL to clipboard
                    $input.select();
                    document.execCommand('copy');

                    // Show copied confirmation
                    var $icon = $(this).find('.dashicons');
                    var originalClass = $icon.attr('class');

                    $icon.removeClass().addClass('dashicons dashicons-yes');

                    setTimeout(function() {
                        $icon.removeClass().addClass(originalClass);
                    }, 1500);
                });
            });
        }

        /**
         * Check API status and get credit information
         */
        function checkApiStatus() {
            const $statusIcon = $('.api-key-wrapper .status-icon');
            const $apiStatus = $('.altomatic-api-status');
            const $creditInfo = $('.api-credit-info');
            const apiKey = $('#altomatic_api_key').val();

            // Reset status
            $statusIcon.removeClass('connected error');
            $apiStatus.hide();
            $creditInfo.hide();

            if (!apiKey) {
                $statusIcon.addClass('error');
                return;
            }

            // Make API request to validate key
            wp.apiFetch({
                path: '/altomatic/v1/validate-api',
                method: 'POST',
                data: {
                    api_key: apiKey
                }
            }).then(function(response) {
                console.log('API validation response:', response);
                if (response.success) {
                    $statusIcon.addClass('connected');
                    $apiStatus.show();

                    // Update credit information directly from the validation response
                    if (response.data && response.data.credits) {
                        const data = {
                            used_credits: response.data.credits.used,
                            total_credits: response.data.credits.total,
                            monthly_optimizations: response.data.optimized_images.monthly,
                            monthly_alt_text: response.data.alt_text_generations.monthly
                        };
                        updateCreditInfo(data);
                    }
                } else {
                    throw new Error(response.message || altomaticL10n.apiError);
                }
            }).catch(function(error) {
                console.error('API error:', error);
                $statusIcon.addClass('error');
                $apiStatus.hide();
                $creditInfo.hide();
            });
        }

        /**
         * Update credit information display
         */
        function updateCreditInfo(data) {
            const $apiStatus = $('.altomatic-api-status');
            const $creditInfo = $('.api-credit-info');
            const $creditCount = $creditInfo.find('.credit-count');
            const $periodText = $creditInfo.find('.period-text');

            if (data) {
                $creditCount.text(`${data.used_credits} / ${data.total_credits} credits used`);
                $periodText.text(`${data.monthly_optimizations} images optimized, ${data.monthly_alt_text} alt texts generated this month (alt texts count for 3 credits each)`);
                $apiStatus.show();
                $creditInfo.show();
            } else {
                $apiStatus.hide();
                $creditInfo.hide();
            }
        }
    });
})(jQuery);