<?php

namespace GoodWP\Altinator\Vendor\GoodWP\Common\WordPress;

use Exception;
use GoodWP\Altinator\Vendor\GoodWP\Common\Contracts\Bootable;
use WP_Taxonomy;

/**
 * A custom taxonomy.
 *
 * @package GoodWP\Common\WordPress
 */
abstract class Taxonomy implements Bootable {
	/**
	 * The taxonomy slug
	 *
	 * @var string
	 */
	public const TAXONOMY = '';

    /**
     * The taxonomy object registered with WordPress.
     *
     * @var WP_Taxonomy
     */
	protected WP_Taxonomy $taxonomy_object;

    /**
     * {@inheritDoc}
     */
    public function boot(): void {
        add_action( 'init', [ $this, 'register' ] );
    }

    /**
     * Register the custom taxonomy with WordPress.
     * Retrieves the arguments, capabilities, and labels for the taxonomy,
     * registers the post type, and assigns the taxonomy object to the class property.
     */
	public function register(): void {
        assert( ! empty( static::TAXONOMY ), 'Taxonomy must declare TAXONOMY const with slug of taxonomy' );

		$args = $this->get_args();

		$capabilities = $this->get_capabilities();
		if ( ! empty( $capabilities ) ) {
			$args['capabilities'] = $capabilities;
		}

		$labels = $this->get_labels();
		if ( ! empty( $labels ) ) {
			$args['labels'] = $labels;
		}

		$taxonomy_object = register_taxonomy( static::TAXONOMY, $this->get_post_type(), $args );
		if ( ! is_wp_error( $taxonomy_object ) ) {
			$this->taxonomy_object = $taxonomy_object;
		}
	}

    /**
     * Build the arguments for the taxonomy.
     *
     * @return array
     */
	abstract protected function get_args(): array;

    /**
     * Build the capabilities for the taxonomy.
     *
     * @return array
     */
	abstract protected function get_capabilities(): array;

    /**
     * Build the labels for the taxonomy.
     *
     * @return array
     */
	abstract protected function get_labels(): array;

    /**
     * The post type(s) to which assign the taxonomy.
     *
     * @return string|array
     */
	abstract protected function get_post_type(): string|array;

    /**
     * Get the taxonomy object registered with WordPress.
     *
     * @return WP_Taxonomy|null Null if called before init/before post type is registered.
     */
    public function get_taxonomy_object(): ?WP_Taxonomy {
        if ( ! isset( $this->taxonomy_object ) ) {
            _doing_it_wrong( __METHOD__, 'Taxonomy is not registered yet', '' );
            return null;
        }
        return $this->taxonomy_object;
    }
}
