<?php

namespace GoodWP\Altinator\Vendor\GoodWP\Common\WordPress;

use GoodWP\Altinator\Vendor\GoodWP\Common\Contracts\Bootable;
use InvalidArgumentException;

/**
 * A service to register settings for a plugin/theme/module.
 */
abstract class Settings implements Bootable {

    /**
     * The option group to which register settings to.
     *
     * @var string
     */
    protected const OPTION_GROUP = '';

    /**
     * Map of registered setting keys.
     * Maps internal short key to full-qualified prefixed key.
     *
     * @var array<string,string>
     */
    protected array $setting_keys = [];

    /**
	 * {@inheritDoc}
     */
    public function boot(): void {
        add_action( 'init', [ $this, 'register' ] );
    }

    /**
     * Registers settings with WordPress.
     *
     * @uses register_setting
     *
     * @return void
     */
    public function register(): void {
        assert( ! empty( static::OPTION_GROUP ), 'Settings class must declare OPTION_GROUP const with option group' );

        foreach ( $this->build_settings() as $key => $args ) {
            $full_key = static::OPTION_GROUP . '_' . $key;
            register_setting( static::OPTION_GROUP, $full_key, $args );
            $this->setting_keys[ $key ] = $full_key;
        }
    }

    /**
     * Build all the settings in a format passed to register_setting,
     *
     * @see register_setting
     *
     * @return array<string,array>
     */
    abstract protected function build_settings(): array;

    /**
     * Return the setting.
     *
     * @uses get_option
     *
     * @param string     $key Unprefixed key.
     * @param mixed|null $default Default value to use if option is not set.
     * @return mixed
     * @throws InvalidArgumentException If the setting is not registered.
     */
    public function get_setting( string $key, mixed $default = null ): mixed { // phpcs:ignore Universal.NamingConventions.NoReservedKeywordParameterNames.defaultFound
        if ( ! $this->has_setting( $key ) ) {
            // phpcs:ignore WordPress.Security.EscapeOutput.ExceptionNotEscaped
            throw new InvalidArgumentException( 'Unknown setting key ' . $key );
        }

        // Distinguish between `null` as a default, and not passing one.
        // Not passing a default to get_option, triggers the standard default filters which also
        // takes the default from register_setting.
        $passed_default = func_num_args() > 1;

        if ( $passed_default ) {
            return get_option( $this->setting_keys[ $key ], $default );
        }
        return get_option( $this->setting_keys[ $key ] );
    }

    /**
     * Get all registered settings.
     *
     * @return array
     */
    public function get_all_settings(): array {
        $settings = [];
        foreach ( $this->setting_keys as $key ) {
            $settings[ $key ] = $this->get_setting( $key );
        }
        return $settings;
    }

    /**
     * Whether a specific setting key was registered in this Settings instance.
     *
     * @param string $key The unprefixed key to check for.
     * @return bool
     */
    public function has_setting( string $key ): bool {
        return array_key_exists( $key, $this->setting_keys );
    }

    /**
     * Update a setting.
     *
     * @uses update_option
     *
     * @param string $key The unprefixed key.
     * @param mixed  $value The value which will be passed to update_option (and sanitized/validated there).
     * @return bool False if setting is not registered in this service, otherwise the result of update_option.
     */
    public function update_setting( string $key, mixed $value ): bool {
        if ( ! $this->has_setting( $key ) ) {
            return false;
        }

        return update_option( $this->setting_keys[ $key ], $value );
    }
}
