<?php

namespace GoodWP\Altinator\Vendor\GoodWP\Common\WordPress;

use Exception;
use GoodWP\Altinator\Vendor\GoodWP\Common\Contracts\Bootable;
use WP_Post_Type;

/**
 * A custom post type.
 *
 * @package GoodWP\Common\WordPress
 */
abstract class Post_Type implements Bootable {

	/**
	 * The post type slug
	 *
	 * @var string
	 */
	public const POST_TYPE = '';

    /**
     * The post type object registered with WordPress.
     *
     * @var WP_Post_Type
     */
	protected WP_Post_Type $post_type_object;

    /**
	 * {@inheritDoc}
     */
    public function boot(): void {
        add_action( 'init', [ $this, 'register' ] );
    }

    /**
     * Register the custom post type with WordPress.
     * Retrieves the arguments, capabilities, and labels for the post type,
     * registers the post type, and assigns the post type object to the class property.
     */
	public function register(): void {
        assert( ! empty( static::POST_TYPE ), 'Post Type must declare POST_TYPE const with slug of post type' );

		$args = $this->get_args();

		$capabilities = $this->get_capabilities();
		if ( ! empty( $capabilities ) ) {
			$args['capabilities'] = $capabilities;
		}

		$labels = $this->get_labels();
		if ( ! empty( $labels ) ) {
			$args['labels'] = $labels;
		}

		$post_type_object = register_post_type( static::POST_TYPE, $args );
		if ( ! is_wp_error( $post_type_object ) ) {
			$this->post_type_object = $post_type_object;
		}
	}

    /**
     * Build the arguments for the post type.
     *
     * @return array
     */
	abstract protected function get_args(): array;

    /**
     * Build the capabilities for the post type.
     *
     * @return array
     */
	abstract protected function get_capabilities(): array;

    /**
     * Build the labels for the post type.
     *
     * @return array
     */
	abstract protected function get_labels(): array;

    /**
     * Get the post type object registered with WordPress.
     *
     * @return WP_Post_Type|null Null if called before init/before post type is registered.
     */
    public function get_post_type_object(): ?WP_Post_Type {
        if ( ! isset( $this->post_type_object ) ) {
            _doing_it_wrong( __METHOD__, 'Post type is not registered yet', '' );
            return null;
        }
        return $this->post_type_object;
    }
}
