<?php

namespace GoodWP\Altinator\Vendor\GoodWP\Common\Templates;

use GoodWP\Altinator\Vendor\GoodWP\Common\Events\Event_Manager_Contract;
use GoodWP\Altinator\Vendor\GoodWP\Common\Events\Has_Events;

/**
 * A service to render templates which allows overwriting templates in themes and filtering of output.
 */
class Template_Renderer implements Template_Renderer_Contract {
    use Has_Events;

    /**
	 * Paths to search for overwritten templates (eg in themes).
	 *
	 * @var array
	 */
	protected array $template_paths;

	/**
	 * Creates a new TemplateService instance.
	 *
	 * @param string                      $template_dir The full path to the main (plugins) template dir (eg plugin/templates).
	 * @param string|null                 $template_folder_name Name of the folder to search for in themes to allow overwriting templates
	 *                                    If null, themes cannot overwrite templates.
	 * @param Event_Manager_Contract|null $event_manager The event manager to use for actions/filters.
	 */
	public function __construct(
        protected string $template_dir,
        ?string $template_folder_name = null,
        ?Event_Manager_Contract $event_manager = null,
	) {
		$this->set_event_manager( $event_manager );

		if ( $template_folder_name ) {
			$this->template_paths = [
				get_stylesheet_directory() . DIRECTORY_SEPARATOR . $template_folder_name,
				get_template_directory() . DIRECTORY_SEPARATOR . $template_folder_name,
				$template_dir,
			];
		} else {
			$this->template_paths = [ $template_dir ];
		}
	}

	/**
	 * Get a rendered template, provides hooks to filter args, template and output.
	 *
	 * @param string|array $template Template name(s) to be rendered (@see locate_template).
	 * @param array        $data Data passed to the template as $data variable.
	 * @return string
	 */
	public function get( $template, array $data ): string {
		ob_start();
		$this->render( $template, $data );
		$output = ob_get_clean();

		/**
		 * Filters the template output if it's returned (and not output)
		 *
		 * @triggers $hookPrefix/template/output
         *
         * @noinspection PhpUnnecessaryLocalVariableInspection
		 *
		 * @param string $output The output of the template (HTML)
		 * @param string $template The template being rendered
		 * @param mixed $data The data passed to the template
		 * @param Template_Renderer $templateService TemplateService being used
		 */
        $output = $this->event_manager ? $this->event_manager->apply_filters( 'template/output', $output, $template, $data, $this ) : $output;
		return $output;
	}

	/**
	 * Renders a rendered template, provides hooks to filter args, template and output.
	 *
	 * @param string|array $template Template name(s) to be rendered (@see locate_template).
	 * @param array        $data Data passed to the template as $data variable.
	 * @return void
	 */
	public function render( $template, array $data ): void {
		/**
		 * Allows the template-rendering to be short-circuited, by returning a non-null value.
		 *
		 * @triggers $hookPrefix/template/pre_render
		 *
		 * @param string|null $pre_render The pre-rendered content. Default null.
		 * @param string $template The template being rendered
		 * @param array $data The data passed to the template
		 * @param Template_Renderer $templateService TemplateService being used
		 */
		$pre_render = $this->event_manager ? $this->event_manager->apply_filters(
			'template/pre_render',
			null,
			$template,
			$data,
			$this
		) : null;

		if ( ! is_null( $pre_render ) ) {
			return;
		}

		/**
		 * Filters the data passed to the template.
		 *
		 * @param mixed $data The data passed to the template
		 * @param string $template The template being rendered
		 * @param Template_Renderer $templateService TemplateService being used
		 */
		$data = $this->event_manager ? $this->event_manager->apply_filters( 'template/data', $data, $template, $this ) : $data;

		/**
		 * Allows doing something / outputting something before the template is located and rendered.
		 *
		 * @param string $template The template being rendered.
		 * @param array $data The data passed to the template.
		 * @param Template_Renderer $templateService TemplateService being used.
		 */
		$this->event_manager?->do_action( 'template/before', $template, $data, $this );

		$template_file = $this->locate_template( $template );
		if ( $template_file ) {
			$this->include( $template_file, $data );
		}

		/**
		 * Allows doing something / outputting something after the template is located and rendered.
		 *
		 * @param string $template The template being rendered.
		 * @param array $data The data passed to the template.
		 * @param Template_Renderer $templateService TemplateService being used.
		 */
		$this->event_manager?->do_action( 'template/after', $template, $data, $this );
	}

	/**
	 * Finds a template in the configured templatePaths based on one or more template names.
	 * Similar to get_template_part from WP Core.
	 *
	 * @param string|array $templates Template name(s) to be rendered (relative to templatePaths).
	 * @return string|null The first found template, or null if no found.
	 */
	public function locate_template( $templates ): ?string {
		$templates = (array) $templates;

		// Custom implementation of WP Core load_template to use template paths in this service.
		$located = null;
		foreach ( $templates as $template_name ) {
			if ( ! $template_name ) {
				continue;
			}
			foreach ( $this->template_paths as $template_path ) {
				if ( file_exists( $template_path . DIRECTORY_SEPARATOR . $template_name ) ) {
					$located = $template_path . DIRECTORY_SEPARATOR . $template_name;
					break 2;
				}
				if ( file_exists( $template_path . DIRECTORY_SEPARATOR . $template_name . '.php' ) ) {
					$located = $template_path . DIRECTORY_SEPARATOR . $template_name . '.php';
					break 2;
				}
			}
		}

		/**
		 * Allows filtering the located template.
		 *
		 * @param string $located The path of the located template.
		 * @param array $templates The names of the templates requested / searched.
		 * @param Template_Renderer $templateService TemplateService being used.
		 */
		$located = $this->event_manager ? $this->event_manager->apply_filters( 'template', $located, $templates, $this ) : $located;
		if ( count( $templates ) === 1 ) {
			/**
			 * Allows filtering the located template on a specific searched template.
			 *
			 * @param string $located The path of the located template.
			 * @param array $templates The names of the templates requested / searched.
			 * @param Template_Renderer $templateService TemplateService being used.
			 */
			$located = $this->event_manager ? $this->event_manager->apply_filters(
				"template/template={$templates[0]}",
				$located,
				$templates,
				$this
			) : $located;
		}

		return $located;
	}

	/**
	 * Includes the template file
	 * in it's own function to reduce variables in scope.
	 *
     * @noinspection PhpUnusedParameterInspection
     *
	 * @param string $template The full path to the file to include.
	 * @param array  $data Data passed to the template.
	 * @return void
     */
	protected function include( $template, $data ): void {
		$template_renderer = $this;
		include $template;
	}
}
