<?php

namespace GoodWP\Altinator\Vendor\GoodWP\Common\Plugin;

use Exception;
use GoodWP\Altinator\Vendor\GoodWP\Common\DI\App;
use GoodWP\Altinator\Vendor\GoodWP\Common\DI\Container;
use GoodWP\Altinator\Vendor\GoodWP\Common\Traits\Forced_Singleton;
use GoodWP\Altinator\Vendor\lucatume\DI52\ContainerException;

/**
 * The main plugin class which initializes DI-Container, core services and everything else
 */
class Plugin extends App {
    use Forced_Singleton;

	/**
	 * Current version of the plugin as of source code
	 *
	 * @var int|string
	 */
	public const VERSION = 0;

	/**
	 * The slug of the plugin used for prefixes/hooks
	 *
	 * @var string
	 */
	public const SLUG = '';

	/**
	 * Directory (absolute path) of the plugin
	 *
	 * @var string
	 */
	public readonly string $path;

	/**
	 * Directory (relative to WP_CONTENT_DIR) of the plugin
	 *
	 * @var string
	 */
	public readonly string $relative_path;

	/**
	 * Main plugin file (used for plugin urls)
	 *
	 * @var string
	 */
	public readonly string $file;

	/**
	 * Creates a new plugin instance main plugin file
	 *
	 * @param string $plugin_file The main plugin file which is loaded by WordPress.
	 */
	public function __construct( string $plugin_file ) {
        assert( ! empty( static::VERSION ), 'VERSION constant in Plugin class must be defined' );
        assert( ! empty( static::SLUG ), 'SLUG constant in Plugin class must be defined' );

		$this->file          = $plugin_file;
		$this->path          = dirname( $plugin_file );
		$this->relative_path = str_replace( WP_CONTENT_DIR, '', $this->path );
	}

	/**
	 * Initialize this plugin instance.
	 * {@inheritDoc}
	 */
	public function boot(): void { // phpcs:ignore Generic.CodeAnalysis.UselessOverridingMethod.Found -- There for docblock.
		parent::boot();
	}

    /**
     * Create the container and set initival values.
     *
     * @param Container|null $base_container The base from which to work. Allows a plugin to setup basic settings.
     *                                       E.g. dependency injection for service providers.
     * @return Container
     * @throws ContainerException If there's an issue while trying to bind the implementation.
     * @throws Exception If container is already initialized.
     */
	protected function init_container( ?Container $base_container = null ): Container {
        $container = $base_container ?? new Container();

		$container->bind( 'plugin', $this );
		$container->bind( static::class, $this );
        $container['plugin.slug']    = static::SLUG;
		$container['plugin.version'] = static::VERSION;
		$container['plugin.file']    = $this->file;
		$container['plugin.path']    = $this->path;

        // Allows for automatically wiring those in constructor parameters of classes/services.
        $container->alias( '$plugin_slug', 'plugin.slug' );
        $container->alias( '$plugin_version', 'plugin.version' );
        $container->alias( '$plugin_file', 'plugin.file' );
        $container->alias( '$plugin_path', 'plugin.path' );

		return parent::init_container( $container );
	}

	/**
	 * Gets the directory (absolute path) of the plugin.
	 *
	 * @param string $relative_path A relative path to add to the plugin path.
	 * @return string The full path to the plugin directory, without trailing slash.
	 */
	public function get_path( string $relative_path = '' ): string {
		if ( ! empty( $relative_path ) ) {
			return $this->path . DIRECTORY_SEPARATOR . $relative_path;
		}

		return $this->path;
	}

	/**
	 * Gets the relative path of the plugin directory to WP_CONTENT_DIR
	 *
	 * @param string $relative_path A relative path to add to the plugin path.
	 * @return string The relative path to the plugin directory, without trailing slash.
	 */
	public function get_relative_path( string $relative_path = '' ): string {
		if ( ! empty( $relative_path ) ) {
			return $this->relative_path . DIRECTORY_SEPARATOR . $relative_path;
		}

		return $this->relative_path;
	}
}
