<?php

namespace GoodWP\Altinator\Vendor\GoodWP\Common\Plugin;

use GoodWP\Altinator\Vendor\GoodWP\Common\Contracts\Runnable;

/**
 * Service for handling plugin migrations
 *
 * @package GoodWP\Common\Plugin
 */
abstract class Migration implements Runnable {

	/**
	 * The option key of the plugin version stored in the database
	 * = previous version on updates.
	 *
	 * @var string
	 */
	protected string $db_option_key;

	/**
	 * Creates a new Migration service instance.
	 *
	 * @param string $plugin_version Current installed (code) plugin version.
	 * @param string $plugin_slug The slug of the plugin to use for the db version.
	 */
	public function __construct( protected string $plugin_version, string $plugin_slug ) {
		$this->db_option_key = "{$plugin_slug}_db_version";
	}

    /**
     * Checks if a migration is required and runs it if so.
     * Should be hooked into the WordPress lifecycle.
     * Best to only do this in wp-admin - e.g. in admin_init.
     *
     * @return void
     */
	public function run(): void {
        $current_version = get_option( $this->db_option_key, 0 );
		if ( $this->should_run_migration( $current_version ) ) {
			$this->run_migration_steps( $this->plugin_version, $current_version );
			update_option( $this->db_option_key, $this->plugin_version, true );
        }
	}

	/**
	 * Checks if a migration is required depending on the stored version in the database.
	 *
     * @param string|int $version The version to check whether it needs a migration.
	 * @return bool
	 */
	protected function should_run_migration( string|int $version ): bool {
		return version_compare( $version, $this->plugin_version, '<' );
	}

	/**
	 * Runs the plugin migration.
	 *
	 * @param string $new_version New plugin version.
	 * @param string $old_version Old/previous plugin version.
	 * @return void
	 */
	abstract protected function run_migration_steps( string $new_version, string $old_version ): void;
}
