<?php

namespace GoodWP\Altinator\Vendor\GoodWP\Common\Helper;

use DateTimeInterface;

/**
 * Sanitization helpers.
 *
 * @package GoodWP\Common\Helper
 */
abstract class Sanitize {


    /**
     * Sanitize a date value and return the datetime in the iso8601/atom format.
     *
     * @param mixed $value The value to check/sanitize.
     * @return null|string
     */
	public static function sanitize_date( $value ): ?string {
		// Try to create a valid date from the date string.
		$date_test = date_create( $value );
		if ( ! $date_test ) {
			return null;
		}
		// Make sure to return datetime in the iso8601/atom format.
		return $date_test->format( DateTimeInterface::ATOM );
	}

    /**
     * Sanitize a boolean.
     *
     * @param mixed $value The value to check/sanitize.
     * @return bool
     */
	public static function sanitize_boolean( $value ): bool {
		return filter_var( $value, FILTER_VALIDATE_BOOLEAN );
	}

    /**
     * Sanitize rich text value which can containe some HTML.
     *
     * @param string $value The value to check/sanitize.
     * @return string
     */
	public static function sanitize_rich_text( string $value ): string {
		// TODO: maybe apply custom context (like acf) as string and hook into wp_kses_allowed_html.
		return wp_kses(
			$value,
			[
				'a'      => array(
					'href'     => true,
					'rel'      => true,
					'rev'      => true,
					'name'     => true,
					'target'   => true,
					'download' => array(
						'valueless' => 'y',
					),
				),
				'b'      => array(),
				'br'     => array(),
				'em'     => array(),
				'i'      => array(),
				'p'      => array(
					'align' => true,
				),
				'span'   => array(
					'align' => true,
				),
				'small'  => array(),
				'strike' => array(),
				'strong' => array(),
			]
		);
	}

	/**
	 * Clean variables using sanitize_text_field. Arrays are cleaned recursively.
	 * Non-scalar values are ignored.
	 *
	 * @param mixed $value The value to check/sanitize.
	 * @return mixed
	 */
	public static function sanitize( $value ) {
		if ( is_array( $value ) ) {
			return array_map( [ self::class, 'sanitize' ], $value );
		} else {
			return is_scalar( $value ) ? sanitize_text_field( $value ) : $value;
		}
	}
}
