<?php

namespace GoodWP\Altinator\Vendor\GoodWP\Common\Events;

/**
 * An Event_Manager providing methods to do hooks and filters.
 *
 * @package GoodWP\Common\Events
 */
interface Event_Manager_Contract {
    /**
     * Adds a callback function to a filter hook.
     *
     * @see add_filter
     *
     * @param string   $hook_name     The name of the filter to add the callback to.
     * @param callable $callback      The callback to be run when the filter is applied.
     * @param int      $priority      Optional. Used to specify the order in which the functions
     *                                associated with a particular filter are executed.
     *                                Lower numbers correspond with earlier execution,
     *                                and functions with the same priority are executed
     *                                in the order in which they were added to the filter. Default 10.
     * @param int      $accepted_args Optional. The number of arguments the function accepts. Default 1.
     * @return true Always returns true.
     */
	public function add_filter( string $hook_name, callable $callback, int $priority = 10, int $accepted_args = 1 ): bool;

    /**
     * Calls the callback functions that have been added to a filter hook.
     *
     * @see apply_filters
     *
     * @param string $hook_name The name of the filter hook.
     * @param mixed  $value     The value to filter.
     * @param mixed  ...$args   Optional. Additional parameters to pass to the callback functions.
     * @return mixed The filtered value after all hooked functions are applied to it.
     */
	public function apply_filters( string $hook_name, mixed $value, ...$args ): mixed;

    /**
     * Checks if any filter has been registered for a hook.
     *
     * When using the `$callback` argument, this function may return a non-boolean value
     * that evaluates to false (e.g. 0), so use the `===` operator for testing the return value.
     *
     * @see has_filter
     *
     * @param string         $hook_name The name of the filter hook.
     * @param callable|false $callback  Optional. The callback to check for.
     *                                               This function can be called unconditionally to speculatively check
     *                                               a callback that may or may not exist. Default false.
     * @return bool|int If `$callback` is omitted, returns boolean for whether the hook has
     *                  anything registered. When checking a specific function, the priority
     *                  of that hook is returned, or false if the function is not attached.
     */
	public function has_filter( string $hook_name, callable|false $callback = false ): bool|int;

    /**
     * Removes a callback function from a filter hook.
     *
     * @see remove_filter
     *
     * @param string   $hook_name The filter hook to which the function to be removed is hooked.
     * @param callable $callback  The callback to be removed from running when the filter is applied.
     *                                         This function can be called unconditionally to speculatively remove
     *                                         a callback that may or may not exist.
     * @param int      $priority  Optional. The exact priority used when adding the original
     *                            filter callback. Default 10.
     * @return bool Whether the function existed before it was removed.
     */
	public function remove_filter( string $hook_name, callable $callback, int $priority = 10 ): bool;

    /**
     * Removes all the callback functions from a filter hook.
     *
     * @see remove_all_filters
     *
     * @param string    $hook_name The filter to remove callbacks from.
     * @param int|false $priority  Optional. The priority number to remove them from.
     *                             Default false.
     * @return true Always returns true.
     */
	public function remove_all_filters( string $hook_name, int|false $priority = false ): bool;

    /**
     * Adds a callback function to an action hook.
     *
     * @see add_action
     *
     * @param string   $hook_name       The name of the action to add the callback to.
     * @param callable $callback        The callback to be run when the action is called.
     * @param int      $priority        Optional. Used to specify the order in which the functions
     *                                  associated with a particular action are executed.
     *                                  Lower numbers correspond with earlier execution,
     *                                  and functions with the same priority are executed
     *                                  in the order in which they were added to the action. Default 10.
     * @param int      $accepted_args   Optional. The number of arguments the function accepts. Default 1.
     * @return true Always returns true.
     */
	public function add_action( string $hook_name, callable $callback, int $priority = 10, int $accepted_args = 1 ): bool;

    /**
     * Calls the callback functions that have been added to an action hook.
     *
     * @see do_action
     *
     * @param string $hook_name The name of the action to be executed.
     * @param mixed  ...$arg    Optional. Additional arguments which are passed on to the
     *                          functions hooked to the action. Default empty.
     */
	public function do_action( string $hook_name, ...$arg ): void;

    /**
     * Checks if any action has been registered for a hook.
     *
     * When using the `$callback` argument, this function may return a non-boolean value
     * that evaluates to false (e.g. 0), so use the `===` operator for testing the return value.
     *
     * @see has_action
     *
     * @param string         $hook_name The name of the action hook.
     * @param callable|false $callback  Optional. The callback to check for.
     *                                               This function can be called unconditionally to speculatively check
     *                                               a callback that may or may not exist. Default false.
     * @return bool|int If `$callback` is omitted, returns boolean for whether the hook has
     *                  anything registered. When checking a specific function, the priority
     *                  of that hook is returned, or false if the function is not attached.
     */
	public function has_action( string $hook_name, callable|false $callback = false ): bool|int;

    /**
     * Removes a callback function from an action hook.
     *
     * @see remove_action
     *
     * @param string   $hook_name The action hook to which the function to be removed is hooked.
     * @param callable $callback  The name of the function which should be removed.
     *                                         This function can be called unconditionally to speculatively remove
     *                                         a callback that may or may not exist.
     * @param int      $priority  Optional. The exact priority used when adding the original
     *                            action callback. Default 10.
     * @return bool Whether the function is removed.
     */
	public function remove_action( string $hook_name, callable $callback, int $priority = 10 ): bool;

    /**
     * Removes all the callback functions from an action hook.
     *
     * @see remove_all_actions
     *
     * @param string    $hook_name The action to remove callbacks from.
     * @param int|false $priority  Optional. The priority number to remove them from.
     *                             Default false.
     * @return true Always returns true.
     */
	public function remove_all_actions( string $hook_name, int|false $priority = false ): bool;
}
