<?php

namespace GoodWP\Altinator\Vendor\GoodWP\Common\Errors;

use WP_Error;

/**
 * A service to collect errors and display them.
 * Stores them in-memory during a request and does not persist them.
 */
class Error_Collector implements Error_Collector_Contract {

    /**
     * All collected errors separated by context
     *
     * @var array<string,array<string,WP_Error>>
     */
    protected array $errors;

    /**
     * Creates a new ErrorService instance
     */
    public function __construct() {
        $this->errors = [];
    }

    public function add_error( WP_Error $error, string $context ): void {
        if ( ! isset( $this->errors[ $context ] ) ) {
            $this->errors[ $context ] = [];
        }
        $this->errors[ $context ][ $error->get_error_code() ] = $error;
    }

    public function has_errors( ?string $context = null ): bool {
        if ( $context ) {
            return isset( $this->errors[ $context ] ) && ! empty( $this->errors[ $context ] );
        }
        return ! empty( $this->errors );
    }

    public function has_error_code( string $error_code, ?string $context = null ): bool {
        if ( $context ) {
            return ! empty( $this->errors[ $context ][ $error_code ] );
        } else {
            foreach ( $this->errors as $context_errors ) {
                foreach ( $context_errors as $ec => $error ) {
                    if ( $ec === $error_code ) {
                        return true;
                    }
                }
            }
        }
        return false;
    }

    /**
     * {@inheritDoc}
     *
     * @return array<string, WP_Error>
     */
    public function get_errors( string $context ): array {
        if ( isset( $this->errors[ $context ] ) ) {
            return $this->errors[ $context ];
        }
        return [];
    }

    public function get_all_errors(): array {
        return $this->errors;
    }
}
