<?php

namespace GoodWP\Altinator\Vendor\GoodWP\Common\DI;

use Exception;
use GoodWP\Altinator\Vendor\GoodWP\Common\Contracts\Bootable;
use GoodWP\Altinator\Vendor\lucatume\DI52\ContainerException;

/**
 * The main class for a plugin, theme, site-part which initializes DI-Container, core services and everything else.
 */
abstract class App implements Bootable {

	/**
	 * The DI container
	 *
	 * @var Container
	 */
	protected Container $container;

    /**
     * Service providers to initialize
     *
     * @var class-string<Service_Provider>[] Array of classnames of service providers - must all extend \GoodWP\Common\DI\ServiceProvider.
     */
	protected array $service_providers = [];

	/**
	 * Service providers to initialize only in wp-admin
	 *
	 * @var class-string<Service_Provider>[] Array of classnames of service providers - must all extend \GoodWP\Common\DI\ServiceProvider.
	 */
	protected array $admin_service_providers = [];

    /**
     * Create the container and set initival values.
     *
     * @param Container|null $base_container The base from which to work. Allows a plugin to setup basic settings.
     *                                       E.g. dependency injection for service providers.
     *
     * @return Container
     * @throws Exception If container is already initialized.
     */
	protected function init_container( ?Container $base_container = null ): Container {
        if ( isset( $this->container ) ) {
            throw new Exception( 'Container was already initialized' );
        }

        $container = $base_container ?? new Container();

		/**
		 * Init Service Providers
		 */
        $service_providers = $this->service_providers;
		if ( is_admin() ) {
			$service_providers = array_merge( $this->service_providers, $this->admin_service_providers );
		}

        // Register service providers.
		foreach ( $service_providers as $provider_class ) {
			$container->register( $provider_class );
		}

        $this->container = $container;
        return $container;
    }

    /**
     * Initialize this app instance
     * Init all services, register it in WordPress, bootup services.
     *
     * @return void
     * @throws Exception If container is already initialized.
     */
    public function boot(): void {
        $this->init_container();
        $this->container->boot();
    }

    /**
     * Get the DI container instance.
     *
     * @return Container
     * @throws Exception Throws exception if container/plugin is not initialized yet.
     */
    public function get_container(): Container {
        if ( ! isset( $this->container ) ) {
            throw new Exception( 'Container was not initialized yet.' );
        }
        return $this->container;
    }

    /**
     * Finds an entry of the container by its identifier and returns it.
     *
     * @template T
     *
     * @param  string|class-string<T> $id  A fully qualified class or interface name or an already built object.
     *
     * @return T|mixed The entry for an id.
     * @phpstan-return ($id is class-string ? T : mixed)
     *
     * @throws ContainerException Error while retrieving the entry.
     */
    public function get( $id ) {
        return $this->container->get( $id );
    }
}
