<?php

namespace GoodWP\Altinator\Vendor\GoodWP\Common\Assets;

/**
 * Manager for registering and enqueueing scripts and styles.
 * It's a wrapper around the WP Scripts API and provides helpers.
 *
 * It's up to the implementing developer whether to register all assets on 'init' on each request,
 * or just doing that on the fly and enqueuing them directly after.
 */
interface Asset_Manager_Contract {

    /**
     * Enqueue a script by handle, has to be registered via register_script.
     *
     * @param string $handle The handle of the script to enqueue.
     * @return bool true if enqueued, false if not registered.
     */
    public function enqueue_script( string $handle ): bool;

    /**
     * Enqueue a style by handle, has to be registered via register_style.
     *
     * @param string $handle The handle of the style to enqueue.
     * @return bool true if enqueued, false if not registered-
     */
    public function enqueue_style( string $handle ): bool;

    /**
     * Adds data to the server so JavaScript can use it.
     * Must be called after the script is registered.
     *
     * @see  wp_add_inline_script
     *
     * @param string $handle The script handle to add data to.
     * @param string $object_name The name of the JavaScript (global) variable/object.
     * @param mixed  $data JSON encodable data.
     * @param string $position 'before' or 'after' (defaults to 'before' so the data is instantly available).
     * @return bool
     */
    public function add_script_data( string $handle, string $object_name, $data, string $position = 'before' ): bool;

    /**
     * Register a script which has an asset manifest built via @wordpress/scripts.
     *
     * @param string     $handle The handle to use for the script.
     * @param string     $file Filename / relative path inside "build" directory - WITHOUT file extension.
     * @param array      $dependencies An array of script handles this script depends on.
     * @param bool|array $args A boolean whether to load script in the footer or an array for the loading strategy. (@see wp_register_script).
     * @return string The full handle (=$prefix + $handle).
     */
    public function register_script( string $handle, string $file, array $dependencies = [], bool|array $args = false ): string;

    /**
     * Register a style which has an asset manifest built via @wordpress/scripts.
     *
     * @param string $handle The handle to use for the style.
     * @param string $file Filename / relative path inside "build" directory - WITHOUT file extension.
     * @param array  $dependencies An array of script handles this script depends on.
     * @return string The full handle (=$prefix + $handle).
     */
    public function register_style( string $handle, string $file, array $dependencies = [] ): string;

    /**
     * Parse an asset manifest built via @wordpress/scripts.
     *
     * @param string $handle The script handle.
     * @param string $file The file path.
     * @return array Parsed data containing dependencies and version.
     */
    public function parse_asset_file( string $handle, string $file ): array;
}
