<?php

namespace GoodWP\Altinator\Vendor\GoodWP\Common\Assets;

/**
 * Service for registering and enqueueing scripts and styles.
 * It's a wrapper around the WP Scripts API and provides helpers.
 *
 * It's up to the implementing developer whether to register all assets on 'init' on each request,
 * or just doing that on the fly and enqueuing them directly after.
 */
class Asset_Manager implements Asset_Manager_Contract, Block_Asset_Manager_Contract {

    /**
     * Base path of the plugin/theme, used for path/url generation.
     * Passed to wp_register_script/wp_register_style which prefixes it with WP_Scripts::$base_url.
     *
     * @var string
     */
    protected string $base_path;

    /**
     * Version, default for assets version.
     * Example: Set to pluginVersion.
     *
     * @var string
     */
    protected string $version;

    /**
     * Prefix for handles (including separator)
     *
     * @var string
     */
    protected string $handle_prefix = '';

    /**
     * The path of the built assets relative to the main plugin file/directory
     *
     * @var string
     */
    protected string $assets_build_directory;

    /**
     * Keeps a log of all scripts registered via this service
     * so only those get enqueued
     *
     * @see enqueueScript
     *
     * @var array
     */
    protected array $registered_scripts;

    /**
     * Keeps a log of all styles registered via this service
     * so only those get enqueued
     *
     * @see enqueueStyle
     *
     * @var array
     */
    protected array $registered_styles;

    /**
	 * Creates a new Asset_Manager instance
     *
     * @param string $assets_build_directory The path of the built assets relative to the main plugin file/directory.
     * @param string $base_path Base path of the plugin/theme, used for path/url generation.
     * @param string $version Version, default for assets version.
     * @param string $handle_prefix The prefix to be used for actions/filters in this service (including separator).
     */
    public function __construct(
        string $assets_build_directory,
        string $base_path,
        string $version,
        string $handle_prefix = ''
    ) {
        $this->assets_build_directory = $assets_build_directory;
        $this->base_path              = $base_path;
        $this->version                = $version;
        $this->handle_prefix          = $handle_prefix;

        $this->registered_scripts = [];
        $this->registered_styles  = [];
    }

    /**
     * {@inheritDoc}
     */
    public function enqueue_script( string $handle ): bool {
        if ( ! isset( $this->registered_scripts[ $handle ] ) ) {
            return false;
        }

        $full_handle = $this->handle_prefix . $handle;
        wp_enqueue_script( $full_handle );
        return true;
    }

    /**
     * {@inheritDoc}
     */
    public function enqueue_style( string $handle ): bool {
        if ( ! isset( $this->registered_styles[ $handle ] ) ) {
            return false;
        }

        $full_handle = $this->handle_prefix . $handle;
        wp_enqueue_style( $full_handle );
        return true;
    }

    /**
     * {@inheritDoc}
     */
    public function add_script_data( string $handle, string $object_name, $data, string $position = 'before' ): bool {
        if ( ! isset( $this->registered_scripts[ $handle ] ) ) {
            return false;
        }

        $script      = "var $object_name = " . wp_json_encode( $data ) . ';';
        $full_handle = $this->handle_prefix . $handle;
        return wp_add_inline_script( $full_handle, $script, $position === 'before' ? 'before' : 'after' );
    }

    /**
     * {@inheritDoc}
     */
    public function register_script( string $handle, string $file, array $dependencies = [], bool|array $args = false ): string {
        $asset = $this->parse_asset_file( $handle, $file );
        if ( ! empty( $dependencies ) ) {
            $asset['dependencies'] = array_unique( array_merge( $dependencies, $asset['dependencies'] ) );
        }

        $relative_path = ltrim( $asset['relative_path'], '/' );
        wp_register_script(
            $asset['handle'],
            content_url( $relative_path . '.js' ),
            $asset['dependencies'],
            $asset['version'],
            $args
        );

        $this->registered_scripts[ $handle ] = $file;

        return $asset['handle'];
    }

    /**
     * {@inheritDoc}
     */
    public function parse_asset_file( string $handle, string $file ): array {
        $relative_path   = $this->base_path . DIRECTORY_SEPARATOR . $this->assets_build_directory . DIRECTORY_SEPARATOR . $file;
        $asset_file_path = WP_CONTENT_DIR . "{$relative_path}.asset.php";

        $dependencies = [];
        $version      = $this->version;

        if ( file_exists( $asset_file_path ) ) {
            $asset_file = require $asset_file_path;
            // Dependencies are for JS deps. Should be used for JS only.
            $dependencies = $asset_file['dependencies'] ?? [];
            $version      = $asset_file['version'] ?? $this->version;
        }

        return [
            'handle'        => $this->handle_prefix . $handle,
            'relative_path' => $relative_path,
            'version'       => $version,
            'dependencies'  => $dependencies,
        ];
    }

    /**
     * {@inheritDoc}
     */
    public function register_style( string $handle, string $file, array $dependencies = [] ): string {
        $asset = $this->parse_asset_file( $handle, $file );

        $relative_path = ltrim( $asset['relative_path'], '/' );
        wp_register_style(
            $asset['handle'],
            content_url( $relative_path . '.css' ),
            $dependencies, // Asset file contains JS-dependencies, which not all might have a style.
            $asset['version'],
        );

        $this->registered_styles[ $handle ] = $file;

        return $asset['handle'];
    }

    /**
     * {@inheritDoc}
     */
    public function get_block_file_path( string $block_name ): string {
        $path = $this->base_path . DIRECTORY_SEPARATOR . $this->assets_build_directory;
        return $path . DIRECTORY_SEPARATOR . $block_name . '.json';
    }
}
