<?php

namespace GoodWP\Altinator\Vendor\GoodWP\Admin\Posts;

use GoodWP\Altinator\Vendor\GoodWP\Admin\Posts\Lists\With_Filters;
use GoodWP\Altinator\Vendor\GoodWP\Admin\Screens\List_Table_Screen;
use GoodWP\Altinator\Vendor\GoodWP\Common\Helper\Sanitize;
use GoodWP\Altinator\Vendor\GoodWP\Common\Helper\Str;
use WP_Post;

abstract class Post_List_Screen extends List_Table_Screen {

    use With_Filters;

    /**
     * The post type
     *
     * @var string
     */
    protected const SCREEN_ID = '';

    protected array $current_filter_values     = [];
    protected array $post_list_filters = [
        'select'   => Lists\Filters\Select_Filter::class,
        'taxonomy' => Lists\Filters\Taxonomy_Filter::class,
        'hidden'   => Lists\Filters\Hidden_Filter::class,
    ];

    /**
     * Whether to store the current filter values for the current user in session.
     * So they are applied again when the user returns to the list without any filters.
     *
     * @var boolean
     */
    protected bool $store_filters_session = false;

    /**
     * {@inheritDoc}
     */
    public function boot(): void {
        parent::boot();
        add_action( 'load-edit.php', [ $this, 'load' ] );
    }

    public function load(): void {
        parent::load();
        if ( $this->is_current_screen() ) {
            add_action( 'restrict_manage_posts', [ $this, 'maybe_render_filters' ] );
            add_filter( 'request', [ $this, 'maybe_handle_filters' ] );
            add_action( 'manage_posts_extra_tablenav', [ $this, 'maybe_render_action' ] );

            add_filter( 'manage_' . static::SCREEN_ID . '_posts_custom_column', [ $this, 'render_column' ], 10, 2 );

            if ( is_post_type_hierarchical( static::SCREEN_ID ) ) {
                add_filter( 'page_row_actions', [ $this, 'maybe_row_actions' ], 10, 2 );
            } else {
                add_filter( 'post_row_actions', [ $this, 'maybe_row_actions' ], 10, 2 );
            }
        }
    }

    /**
     * Gets all available filters
     *
     * @return array
     *      $name (string) => [
     *          'type' => (string) a registered filter type
     *          [...] other config for the filter
     *      ]
     */
    protected function get_filters(): array {
        return [];
    }

    /**
     * Handle any filters.
     *
     * @param array $query Query vars.
     * @return array
     */
    public function maybe_handle_filters( array $query ): array {
        global $typenow;

        if ( $this->is_current_screen() ) {
            // Use $_GET variable because not all filters have registered query vars
            // and query vars are intended for frontend usage.
            $all_query_vars = array_merge( Sanitize::sanitize( $_GET ), $query );
            $this->current_filter_values = $this->parse_filters( $all_query_vars );
            return $this->map_filters_to_query( $all_query_vars );
        }

        return $query;
    }

    /**
     * Handle any custom filters.
     *
     * @param array $query Query vars.
     * @return array
     */
    protected function parse_filters( array $query ): array {
        $filter_values = [];
        $default_values = [];
        foreach ( $this->get_filters() as $key => $config ) {
            $query_var = isset( $config['query_var'] ) ? $config['query_var'] : $key;
            if ( isset( $query[ $query_var ] ) ) {
                $filter_values[ $key ] = $query[ $query_var ];
            }
            if ( array_key_exists( 'default', $config ) ) {
                $default_values[ $key ] = $config['default'];
            }
        }
        if ( $this->store_filters_session ) {
            if ( empty( $filter_values ) ) {
                $stored_filter_values = get_user_option( static::SCREEN_ID . '_filters', get_current_user_id(), );
				if ( is_array( $stored_filter_values ) ) {
					$filter_values = $stored_filter_values;
				}
            } else {
                update_user_option( get_current_user_id(), static::SCREEN_ID . '_filters', $filter_values );
            }
        }
        return wp_parse_args( $filter_values, $default_values );
    }

    /**
     * Maps the current filters value on to a query array.
     *
     * @param array $query
     * @return array
     */
    protected function map_filters_to_query( array $query ): array {
        foreach ( $this->get_filters() as $key => $config ) {
            $query_var = isset( $config['query_var'] ) ? $config['query_var'] : $key;
            if ( isset( $this->current_filter_values[ $key ] ) ) {
                $query[ $query_var ] = $this->current_filter_values[ $key ];
            }
        }
        return $query;
    }

    /**
     * See if we should render search filters or not.
     *
     * @return void
     */
    public function maybe_render_filters(): void {
        global $typenow;

        if ( static::SCREEN_ID === $typenow ) {
            $this->render_filters();
        }
    }

    /**
     * Renders any extra actions for the whole table (eg export)
     *
     * @param string $which
     * @return void
     */
    public function maybe_render_action( $which ): void {
        global $typenow;
        if ( static::SCREEN_ID === $typenow ) {
            $this->render_actions();
        }
    }

    /**
     * Render any custom filters and search inputs for the list table.
     *
     * @return void
     */
    protected function render_filters(): void {
        $this->render_post_filters();
    }

    /**
     * Render any custom actions for the whole table (eg export)
     *
     * @return void
     */
    protected function render_actions(): void {
    }

    /**
     * Render individual columns.
     *
     * @param string $column Column ID to render.
     * @param int    $post_id Post ID being shown.
     */
    public function render_column( string $column, int $post_id ) {
        $function = 'render_' . Str::snake( $column ) . '_column';
        if ( is_callable( [ $this, $function ] ) ) {
            $this->$function( $post_id );
        }
    }

    /**
     * Set row actions.
     *
     * @param array   $actions Array of actions.
     * @param WP_Post $post Current post object.
     * @return array
     */
    public function maybe_row_actions( array $actions, WP_Post $post ): array {
        if ( static::SCREEN_ID === $post->post_type ) {
            return $this->row_actions( $actions, $post );
        }
        return $actions;
    }

    /**
     * Get row actions to show in the list table.
     *
     * @param array   $actions Array of actions.
     * @param WP_Post $post Current post object.
     * @return array
     */
    protected function row_actions( array $actions, WP_Post $post ): array {
        return $actions;
    }

    public function is_current_screen(): bool {
        $is_current_screen = parent::is_current_screen();
        if ( ! $is_current_screen ) {
            return false;
        }
        $current_screen = get_current_screen();

        return $current_screen && $current_screen->post_type === static::SCREEN_ID;
    }
}
