<?php

namespace GoodWP\Altinator\Vendor\GoodWP\Admin\Posts;

use GoodWP\Altinator\Vendor\GoodWP\Admin\Screens\Admin_Screen;
use GoodWP\Altinator\Vendor\GoodWP\Common\Assets\Asset_Manager_Contract;
use GoodWP\Altinator\Vendor\GoodWP\Common\Assets\Has_Assets;
use WP_Post;

/**
 * Block-editor based screen for event post type.
 */
abstract class Post_Block_Editor_Screen extends Admin_Screen {
    use Has_Assets;

    /**
     * The post type
     *
     * @var string
     */
    protected const POST_TYPE = '';

    /**
     * Create new screen instance.
     *
     * @param Asset_Manager_Contract $asset_manager The asset manager to enqueue block editor assets.
     */
    public function __construct(
        Asset_Manager_Contract $asset_manager,
    ) {
        assert( ! empty( static::POST_TYPE ), 'POST_TYPE constant must be set on ' . static::class );
        $this->screen_id = static::POST_TYPE;

        $this->asset_manager = $asset_manager;
    }

    /**
     * {@inheritDoc}
     */
    public function boot(): void {
        if ( ! use_block_editor_for_post_type( static::POST_TYPE ) ) {
            return;
        }

        add_action( 'load-post.php', [ $this, 'load' ] );
        add_action( 'load-post-new.php', [ $this, 'load' ] );

        add_action( 'enqueue_block_editor_assets', [ $this, 'enqueue_assets' ] );
    }

    /**
     * {@inheritDoc}
     */
    public function load(): void {
        $this->register_assets();
        if ( $this->is_current_screen() ) {
            add_action( 'save_post_' . static::POST_TYPE, [ $this, 'save_post' ], 10, 3 );
        }
    }

    /**
     * Registers all the assets for the block editor.
     * Do this on all screens, because with site-editor/page-editor,
     * user could load a different post type, but then switch to event.
     *
     * @return void
     */
    protected function register_assets(): void {
    }

    /**
     * Enqueues all the assets for the block editor.
     * Do this on all screens, because with site-editor/page-editor,
     * user could load a different post type, but then switch to event.
     *
     * @return void
     */
    public function enqueue_assets(): void {
    }

    /**
     * {@inheritDoc}
     */
    public function is_current_screen(): bool {
        if ( ! is_admin() ) {
            return false;
        }

        assert( ! empty( $this->screen_id ), 'screen_id constant must be set to post type on ' . static::class );

        $current_screen = get_current_screen();
        return $current_screen &&
            $current_screen->id === $this->screen_id &&
            $current_screen->post_type === static::POST_TYPE &&
            $current_screen->is_block_editor();
    }

    /**
     * Handle saving your custom meta boxes and fields here.
     *
     * @param int     $post_id
     * @param WP_Post $post
     * @param bool    $update
     * @return void
     */
    public function save_post( int $post_id, WP_Post $post, bool $update ) {
    }
}
