<?php
/**
 * Main plugin bootstrap class
 *
 * @package AltAudit
 * @since 1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Main plugin bootstrap class
 *
 * @since 1.0.0
 */
final class Altaudit82ai {


	/**
	 * Plugin instance
	 *
	 * @var Altaudit82ai
	 */
	private static $instance = null;

	/**
	 * Container instance
	 *
	 * @var Altaudit82ai_Container
	 */
	private $container;

	/**
	 * Plugin initialization flag
	 *
	 * @var bool
	 */
	private $initialized = false;

	/**
	 * Get plugin instance
	 *
	 * @return Altaudit82ai
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		$this->setup_autoloader();
		$this->init_hooks();
	}

	/**
	 * Setup autoloader for plugin classes
	 *
	 * @since 1.0.0
	 */
	private function setup_autoloader() {
		// Custom autoloader for plugin classes.
		spl_autoload_register( array( $this, 'autoload' ) );
	}

	/**
	 * Autoload plugin classes
	 *
	 * @param string $class_name The class name to load.
	 */
	public function autoload( $class_name ) {
		// Only autoload Altaudit82ai classes.
		if ( 0 !== strpos( $class_name, 'Altaudit82ai_' ) ) {
			return;
		}

		// Convert class name to file path.
		$class_name = str_replace( 'Altaudit82ai_', '', $class_name );
		$class_name = str_replace( '_', '-', strtolower( $class_name ) );
		$file_path  = ALTAUDIT82AI_INCLUDES_DIR . 'class-altaudit82ai-' . $class_name . '.php';

		// Check for file in subdirectories.
		if ( ! file_exists( $file_path ) ) {
			$subdirs = array(
				'Controllers',
				'Models',
				'Views',
				'Services',
				'Services/Quality',
				'Services/Analysis',
				'Services/Validation',
				'Container',
			);
			foreach ( $subdirs as $subdir ) {
				$subdir_path = ALTAUDIT82AI_INCLUDES_DIR . $subdir . '/class-altaudit82ai-' . $class_name . '.php';
				if ( file_exists( $subdir_path ) ) {
					$file_path = $subdir_path;
					break;
				}
			}
		}

		// Load the file if it exists.
		if ( file_exists( $file_path ) ) {
			require_once $file_path;
		}
	}

	/**
	 * Initialize hooks
	 *
	 * @since 1.0.0
	 */
	private function init_hooks() {
		// Core WordPress hooks.
		add_action( 'init', array( $this, 'init' ) );

		// Activation and deactivation hooks.
		register_activation_hook( ALTAUDIT82AI_PLUGIN_FILE, array( $this, 'activate' ) );
		register_deactivation_hook( ALTAUDIT82AI_PLUGIN_FILE, array( $this, 'deactivate' ) );

		// Admin hooks.
		if ( is_admin() ) {
			add_action( 'admin_init', array( $this, 'admin_init' ) );
			add_action( 'admin_menu', array( $this, 'admin_menu' ) );
			add_action( 'admin_enqueue_scripts', array( $this, 'admin_enqueue_scripts' ) );
		}

		// Frontend hooks.
		add_action( 'wp_enqueue_scripts', array( $this, 'frontend_enqueue_scripts' ) );

		// AJAX hooks.
		add_action( 'wp_ajax_altaudit82ai_generate', array( $this, 'ajax_generate_alt_text' ) );
		add_action( 'wp_ajax_altaudit82ai_test_connection', array( $this, 'ajax_test_connection' ) );
		add_action( 'wp_ajax_altaudit82ai_bulk_process', array( $this, 'ajax_bulk_process' ) );

		// Gutenberg hooks.
		add_action( 'enqueue_block_editor_assets', array( $this, 'enqueue_block_editor_assets' ) );

		// Media library hooks.
		add_filter( 'attachment_fields_to_edit', array( $this, 'attachment_fields_to_edit' ), 10, 2 );
		add_filter( 'attachment_fields_to_save', array( $this, 'attachment_fields_to_save' ), 10, 2 );
	}

	/**
	 * Initialize plugin
	 *
	 * @since 1.0.0
	 */
	public function init() {
		if ( $this->initialized ) {
			return;
		}

		// Register custom taxonomy for status filtering.
		$taxonomy = new Altaudit82ai_Taxonomy();
		$taxonomy->register();

		// Check and create database tables if they don't exist.
		// This handles cases where plugin was copied without activation.
		$this->maybe_create_tables();

		// Initialize container.
		$this->container = new Altaudit82ai_Container();
		$this->register_services();

		// Set initialization flag.
		$this->initialized = true;

		/**
		 * Plugin initialized action
		 *
		 * @param Altaudit82ai $plugin The plugin instance.
		 */
		do_action( 'altaudit82ai_initialized', $this );
	}

	/**
	 * Check and create database tables if they don't exist
	 *
	 * @since 1.0.0
	 */
	private function maybe_create_tables() {
		// No custom tables needed - using wp_options for settings.
	}

	/**
	 * Register services in container
	 *
	 * @since 1.0.0
	 */
	private function register_services() {
		// Register core services.
		$this->container->register( 'api_client', 'Altaudit82ai_Api_Client' );
		$this->container->register( 'settings', 'Altaudit82ai_Settings' );
		$this->container->register( 'template_service', 'Altaudit82ai_Template_Service' );

		// Register Quality MVC components.
		$this->register_quality_services();

		// Register controllers (after models and services they depend on).
		$this->container->register( 'admin_controller', 'Altaudit82ai_Admin_Controller' );
		$this->container->register( 'api_controller', 'Altaudit82ai_Api_Controller' );
		$this->container->register( 'gutenberg_controller', 'Altaudit82ai_Gutenberg_Controller' );
		$this->container->register( 'media_controller', 'Altaudit82ai_Media_Controller' );
		$this->container->register( 'quality_controller', 'Altaudit82ai_Quality_Controller' );

		// Initialize services that need early initialization.
		$this->container->get( 'admin_controller' );
		$this->container->get( 'api_controller' );
		$this->container->get( 'gutenberg_controller' );
		$this->container->get( 'media_controller' );
		$this->container->get( 'quality_controller' )->init();
	}

	/**
	 * Register Quality system services
	 *
	 * @since 1.0.0
	 */
	private function register_quality_services() {
		// Register Quality Models.
		$this->container->register( 'quality_criteria_model', 'Altaudit82ai_Quality_Criteria_Model' );
		$this->container->register( 'quality_result_model', 'Altaudit82ai_Quality_Result_Model' );

		// Register Quality Services.
		$this->container->factory(
			'quality_analyzer',
			function ( $container ) {
				return new Altaudit82ai_Quality_Analyzer( $container->get( 'quality_criteria_model' ) );
			}
		);

		$this->container->factory(
			'quality_scorer',
			function ( $container ) {
				return new Altaudit82ai_Quality_Scorer( $container->get( 'quality_criteria_model' ) );
			}
		);

		$this->container->factory(
			'quality_service',
			function ( $container ) {
				return new Altaudit82ai_Quality_Service( $container );
			}
		);

		// Register Quality Analyzers.
		$this->container->factory(
			'wcag_analyzer',
			function () {
				return new Altaudit82ai_WCAG_Analyzer();
			}
		);

		$this->container->factory(
			'content_analyzer',
			function () {
				return new Altaudit82ai_Content_Analyzer();
			}
		);

		// Register Quality Validators.
		$this->container->register( 'length_validator', 'Altaudit82ai_Length_Validator' );

		// Register Quality Checkers.
		$this->container->register( 'missing_checker', 'Altaudit82ai_Missing_Checker' );
		$this->container->register( 'decorative_checker', 'Altaudit82ai_Decorative_Checker' );
		$this->container->register( 'weak_checker', 'Altaudit82ai_Weak_Checker' );
		$this->container->register( 'good_checker', 'Altaudit82ai_Good_Checker' );
		$this->container->register( 'excellent_checker', 'Altaudit82ai_Excellent_Checker' );

		// Legacy compatibility - register old quality checker with new implementation.
		$this->container->factory(
			'quality_checker',
			function ( $container ) {
				return $container->get( 'quality_service' );
			}
		);
	}

	/**
	 * Get service from container
	 *
	 * @param string $service_name The service name.
	 * @return mixed The service instance.
	 */
	public function get_service( $service_name ) {
		return $this->container->get( $service_name );
	}

	/**
	 * Plugin activation
	 *
	 * @since 1.0.0
	 */
	public function activate() {
		// Set default options (using wp_options).
		$this->set_default_options();

		// Schedule any cron jobs.
		$this->schedule_cron_jobs();

		// Flush rewrite rules.
		flush_rewrite_rules();

		/**
		 * Plugin activation action
		 */
		do_action( 'altaudit82ai_activated' );
	}

	/**
	 * Plugin deactivation
	 *
	 * @since 1.0.0
	 */
	public function deactivate() {
		// Clear scheduled cron jobs.
		$this->clear_cron_jobs();

		// Flush rewrite rules.
		flush_rewrite_rules();

		/**
		 * Plugin deactivation action
		 */
		do_action( 'altaudit82ai_deactivated' );
	}

	/**
	 * Set default options
	 *
	 * @since 1.0.0
	 */
	private function set_default_options() {
		$default_options = array(
			'altaudit82ai_api_key'           => '',
			'altaudit82ai_api_url'           => ALTAUDIT82AI_API_BASE_URL,
			'altaudit82ai_auto_generate'     => 'no',
			'altaudit82ai_quality_threshold' => 70,
			'altaudit82ai_max_length'        => 125,
			'altaudit82ai_style'             => 'descriptive',
			'altaudit82ai_context_aware'     => 'yes',
			'altaudit82ai_bulk_processing'   => 'yes',
			'altaudit82ai_cache_results'     => 'yes',
			'altaudit82ai_debug_mode'        => 'no',
		);

		foreach ( $default_options as $option_name => $default_value ) {
			add_option( $option_name, $default_value );
		}
	}

	/**
	 * Schedule cron jobs
	 *
	 * @since 1.0.0
	 */
	private function schedule_cron_jobs() {
		// Schedule daily cleanup of logs.
		if ( ! wp_next_scheduled( 'altaudit82ai_daily_cleanup' ) ) {
			wp_schedule_event( time(), 'daily', 'altaudit82ai_daily_cleanup' );
		}
	}

	/**
	 * Clear cron jobs
	 *
	 * @since 1.0.0
	 */
	private function clear_cron_jobs() {
		wp_clear_scheduled_hook( 'altaudit82ai_daily_cleanup' );
	}

	/**
	 * Initialize admin area
	 *
	 * @since 1.0.0
	 */
	public function admin_init() {
		// Admin initialization handled by Admin_Controller.
		if ( $this->container ) {
			$admin_controller = $this->container->get( 'admin_controller' );
			if ( method_exists( $admin_controller, 'admin_init' ) ) {
				$admin_controller->admin_init();
			}
		}
	}

	/**
	 * Add admin menu
	 *
	 * @since 1.0.0
	 */
	public function admin_menu() {
		// Admin menu handled by Admin_Controller.
		if ( $this->container ) {
			$admin_controller = $this->container->get( 'admin_controller' );
			if ( method_exists( $admin_controller, 'admin_menu' ) ) {
				$admin_controller->admin_menu();
			}
		}
	}

	/**
	 * Enqueue admin scripts and styles
	 *
	 * @param string $hook_suffix The current admin page.
	 */
	public function admin_enqueue_scripts( $hook_suffix ) {
		// Admin assets handled by Admin_Controller.
		if ( $this->container ) {
			$admin_controller = $this->container->get( 'admin_controller' );
			if ( method_exists( $admin_controller, 'enqueue_scripts' ) ) {
				$admin_controller->enqueue_scripts( $hook_suffix );
			}
		}
	}

	/**
	 * Enqueue frontend scripts and styles
	 */
	public function frontend_enqueue_scripts() {
	}

	/**
	 * Enqueue block editor assets
	 */
	public function enqueue_block_editor_assets() {
		// Gutenberg assets handled by Gutenberg_Controller.
		if ( $this->container ) {
			$gutenberg_controller = $this->container->get( 'gutenberg_controller' );
			if ( method_exists( $gutenberg_controller, 'enqueue_block_editor_assets' ) ) {
				$gutenberg_controller->enqueue_block_editor_assets();
			}
		}
	}

	/**
	 * Handle AJAX request for alt text generation
	 */
	public function ajax_generate_alt_text() {
		// AJAX handled by API_Controller.
		if ( $this->container ) {
			$api_controller = $this->container->get( 'api_controller' );
			if ( method_exists( $api_controller, 'ajax_generate_alt_text' ) ) {
				$api_controller->ajax_generate_alt_text();
			}
		}
	}

	/**
	 * Handle AJAX request for connection testing
	 */
	public function ajax_test_connection() {
		// AJAX handled by API_Controller.
		if ( $this->container ) {
			$api_controller = $this->container->get( 'api_controller' );
			if ( method_exists( $api_controller, 'ajax_test_connection' ) ) {
				$api_controller->ajax_test_connection();
			}
		}
	}

	/**
	 * Handle AJAX request for bulk processing
	 */
	public function ajax_bulk_process() {
		// AJAX handled by API_Controller.
		if ( $this->container ) {
			$api_controller = $this->container->get( 'api_controller' );
			if ( method_exists( $api_controller, 'ajax_bulk_process' ) ) {
				$api_controller->ajax_bulk_process();
			}
		}
	}

	/**
	 * Add fields to attachment edit form
	 *
	 * @param array   $form_fields The form fields.
	 * @param WP_Post $post        The attachment post object.
	 * @return array Modified form fields.
	 */
	public function attachment_fields_to_edit( $form_fields, $post ) {
		// Media library integration handled by Media_Controller.
		if ( $this->container ) {
			$media_controller = $this->container->get( 'media_controller' );
			if ( method_exists( $media_controller, 'attachment_fields_to_edit' ) ) {
				return $media_controller->attachment_fields_to_edit( $form_fields, $post );
			}
		}
		return $form_fields;
	}

	/**
	 * Save attachment fields
	 *
	 * @param array $post       The post data.
	 * @param array $attachment The attachment data.
	 * @return array The post data.
	 */
	public function attachment_fields_to_save( $post, $attachment ) {
		// Media library integration handled by Media_Controller.
		if ( $this->container ) {
			$media_controller = $this->container->get( 'media_controller' );
			if ( method_exists( $media_controller, 'attachment_fields_to_save' ) ) {
				return $media_controller->attachment_fields_to_save( $post, $attachment );
			}
		}
		return $post;
	}

	/**
	 * Get supported MIME types for image analysis
	 *
	 * Centralized list of supported image types for all plugin features.
	 *
	 * @since 1.0.0
	 * @return array List of MIME types.
	 */
	public static function get_supported_mime_types() {
		return array(
			'image/jpeg',
			'image/jpg',
			'image/jfif',
			'image/jpe',
			'image/png',
			'image/gif',
			'image/webp',
			'image/bmp',
			'image/tiff',
			'image/avif',
		);
	}

	/**
	 * Get plugin version
	 *
	 * @return string Plugin version.
	 */
	public function get_version() {
		return ALTAUDIT82AI_VERSION;
	}

	/**
	 * Check if plugin is pro version
	 *
	 * @return bool True if pro version.
	 */
	public function is_pro() {
		return defined( 'ALTAUDIT82AI_PRO_VERSION' );
	}

	/**
	 * Prevent cloning
	 */
	private function __clone() {
	}

	/**
	 * Prevent unserializing
	 */
	public function __wakeup() {
	}
}
