<?php
/**
 * Alt Text Template Service
 *
 * Handles rule-based alt text generation using templates with dynamic variables.
 *
 * @package AltAudit
 * @since   1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Template Service class for rule-based alt text generation
 *
 * Provides functionality to generate alt text using predefined templates
 * with dynamic variables based on image context.
 *
 * @since 1.0.0
 */
class Altaudit82ai_Template_Service {





	/**
	 * Settings service instance
	 *
	 * @var Altaudit82ai_Settings
	 */
	private $settings;

	/**
	 * Predefined template definitions
	 *
	 * @var array
	 */
	private $predefined_templates = array();

	/**
	 * Available template variables
	 *
	 * @var array
	 */
	private $available_variables = array();

	/**
	 * Constructor
	 *
	 * @param Altaudit82ai_Settings $settings Settings service instance.
	 */
	public function __construct( ?Altaudit82ai_Settings $settings = null ) {
		$this->settings = $settings;
		$this->init_predefined_templates();
		$this->init_available_variables();
	}

	/**
	 * Initialize predefined templates
	 *
	 * @since  1.0.0
	 * @access private
	 * @return void
	 */
	private function init_predefined_templates() {
		$this->predefined_templates = array(
			'basic'         => array(
				'template'    => '{page_title} - {image_name}',
				'label'       => __( 'Basic', 'alt-audit' ),
				'description' => __( 'Simple combination of page title and image name', 'alt-audit' ),
			),
			'descriptive'   => array(
				'template'    => '{post_title}: {image_name} image',
				'label'       => __( 'Descriptive', 'alt-audit' ),
				'description' => __( 'Adds context with post title and labels as image', 'alt-audit' ),
			),
			'seo_friendly'  => array(
				'template'    => '{site_name} | {page_title} - {image_name}',
				'label'       => __( 'SEO Friendly', 'alt-audit' ),
				'description' => __( 'Includes site branding for search engine optimization', 'alt-audit' ),
			),
			'ecommerce'     => array(
				'template'    => '{product_title} - {image_name} photo',
				'label'       => __( 'E-commerce', 'alt-audit' ),
				'description' => __( 'Optimized for product images with photo suffix', 'alt-audit' ),
			),
			'blog'          => array(
				'template'    => '{post_title} featured image - {image_name}',
				'label'       => __( 'Blog', 'alt-audit' ),
				'description' => __( 'Designed for blog post featured images', 'alt-audit' ),
			),
			'simple'        => array(
				'template'    => '{image_name}',
				'label'       => __( 'Simple', 'alt-audit' ),
				'description' => __( 'Just the clean image filename', 'alt-audit' ),
			),
			'context_aware' => array(
				'template'    => '{page_title} {image_name}',
				'label'       => __( 'Context Aware', 'alt-audit' ),
				'description' => __( 'Minimal context with page title and image name', 'alt-audit' ),
			),
		);
	}

	/**
	 * Initialize available variables
	 *
	 * @since  1.0.0
	 * @access private
	 * @return void
	 */
	private function init_available_variables() {
		$this->available_variables = array(
			'{site_name}'         => __( 'Website name', 'alt-audit' ),
			'{site_description}'  => __( 'Website tagline', 'alt-audit' ),
			'{page_title}'        => __( 'Page title', 'alt-audit' ),
			'{post_title}'        => __( 'Post title', 'alt-audit' ),
			'{product_title}'     => __( 'Product title (WooCommerce)', 'alt-audit' ),
			'{image_name}'        => __( 'Clean image filename', 'alt-audit' ),
			'{image_caption}'     => __( 'Image caption', 'alt-audit' ),
			'{image_description}' => __( 'Image description', 'alt-audit' ),
			'{image_alt}'         => __( 'Existing alt text', 'alt-audit' ),
		);
	}

	/**
	 * Get all predefined templates
	 *
	 * @since  1.0.0
	 * @return array Predefined templates.
	 */
	public function get_predefined_templates() {
		return $this->predefined_templates;
	}

	/**
	 * Get available template variables
	 *
	 * @since  1.0.0
	 * @return array Available variables.
	 */
	public function get_available_variables() {
		return $this->available_variables;
	}

	/**
	 * Generate alt text using template
	 *
	 * @since  1.0.0
	 * @param  int    $attachment_id Attachment ID.
	 * @param  string $template      Optional template string. Kept for backward compatibility.
	 * @return string Generated alt text.
	 */
	public function generate_from_template( $attachment_id, $template = '' ) {
		// phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.FoundAfterLastUsed
		// Get the pattern array based on post type context.
		$pattern = $this->get_pattern_for_context( $attachment_id );

		if ( empty( $pattern ) ) {
			return '';
		}

		// Get variable values for this image.
		$variables = $this->get_variable_values( $attachment_id );

		// Process the array pattern.
		$alt_text = $this->process_array_pattern( $pattern, $variables );

		// Apply formatting rules.
		$alt_text = $this->apply_formatting_rules( $alt_text );

		// Validate and clean result.
		$alt_text = $this->validate_generated_text( $alt_text );

		return $alt_text;
	}

	/**
	 * Get active template from settings
	 *
	 * @since  1.0.0
	 * @access private
	 * @return string Active template.
	 */
	private function get_active_template() {
		$template_type = $this->settings->get( 'template_type', 'basic' );

		if ( 'custom' === $template_type ) {
			return $this->settings->get( 'custom_template', '{image_name}' );
		}

		if ( isset( $this->predefined_templates[ $template_type ] ) ) {
			return $this->predefined_templates[ $template_type ]['template'];
		}

		return '{image_name}';
	}

	/**
	 * Get variable values for an attachment
	 *
	 * @since  1.0.0
	 * @access private
	 * @param  int $attachment_id Attachment ID.
	 * @return array Variable values.
	 */
	/**
	 * Get variable values for an attachment
	 *
	 * @since  1.0.0
	 * @access private
	 * @param  int $attachment_id Attachment ID.
	 * @return array Variable values.
	 */
	private function get_variable_values( $attachment_id ) {
		$attachment = get_post( $attachment_id );
		$parent     = $this->get_parent_context( $attachment_id );

		$values = array(
			'{site_name}'         => get_bloginfo( 'name' ),
			'{site_description}'  => get_bloginfo( 'description' ),
			'{image_name}'        => $this->get_clean_image_name( $attachment ),
			'{image_caption}'     => $attachment->post_excerpt ?? '',
			'{image_description}' => $attachment->post_content ?? '',
			'{image_alt}'         => get_post_meta( $attachment_id, '_wp_attachment_image_alt', true ),
		);

		// Get parent post/page context.
		if ( $parent ) {
			$values['{page_title}'] = $parent->post_title ?? '';
			$values['{post_title}'] = $parent->post_title ?? '';

			// Check if it's a WooCommerce product.
			if ( 'product' === $parent->post_type ) {
				$values['{product_title}'] = $parent->post_title ?? '';
			} else {
				$values['{product_title}'] = '';
			}
		} else {
			$values['{page_title}']    = '';
			$values['{post_title}']    = '';
			$values['{product_title}'] = '';
		}

		return apply_filters( 'altaudit82ai_template_variables', $values, $attachment_id );
	}

	/**
	 * Get pattern array for the current context
	 *
	 * @since  1.0.0
	 * @access private
	 * @param  int $attachment_id Attachment ID.
	 * @return array Pattern array.
	 */
	private function get_pattern_for_context( $attachment_id ) {
		$parent = $this->get_parent_context( $attachment_id );

		// If no parent context found, use a default fallback pattern.
		if ( ! $parent ) {
			return $this->settings->get( 'home_images_alt', array( 'Image Name' ) );
		}

		// Check if we're on the homepage (and the image is attached to the page set as front page).
		if ( is_front_page() && (int) get_option( 'page_on_front' ) === $parent->ID ) {
			return $this->settings->get( 'home_images_alt', array( 'Site Name', '|', 'Page Title' ) );
		}

		// Determine pattern based on post type.
		switch ( $parent->post_type ) {
			case 'page':
				return $this->settings->get( 'pages_images_alt', array( 'Site Name', '|', 'Page Title', 'Site Description' ) );

			case 'post':
				return $this->settings->get( 'post_images_alt', array( 'Site Name', '|', 'Post Title' ) );

			case 'product':
				return $this->settings->get( 'product_images_alt', array( 'Site Name', '|', 'Product Title' ) );

			default:
				// Custom post type.
				return $this->settings->get( 'cpt_images_alt', array( 'Site Name', '|', 'Post Title' ) );
		}
	}

	/**
	 * Get parent context for an attachment
	 *
	 * Tries to find the parent post even if the attachment is not directly attached
	 * (e.g. if it is a featured image).
	 *
	 * @since  1.0.2
	 * @access private
	 * @param  int $attachment_id Attachment ID.
	 * @return WP_Post|null Parent post object or null.
	 */
	private function get_parent_context( $attachment_id ) {
		$parent_id = wp_get_post_parent_id( $attachment_id );

		// If direct parent exists, use it.
		if ( $parent_id ) {
			return get_post( $parent_id );
		}

		// If no direct parent, check if it's a featured image for any post.
		// We limit to 1 result for performance.
		// phpcs:disable WordPress.DB.SlowDBQuery.slow_db_query_meta_key, WordPress.DB.SlowDBQuery.slow_db_query_meta_value -- Required to find parent post by featured image.
		$posts = get_posts(
			array(
				'posts_per_page' => 1,
				'post_type'      => 'any',
				'post_status'    => 'any',
				'meta_key'       => '_thumbnail_id',
				'meta_value'     => $attachment_id,
				'fields'         => 'ids',
			)
		);
		// phpcs:enable WordPress.DB.SlowDBQuery.slow_db_query_meta_key, WordPress.DB.SlowDBQuery.slow_db_query_meta_value

		if ( ! empty( $posts ) ) {
			return get_post( $posts[0] );
		}

		// Check if it's in a WooCommerce product gallery.
		// phpcs:disable WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Required to find WooCommerce product gallery images.
		$gallery_posts = get_posts(
			array(
				'posts_per_page' => 1,
				'post_type'      => 'product',
				'post_status'    => 'any',
				'meta_query'     => array(
					array(
						'key'     => '_product_image_gallery',
						'value'   => (string) $attachment_id,
						'compare' => 'LIKE',
					),
				),
				'fields'         => 'ids',
			)
		);
		// phpcs:enable WordPress.DB.SlowDBQuery.slow_db_query_meta_query

		if ( ! empty( $gallery_posts ) ) {
			// Verify exact match in comma-separated list.
			foreach ( $gallery_posts as $post_id ) {
				$gallery_ids = get_post_meta( $post_id, '_product_image_gallery', true );
				$gallery_ids = explode( ',', $gallery_ids );
				if ( in_array( (string) $attachment_id, $gallery_ids, true ) ) {
					return get_post( $post_id );
				}
			}
		}

		return null;
	}

	/**
	 * Process array pattern into text
	 *
	 * @since  1.0.0
	 * @access private
	 * @param  array $pattern   Pattern array.
	 * @param  array $variables Variable values.
	 * @return string Processed text.
	 */
	private function process_array_pattern( $pattern, $variables ) {
		$result_parts = array();

		foreach ( $pattern as $element ) {
			// Check if this is a separator (literal text).
			if ( in_array( $element, array( '|', '-', '_', '&' ), true ) ) {
				$result_parts[] = $element;
				continue;
			}

			// Map UI label to variable key.
			$variable_key = $this->map_label_to_variable( $element );

			// Get the value for this variable.
			if ( isset( $variables[ $variable_key ] ) && ! empty( $variables[ $variable_key ] ) ) {
				$result_parts[] = $variables[ $variable_key ];
			}
		}

		// Join with spaces and clean up extra spaces.
		$result = implode( ' ', $result_parts );
		$result = preg_replace( '/\s+/', ' ', $result );
		$result = trim( $result );

		return $result;
	}

	/**
	 * Map UI label to template variable key
	 *
	 * @since  1.0.0
	 * @access private
	 * @param  string $label UI label.
	 * @return string Variable key.
	 */
	private function map_label_to_variable( $label ) {
		$map = array(
			'Site Name'         => '{site_name}',
			'Site Description'  => '{site_description}',
			'Page Title'        => '{page_title}',
			'Post Title'        => '{post_title}',
			'Product Title'     => '{product_title}',
			'Image Name'        => '{image_name}',
			'Image Caption'     => '{image_caption}',
			'Image Description' => '{image_description}',
			'Image Alt'         => '{image_alt}',
		);

		return isset( $map[ $label ] ) ? $map[ $label ] : '';
	}

	/**
	 * Get clean image name from attachment
	 *
	 * @since  1.0.0
	 * @access private
	 * @param  WP_Post $attachment Attachment post object.
	 * @return string Clean image name.
	 */
	private function get_clean_image_name( $attachment ) {
		$filename = $attachment->post_name ?? '';

		// Remove file extension.
		$filename = preg_replace( '/\.[^.]+$/', '', $filename );

		// Remove scaling suffixes WordPress adds (e.g., -scaled, -150x150).
		$filename = preg_replace( '/-\d+x\d+$/', '', $filename );
		$filename = preg_replace( '/-scaled$/', '', $filename );

		return $filename;
	}

	/**
	 * Process template by replacing variables
	 *
	 * @since  1.0.0
	 * @access private
	 * @param  string $template  Template string.
	 * @param  array  $variables Variable values.
	 * @return string Processed template.
	 */
	private function process_template( $template, $variables ) {
		$result = $template;

		foreach ( $variables as $variable => $value ) {
			// Skip empty values to avoid double spaces.
			if ( empty( $value ) ) {
				// Remove the variable and any surrounding separators.
				$result = str_replace( array( $variable . ' - ', ' - ' . $variable, $variable . ': ', ': ' . $variable, $variable . ' | ', ' | ' . $variable ), '', $result );
				$result = str_replace( $variable, '', $result );
			} else {
				$result = str_replace( $variable, $value, $result );
			}
		}

		return $result;
	}

	/**
	 * Apply formatting rules from settings
	 *
	 * @since  1.0.0
	 * @access private
	 * @param  string $text Text to format.
	 * @return string Formatted text.
	 */
	private function apply_formatting_rules( $text ) {
		// Remove specific characters based on settings.
		if ( $this->settings->get( 'template_remove_hyphens', true ) ) {
			$text = str_replace( '-', ' ', $text );
		}

		if ( $this->settings->get( 'template_remove_underscores', true ) ) {
			$text = str_replace( '_', ' ', $text );
		}

		if ( $this->settings->get( 'template_remove_dots', true ) ) {
			$text = str_replace( '.', '', $text );
		}

		if ( $this->settings->get( 'template_remove_commas', false ) ) {
			$text = str_replace( ',', '', $text );
		}

		if ( $this->settings->get( 'template_remove_numbers', false ) ) {
			$text = preg_replace( '/\d+/', '', $text );
		}

		// Apply case formatting.
		$case_format = $this->settings->get( 'template_case_format', 'sentence' );
		$text        = $this->apply_case_format( $text, $case_format );

		// Clean up multiple spaces.
		$text = preg_replace( '/\s+/', ' ', $text );

		// Remove leading/trailing separators.
		$text = trim( $text, " \t\n\r\0\x0B-:|" );

		return $text;
	}

	/**
	 * Apply case formatting to text
	 *
	 * @since  1.0.0
	 * @access private
	 * @param  string $text   Text to format.
	 * @param  string $format Format type (sentence, title, lowercase, uppercase).
	 * @return string Formatted text.
	 */
	private function apply_case_format( $text, $format ) {
		switch ( $format ) {
			case 'sentence':
				return ucfirst( strtolower( $text ) );

			case 'title':
				return ucwords( strtolower( $text ) );

			case 'lowercase':
				return strtolower( $text );

			case 'uppercase':
				return strtoupper( $text );

			default:
				return $text;
		}
	}

	/**
	 * Validate and clean generated text
	 *
	 * @since  1.0.0
	 * @access private
	 * @param  string $text Generated text.
	 * @return string Validated text.
	 */
	private function validate_generated_text( $text ) {
		// Remove any remaining placeholders.
		$text = preg_replace( '/\{[^}]+\}/', '', $text );

		// Trim whitespace.
		$text = trim( $text );

		// Clean up multiple spaces again.
		$text = preg_replace( '/\s+/', ' ', $text );

		// Enforce max length from settings.
		$max_length = $this->settings->get( 'max_length', 125 );
		if ( strlen( $text ) > $max_length ) {
			$text = substr( $text, 0, $max_length );
			// Try to break at last word boundary.
			$last_space = strrpos( $text, ' ' );
			if ( false !== $last_space && $last_space > ( $max_length * 0.8 ) ) {
				$text = substr( $text, 0, $last_space );
			}
		}

		return $text;
	}

	/**
	 * Generate preview using template and sample data
	 *
	 * @since  1.0.0
	 * @param  string $template Template string.
	 * @return string Preview text.
	 */
	public function generate_preview( $template ) {
		// Sample data for preview.
		$sample_variables = array(
			'{site_name}'         => get_bloginfo( 'name' ),
			'{site_description}'  => get_bloginfo( 'description' ),
			'{page_title}'        => __( 'Sample Page Title', 'alt-audit' ),
			'{post_title}'        => __( 'Sample Post Title', 'alt-audit' ),
			'{product_title}'     => __( 'Sample Product', 'alt-audit' ),
			'{image_name}'        => __( 'beautiful sunset', 'alt-audit' ),
			'{image_caption}'     => __( 'A stunning sunset view', 'alt-audit' ),
			'{image_description}' => __( 'Vibrant orange and pink sunset', 'alt-audit' ),
			'{image_alt}'         => __( 'Sunset over ocean', 'alt-audit' ),
		);

		$preview = $this->process_template( $template, $sample_variables );
		$preview = $this->apply_formatting_rules( $preview );

		return $preview;
	}

	/**
	 * Validate template string
	 *
	 * @since  1.0.0
	 * @param  string $template Template string.
	 * @return array Validation result with 'valid' boolean and 'message' string.
	 */
	public function validate_template( $template ) {
		if ( empty( $template ) ) {
			return array(
				'valid'   => false,
				'message' => __( 'Template cannot be empty.', 'alt-audit' ),
			);
		}

		// Check minimum length.
		if ( strlen( $template ) < 3 ) {
			return array(
				'valid'   => false,
				'message' => __( 'Template is too short. Minimum 3 characters.', 'alt-audit' ),
			);
		}

		// Check if template contains at least one variable.
		if ( ! preg_match( '/\{[^}]+\}/', $template ) ) {
			return array(
				'valid'   => false,
				'message' => __( 'Template must contain at least one variable placeholder.', 'alt-audit' ),
			);
		}

		// Extract variables from template.
		preg_match_all( '/\{([^}]+)\}/', $template, $matches );
		$used_variables = $matches[0];

		// Check if all variables are valid.
		$valid_variables = array_keys( $this->available_variables );
		foreach ( $used_variables as $variable ) {
			if ( ! in_array( $variable, $valid_variables, true ) ) {
				return array(
					'valid'   => false,
					/* translators: %s: Invalid variable name */
					'message' => sprintf( __( 'Invalid variable: %s', 'alt-audit' ), $variable ),
				);
			}
		}

		return array(
			'valid'   => true,
			'message' => __( 'Template is valid.', 'alt-audit' ),
		);
	}

	/**
	 * Check if image should be skipped based on template settings
	 *
	 * @since  1.0.0
	 * @param  int $attachment_id Attachment ID.
	 * @return bool True if should skip.
	 */
	public function should_skip_image( $attachment_id ) {
		$skip_existing = $this->settings->get( 'template_skip_existing', true );

		if ( $skip_existing ) {
			$existing_alt = get_post_meta( $attachment_id, '_wp_attachment_image_alt', true );
			if ( ! empty( $existing_alt ) ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Get case format options
	 *
	 * @since  1.0.0
	 * @return array Case format options.
	 */
	public function get_case_format_options() {
		return array(
			'sentence'  => __( 'Sentence case', 'alt-audit' ),
			'title'     => __( 'Title Case', 'alt-audit' ),
			'lowercase' => __( 'lowercase', 'alt-audit' ),
			'uppercase' => __( 'UPPERCASE', 'alt-audit' ),
		);
	}
}
