<?php
/**
 * Taxonomy Service
 *
 * Registers and manages the altaudit82ai_status taxonomy for attachments.
 *
 * @package AltAudit
 * @since 1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Taxonomy service for managing attachment status taxonomy
 *
 * Uses custom taxonomy instead of post meta for status filtering
 * to avoid slow meta_query database queries.
 *
 * @since 1.0.0
 */
class Altaudit82ai_Taxonomy {

	/**
	 * Taxonomy name
	 *
	 * @var string
	 */
	const TAXONOMY = 'altaudit82ai_status';

	/**
	 * Valid status terms
	 *
	 * @var array
	 */
	const TERMS = array(
		'missing',
		'weak',
		'good',
		'excellent',
		'decorative',
	);

	/**
	 * Register the taxonomy
	 *
	 * @return void
	 */
	public function register() {
		$labels = array(
			'name'          => _x( 'Alt Text Status', 'taxonomy general name', 'alt-audit' ),
			'singular_name' => _x( 'Alt Text Status', 'taxonomy singular name', 'alt-audit' ),
			'menu_name'     => __( 'Alt Text Status', 'alt-audit' ),
		);

		$args = array(
			'labels'            => $labels,
			'hierarchical'      => false,
			'public'            => false,
			'show_ui'           => false,
			'show_admin_column' => false,
			'show_in_nav_menus' => false,
			'show_tagcloud'     => false,
			'show_in_rest'      => false,
			'rewrite'           => false,
			'query_var'         => false,
		);

		register_taxonomy( self::TAXONOMY, 'attachment', $args );

		$this->ensure_terms_exist();
	}

	/**
	 * Ensure all status terms exist
	 *
	 * @return void
	 */
	private function ensure_terms_exist() {
		foreach ( self::TERMS as $term ) {
			if ( ! term_exists( $term, self::TAXONOMY ) ) {
				wp_insert_term( $term, self::TAXONOMY );
			}
		}
	}

	/**
	 * Set status term for an attachment
	 *
	 * @param int    $attachment_id Attachment ID.
	 * @param string $status        Status term (missing, weak, good, excellent, decorative).
	 * @return bool|WP_Error True on success, WP_Error on failure.
	 */
	public static function set_status( $attachment_id, $status ) {
		// Sanitize inputs.
		$attachment_id = absint( $attachment_id );
		$status        = sanitize_key( $status );

		if ( ! $attachment_id ) {
			return new WP_Error( 'invalid_attachment', __( 'Invalid attachment ID.', 'alt-audit' ) );
		}

		if ( ! in_array( $status, self::TERMS, true ) ) {
			return new WP_Error( 'invalid_status', __( 'Invalid status term.', 'alt-audit' ) );
		}

		$result = wp_set_object_terms( $attachment_id, $status, self::TAXONOMY );

		if ( is_wp_error( $result ) ) {
			return $result;
		}

		return true;
	}

	/**
	 * Get status term for an attachment
	 *
	 * @param int $attachment_id Attachment ID.
	 * @return string|null Status term or null if not set.
	 */
	public static function get_status( $attachment_id ) {
		$attachment_id = absint( $attachment_id );

		if ( ! $attachment_id ) {
			return null;
		}

		$terms = wp_get_object_terms( $attachment_id, self::TAXONOMY, array( 'fields' => 'slugs' ) );

		if ( is_wp_error( $terms ) || empty( $terms ) ) {
			return null;
		}

		// Sanitize the returned term slug.
		return sanitize_key( $terms[0] );
	}

	/**
	 * Remove status term from an attachment
	 *
	 * @param int $attachment_id Attachment ID.
	 * @return bool True on success, false on invalid input.
	 */
	public static function remove_status( $attachment_id ) {
		$attachment_id = absint( $attachment_id );

		if ( ! $attachment_id ) {
			return false;
		}

		wp_set_object_terms( $attachment_id, array(), self::TAXONOMY );
		return true;
	}

	/**
	 * Build tax_query for status filtering
	 *
	 * @param string $status Status to filter by.
	 * @return array Tax query array.
	 */
	public static function build_status_tax_query( $status ) {
		// Sanitize status input.
		$status = sanitize_key( $status );

		if ( 'all' === $status || empty( $status ) ) {
			return array();
		}

		if ( 'has_alt' === $status ) {
			return array(
				array(
					'taxonomy' => self::TAXONOMY,
					'field'    => 'slug',
					'terms'    => array( 'weak', 'good', 'excellent' ),
					'operator' => 'IN',
				),
			);
		}

		if ( 'missing' === $status ) {
			return array(
				'relation' => 'OR',
				array(
					'taxonomy' => self::TAXONOMY,
					'field'    => 'slug',
					'terms'    => array( 'missing' ),
					'operator' => 'IN',
				),
				array(
					'taxonomy' => self::TAXONOMY,
					'operator' => 'NOT EXISTS',
				),
			);
		}

		// Only allow valid status terms to prevent injection.
		if ( ! in_array( $status, self::TERMS, true ) ) {
			return array();
		}

		return array(
			array(
				'taxonomy' => self::TAXONOMY,
				'field'    => 'slug',
				'terms'    => array( $status ),
				'operator' => 'IN',
			),
		);
	}

	/**
	 * Determine status from quality score
	 *
	 * @param int  $score       Quality score (0-100).
	 * @param bool $decorative  Whether image is decorative.
	 * @param bool $has_alt     Whether image has alt text.
	 * @return string Status term.
	 */
	public static function get_status_from_score( $score, $decorative = false, $has_alt = true ) {
		if ( $decorative ) {
			return 'decorative';
		}

		if ( ! $has_alt ) {
			return 'missing';
		}

		if ( $score >= 80 ) {
			return 'excellent';
		}

		if ( $score >= 50 ) {
			return 'good';
		}

		return 'weak';
	}
}
