<?php
/**
 * Statistics Service
 *
 * Centralized service for calculating image statistics across the plugin.
 * Ensures consistent statistics between dashboard and audit pages.
 *
 * @package AltAudit
 * @since 1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Statistics service for centralized statistics calculation
 *
 * Provides consistent statistics across all plugin pages with caching support.
 *
 * @since 1.0.0
 */
class Altaudit82ai_Statistics_Service {

	/**
	 * Cache group name
	 *
	 * @var string
	 */
	const CACHE_GROUP = 'altaudit82ai_statistics';

	/**
	 * Cache key for statistics
	 *
	 * @var string
	 */
	const CACHE_KEY = 'image_statistics';

	/**
	 * Cache expiration time (1 hour)
	 *
	 * @var int
	 */
	const CACHE_EXPIRATION = 3600;

	/**
	 * Get comprehensive image statistics
	 *
	 * Returns statistics with both legacy keys (for dashboard) and new keys (for audit page).
	 *
	 * @since  1.0.0
	 * @param  bool $force_refresh Whether to bypass cache and recalculate.
	 * @return array Comprehensive statistics array.
	 */
	public function get_statistics( $force_refresh = false ) {

		// Try to get from cache unless force refresh is requested.
		if ( ! $force_refresh ) {
			$cached_stats = wp_cache_get( self::CACHE_KEY, self::CACHE_GROUP );
			if ( false !== $cached_stats ) {
				return $cached_stats;
			}
		}
		// Calculate fresh statistics.
		$stats = $this->calculate_statistics();
		// Cache the results.
		wp_cache_set( self::CACHE_KEY, $stats, self::CACHE_GROUP, self::CACHE_EXPIRATION );

		return $stats;
	}

	/**
	 * Calculate image statistics from database
	 *
	 * @since  1.0.0
	 * @access private
	 * @return array Statistics array.
	 */
	private function calculate_statistics() {
		// Initialize statistics array with all required keys.
		$stats = array(
			// Dashboard keys (legacy).
			'total_images'   => 0,
			'missing_alt'    => 0,
			'decorative_alt' => 0,
			'weak_alt'       => 0,
			'good_alt'       => 0,
			'excellent_alt'  => 0,
			'auto_generated' => 0,
			'total_score'    => 0,
			'avg_score'      => 0,

			// Audit page keys (alternative naming).
			'total'          => 0,
			'missing'        => 0,
			'decorative'     => 0,
			'weak'           => 0,
			'good'           => 0,
			'excellent'      => 0,
		);

				// 1. Get total image count using optimized WP_Query.
				// We only need the count, so we limit to 1 post and use fields=ids to minimize data transfer.
				$images_query = new WP_Query(
					array(
						'post_type'      => 'attachment',
						'post_status'    => 'inherit',
						'post_mime_type' => Altaudit82ai::get_supported_mime_types(),
						'posts_per_page' => -1,
						'fields'         => 'ids',
						'no_found_rows'  => false, // Ensure found_posts is calculated.
					)
				);
		// Set total image counts.
		$stats['total_images'] = $images_query->found_posts;
		$stats['total']        = $images_query->found_posts;

		// Process each image to gather statistics.
		foreach ( $images_query->posts as $image_id ) {
			$this->process_image_stats( $image_id, $stats );
		}

		// Calculate average score.
		if ( $stats['total_images'] > 0 ) {
			$stats['avg_score'] = round( $stats['total_score'] / $stats['total_images'] );
		}

		return $stats;
	}

	/**
	 * Process individual image and update statistics
	 *
	 * @since  1.0.0
	 * @access private
	 * @param  int   $image_id Image attachment ID.
	 * @param  array &$stats   Statistics array (passed by reference).
	 * @return void
	 */
	private function process_image_stats( $image_id, &$stats ) {
		// Get image metadata.
		$alt_text       = get_post_meta( $image_id, '_wp_attachment_image_alt', true );
		$quality_status = Altaudit82ai_Taxonomy::get_status( $image_id );
		$quality_score  = absint( get_post_meta( $image_id, '_altaudit82ai_quality_score', true ) );
		$is_decorative  = '1' === get_post_meta( $image_id, '_altaudit82ai_decorative', true );

		// Handle decorative images.
		if ( $is_decorative ) {
			++$stats['decorative_alt'];
			++$stats['decorative'];
			$stats['total_score'] += 100; // Decorative with empty alt is perfect.
			return;
		}

		// Check if we have stored quality status.
		if ( ! empty( $quality_status ) ) {
			// Use stored status for consistency.
			$stats['total_score'] += $quality_score;

			// Update status counts.
			if ( 'weak' === $quality_status ) {
				++$stats['weak_alt'];
				++$stats['weak'];
			} elseif ( 'good' === $quality_status ) {
				++$stats['good_alt'];
				++$stats['good'];
			} elseif ( 'excellent' === $quality_status ) {
				++$stats['excellent_alt'];
				++$stats['excellent'];
			} elseif ( 'missing' === $quality_status || empty( $alt_text ) ) {
				++$stats['missing_alt'];
				++$stats['missing'];
			}

			// Check if auto-generated.
			$is_auto_generated = get_post_meta( $image_id, '_altaudit82ai_auto_generated', true );
			if ( $is_auto_generated ) {
				++$stats['auto_generated'];
			}
		} elseif ( empty( $alt_text ) ) {
			// Fallback for images without stored quality data.
			++$stats['missing_alt'];
			++$stats['missing'];
			// Missing gets 0 score.
		} else {
			// Simple fallback scoring based on length.
			$this->apply_fallback_scoring( $alt_text, $stats );
		}
	}

	/**
	 * Apply fallback scoring for images without stored quality data
	 *
	 * @since  1.0.0
	 * @access private
	 * @param  string $alt_text Alt text to analyze.
	 * @param  array  &$stats   Statistics array (passed by reference).
	 * @return void
	 */
	private function apply_fallback_scoring( $alt_text, &$stats ) {
		$length = strlen( $alt_text );

		if ( $length < 20 ) {
			++$stats['weak_alt'];
			++$stats['weak'];
			$stats['total_score'] += 30;
		} elseif ( $length < 100 ) {
			++$stats['good_alt'];
			++$stats['good'];
			$stats['total_score'] += 70;
		} else {
			++$stats['excellent_alt'];
			++$stats['excellent'];
			$stats['total_score'] += 90;
		}
	}

	/**
	 * Invalidate statistics cache
	 *
	 * Call this method whenever image data changes to ensure fresh statistics.
	 *
	 * @since  1.0.0
	 * @return bool True on success, false on failure.
	 */
	public function invalidate_cache() {
		return wp_cache_delete( self::CACHE_KEY, self::CACHE_GROUP );
	}

	/**
	 * Get statistics for dashboard display
	 *
	 * Returns statistics with dashboard-compatible keys.
	 *
	 * @since  1.0.0
	 * @param  bool $force_refresh Whether to bypass cache.
	 * @return array Dashboard statistics.
	 */
	public function get_dashboard_stats( $force_refresh = false ) {
		return $this->get_statistics( $force_refresh );
	}

	/**
	 * Get statistics for audit page display
	 *
	 * Returns statistics with audit page-compatible keys.
	 *
	 * @since  1.0.0
	 * @param  bool $force_refresh Whether to bypass cache.
	 * @return array Audit page statistics.
	 */
	public function get_audit_stats( $force_refresh = false ) {
		return $this->get_statistics( $force_refresh );
	}
}
