<?php
/**
 * Settings Service
 *
 * @package AltAudit
 * @since 1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Settings service for managing plugin configuration
 *
 * Provides a centralized way to manage all plugin settings
 * with validation, sanitization, and caching.
 *
 * @since 1.0.0
 */
class Altaudit82ai_Settings {

	/**
	 * Settings cache
	 *
	 * @var array
	 */
	private $cache = array();

	/**
	 * Default settings
	 *
	 * @var array
	 */
	private $defaults = array(
		'api_key'                     => '',
		'api_url'                     => ALTAUDIT82AI_API_BASE_URL,
		'auto_generate'               => false,
		'quality_threshold'           => 70,
		'max_length'                  => 125,
		'style'                       => 'descriptive',
		'context_aware'               => true,
		'bulk_processing'             => true,
		'cache_results'               => true,
		'debug_mode'                  => false,
		'enabled_post_types'          => array( 'post', 'page' ),
		'skip_existing'               => true,
		'notification_level'          => 'errors',
		'rate_limit'                  => 10,
		'batch_size'                  => 5,
		// Custom builder settings.
		'home_images_alt'             => array( 'Site Name', '|', 'Page Title' ),
		'pages_images_alt'            => array( 'Site Name', '|', 'Page Title', 'Site Description' ),
		'post_images_alt'             => array( 'Site Name', '|', 'Post Title' ),
		'product_images_alt'          => array( 'Site Name', '|', 'Product Title' ),
		'cpt_images_alt'              => array( 'Site Name', '|', 'Post Title' ),
		// Template application settings.
		'template_type'               => 'basic',
		'custom_template'             => '',
		'template_skip_existing'      => true,
		'template_override_weak'      => false,
		'template_remove_hyphens'     => false,
		'template_remove_underscores' => false,
		'template_remove_dots'        => false,
		'template_remove_commas'      => false,
		'template_remove_numbers'     => false,
		'template_case_format'        => 'sentence',
	);

	/**
	 * Settings validation rules
	 *
	 * @var array
	 */
	private $validation_rules = array(
		'api_key'                     => 'string',
		'api_url'                     => 'url',
		'auto_generate'               => 'boolean',
		'quality_threshold'           => 'integer',
		'max_length'                  => 'integer',
		'style'                       => 'string',
		'context_aware'               => 'boolean',
		'bulk_processing'             => 'boolean',
		'cache_results'               => 'boolean',
		'debug_mode'                  => 'boolean',
		'enabled_post_types'          => 'array',
		'skip_existing'               => 'boolean',
		'notification_level'          => 'string',
		'rate_limit'                  => 'integer',
		'batch_size'                  => 'integer',
		// Custom builder validation.
		'home_images_alt'             => 'array',
		'pages_images_alt'            => 'array',
		'post_images_alt'             => 'array',
		'product_images_alt'          => 'array',
		'cpt_images_alt'              => 'array',
		// Template application validation.
		'template_type'               => 'string',
		'custom_template'             => 'string',
		'template_skip_existing'      => 'boolean',
		'template_override_weak'      => 'boolean',
		'template_remove_hyphens'     => 'boolean',
		'template_remove_underscores' => 'boolean',
		'template_remove_dots'        => 'boolean',
		'template_remove_commas'      => 'boolean',
		'template_remove_numbers'     => 'boolean',
		'template_case_format'        => 'string',
	);

	/**
	 * Constructor
	 */
	public function __construct() {
		$this->load_settings();
	}

	/**
	 * Get setting value
	 *
	 * @param string $key     Setting key.
	 * @param mixed  $default Default value.
	 * @return mixed Setting value.
	 */
	public function get( $key, $default = null ) {
		if ( isset( $this->cache[ $key ] ) ) {
			return $this->cache[ $key ];
		}

		$option_name = $this->get_option_name( $key );
		$value       = get_option( $option_name, $default );

		// Use default if value is null.
		if ( null === $value && isset( $this->defaults[ $key ] ) ) {
			$value = $this->defaults[ $key ];
		}

		// Cache the value.
		$this->cache[ $key ] = $value;

		return $value;
	}

	/**
	 * Set setting value
	 *
	 * @param string $key   Setting key.
	 * @param mixed  $value Setting value.
	 * @return bool True on success.
	 */
	public function set( $key, $value ) {
		// Validate value.
		$value = $this->validate_setting( $key, $value );

		// Note: boolean false is a valid value, not a validation failure.
		$validation_rule = isset( $this->validation_rules[ $key ] ) ? $this->validation_rules[ $key ] : null;
		if ( false === $value && 'boolean' !== $validation_rule ) {
			return false;
		}

		$option_name = $this->get_option_name( $key );
		$result      = update_option( $option_name, $value );

		// Update cache.
		$this->cache[ $key ] = $value;

		// update_option returns false if value hasn't changed, which is OK.
		// We consider this a successful operation.
		return true;
	}

	/**
	 * Delete setting
	 *
	 * @param string $key Setting key.
	 * @return bool True on success.
	 */
	public function delete( $key ) {
		$option_name = $this->get_option_name( $key );
		$result      = delete_option( $option_name );

		if ( $result ) {
			unset( $this->cache[ $key ] );
		}

		return $result;
	}

	/**
	 * Get all settings
	 *
	 * @return array All settings.
	 */
	public function get_all() {
		$settings = array();

		foreach ( array_keys( $this->defaults ) as $key ) {
			$settings[ $key ] = $this->get( $key );
		}

		return $settings;
	}

	/**
	 * Set multiple settings
	 *
	 * @param array $settings Settings array.
	 * @return bool True on success.
	 */
	public function set_multiple( $settings ) {
		$success       = true;
		$actual_errors = false;

		foreach ( $settings as $key => $value ) {
			// Get current value to check if it's actually changing.
			$current_value = $this->get( $key );

			// Validate the new value.
			$validated_value = $this->validate_setting( $key, $value );

			// If validation failed, mark as error.
			// Note: boolean false is a valid value, not a validation failure.
			$validation_rule = isset( $this->validation_rules[ $key ] ) ? $this->validation_rules[ $key ] : null;
			if ( false === $validated_value && 'boolean' !== $validation_rule && '' !== $value ) {
				$actual_errors = true;
				$success       = false;
				continue;
			}

			// Only attempt to save if value is different.
			// WordPress update_option returns false when value hasn't changed.
			if ( $current_value !== $validated_value ) {
				if ( ! $this->set( $key, $validated_value ) ) {
					$actual_errors = true;
					$success       = false;
				}
			} else {
				// Value is the same - update cache to ensure consistency.
				$this->cache[ $key ] = $validated_value;
			}
		}

		// If no actual errors occurred, return true even if no values changed.
		return ! $actual_errors;
	}

	/**
	 * Reset settings to defaults
	 *
	 * @param array $keys Specific keys to reset, or empty for all.
	 * @return bool True on success.
	 */
	public function reset( $keys = array() ) {
		$keys_to_reset = empty( $keys ) ? array_keys( $this->defaults ) : $keys;
		$success       = true;

		foreach ( $keys_to_reset as $key ) {
			if ( isset( $this->defaults[ $key ] ) ) {
				if ( ! $this->set( $key, $this->defaults[ $key ] ) ) {
					$success = false;
				}
			}
		}

		return $success;
	}

	/**
	 * Get option name for WordPress options table
	 *
	 * @param string $key Setting key.
	 * @return string Option name.
	 */
	private function get_option_name( $key ) {
		return 'altaudit82ai_' . $key;
	}

	/**
	 * Load settings into cache
	 *
	 * @return void
	 */
	private function load_settings() {
		foreach ( array_keys( $this->defaults ) as $key ) {
			$this->get( $key );
		}
	}

	/**
	 * Validate setting value
	 *
	 * @param string $key   Setting key.
	 * @param mixed  $value Setting value.
	 * @return mixed Validated value or false on failure.
	 */
	private function validate_setting( $key, $value ) {
		if ( ! isset( $this->validation_rules[ $key ] ) ) {
			return $value;
		}

		$rule = $this->validation_rules[ $key ];

		switch ( $rule ) {
			case 'string':
				return sanitize_text_field( $value );

			case 'url':
				return esc_url_raw( $value );

			case 'boolean':
				return (bool) $value;

			case 'integer':
				return intval( $value );

			case 'array':
				return is_array( $value ) ? array_map( 'sanitize_text_field', $value ) : array();

			default:
				return $value;
		}
	}

	/**
	 * Get validation rules
	 *
	 * @return array Validation rules.
	 */
	public function get_validation_rules() {
		return $this->validation_rules;
	}

	/**
	 * Get default settings
	 *
	 * @return array Default settings.
	 */
	public function get_defaults() {
		return $this->defaults;
	}

	/**
	 * Export settings
	 *
	 * @return string JSON encoded settings.
	 */
	public function export() {
		return wp_json_encode( $this->get_all() );
	}

	/**
	 * Import settings
	 *
	 * @param string $json JSON encoded settings.
	 * @return bool True on success.
	 */
	public function import( $json ) {
		$settings = json_decode( $json, true );

		if ( ! is_array( $settings ) ) {
			return false;
		}

		return $this->set_multiple( $settings );
	}

	/**
	 * Check if API is configured
	 *
	 * @return bool True if API key is set.
	 */
	public function is_api_configured() {
		$api_key = $this->get( 'api_key' );
		return ! empty( $api_key );
	}

	/**
	 * Get style options
	 *
	 * @return array Style options.
	 */
	public function get_style_options() {
		return array(
			'descriptive' => __( 'Descriptive', 'alt-audit' ),
			'technical'   => __( 'Technical', 'alt-audit' ),
			'creative'    => __( 'Creative', 'alt-audit' ),
			'brief'       => __( 'Brief', 'alt-audit' ),
		);
	}

	/**
	 * Get notification level options
	 *
	 * @return array Notification level options.
	 */
	public function get_notification_level_options() {
		return array(
			'none'     => __( 'None', 'alt-audit' ),
			'errors'   => __( 'Errors Only', 'alt-audit' ),
			'warnings' => __( 'Warnings and Errors', 'alt-audit' ),
			'all'      => __( 'All Notifications', 'alt-audit' ),
		);
	}

	/**
	 * Get enabled post types
	 *
	 * @return array Enabled post types.
	 */
	public function get_enabled_post_types() {
		$enabled    = $this->get( 'enabled_post_types', array() );
		$post_types = get_post_types( array( 'public' => true ), 'objects' );

		$options = array();
		foreach ( $post_types as $post_type ) {
			$options[ $post_type->name ] = $post_type->label;
		}

		return array_intersect_key( $options, array_flip( $enabled ) );
	}

	/**
	 * Clear settings cache
	 *
	 * @return void
	 */
	public function clear_cache() {
		$this->cache = array();
		$this->load_settings();
	}
}
