<?php
/**
 * Rule-Based Alt Text Generator
 *
 * @package AltAudit
 * @since 1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Rule-based alt text generator using WordPress content
 *
 * Generates alt text using dynamic values like image names, post titles,
 * site names, and other contextual information from WordPress.
 *
 * @since 1.0.0
 */
class Altaudit82ai_Rule_Generator {


	/**
	 * Settings service
	 *
	 * @var Altaudit82ai_Settings
	 */
	private $settings;

	/**
	 * Constructor
	 *
	 * @param Altaudit82ai_Settings $settings Settings service.
	 */
	public function __construct( ?Altaudit82ai_Settings $settings = null ) {
		$this->settings = $settings;
	}

	/**
	 * Generate alt text for image using rule-based approach
	 *
	 * @param int    $attachment_id Attachment ID.
	 * @param string $template      Template to use for generation.
	 * @return string Generated alt text.
	 */
	public function generate_for_image( $attachment_id, $template = null ) {
		if ( ! wp_attachment_is_image( $attachment_id ) ) {
			return '';
		}

		$template = $template ? $template : $this->get_default_template();
		$values   = $this->get_dynamic_values( $attachment_id );

		return $this->parse_template( $template, $values );
	}

	/**
	 * Get dynamic values for template parsing
	 *
	 * @param int $attachment_id Attachment ID.
	 * @return array Dynamic values.
	 */
	private function get_dynamic_values( $attachment_id ) {
		$attachment = get_post( $attachment_id );
		if ( ! $attachment ) {
			return array();
		}

		$parent_post = null;
		if ( $attachment->post_parent ) {
			$parent_post = get_post( $attachment->post_parent );
		}

		// Clean up image name.
		$image_name = $this->clean_image_name( $attachment->post_title );

		// Get image dimensions for context.
		$metadata   = wp_get_attachment_metadata( $attachment_id );
		$dimensions = '';
		if ( $metadata && isset( $metadata['width'] ) && isset( $metadata['height'] ) ) {
			$dimensions = sprintf( '%dx%d', $metadata['width'], $metadata['height'] );
		}

		$values = array(
			'image_name'    => $image_name,
			'image_title'   => $attachment->post_title,
			'image_desc'    => $attachment->post_content,
			'image_caption' => $attachment->post_excerpt,
			'site_name'     => get_bloginfo( 'name' ),
			'site_desc'     => get_bloginfo( 'description' ),
			'dimensions'    => $dimensions,
			'file_name'     => basename( get_attached_file( $attachment_id ) ),
		);

		if ( $parent_post ) {
			$values['post_title'] = $parent_post->post_title;
			$values['post_type']  = get_post_type_object( $parent_post->post_type )->labels->singular_name;
			$values['page_title'] = $parent_post->post_title;
		} else {
			$values['post_title'] = '';
			$values['post_type']  = '';
			$values['page_title'] = get_bloginfo( 'name' );
		}

		return $values;
	}

	/**
	 * Parse template with dynamic values
	 *
	 * @param string $template Template string.
	 * @param array  $values   Dynamic values.
	 * @return string Parsed template.
	 */
	private function parse_template( $template, $values ) {
		$parsed = $template;

		foreach ( $values as $key => $value ) {
			$placeholder = '{' . $key . '}';
			$parsed      = str_replace( $placeholder, $value, $parsed );
		}

		// Clean up the result.
		$parsed = $this->clean_generated_text( $parsed );

		// Ensure reasonable length.
		$max_length = $this->settings ? $this->settings->get( 'max_length', 125 ) : 125;
		if ( strlen( $parsed ) > $max_length ) {
			$parsed = substr( $parsed, 0, $max_length - 3 ) . '...';
		}

		return $parsed;
	}

	/**
	 * Clean image name for better alt text
	 * Uses proven approach from Auto Image Attributes plugin with enhancements
	 *
	 * @param string $image_name Raw image name.
	 * @return string Cleaned image name.
	 */
	private function clean_image_name( $image_name ) {
		// Remove file extension.
		$cleaned = preg_replace( '/\.[^.]+$/', '', $image_name );

		// Get cleaning preferences (with sensible defaults).
		$remove_hyphens     = $this->settings ? $this->settings->get( 'rule_remove_hyphens', true ) : true;
		$remove_underscores = $this->settings ? $this->settings->get( 'rule_remove_underscores', true ) : true;
		$remove_dots        = $this->settings ? $this->settings->get( 'rule_remove_dots', true ) : true;
		$remove_commas      = $this->settings ? $this->settings->get( 'rule_remove_commas', true ) : true;
		$remove_numbers     = $this->settings ? $this->settings->get( 'rule_remove_numbers', false ) : false;

		// Build array of characters to replace with spaces.
		$filter_chars = array();

		if ( $remove_hyphens ) {
			$filter_chars[] = '-';
		}

		if ( $remove_underscores ) {
			$filter_chars[] = '_';
		}

		if ( $remove_dots ) {
			$filter_chars[] = '.';
		}

		if ( $remove_commas ) {
			$filter_chars[] = ',';
		}

		// Replace filtered characters with spaces.
		if ( ! empty( $filter_chars ) ) {
			$cleaned = str_replace( $filter_chars, ' ', $cleaned );
		}

		// Remove numbers only if enabled by user.
		if ( $remove_numbers ) {
			$cleaned = preg_replace( '/[0-9]+/', '', $cleaned );
		}
		// If remove_numbers is false, preserve all numbers including those at start/end.

		// Advanced character cleaning.
		$cleaned = $this->clean_special_characters( $cleaned );

		// Clean up multiple spaces.
		$cleaned = preg_replace( '/\s+/', ' ', $cleaned );

		// Trim and apply case formatting.
		$cleaned = trim( $cleaned );
		$cleaned = $this->apply_case_formatting( $cleaned );

		return $cleaned;
	}

	/**
	 * Clean special characters from filename
	 *
	 * @param string $text Text to clean.
	 * @return string Cleaned text.
	 */
	private function clean_special_characters( $text ) {
		// Remove common problematic characters.
		$special_chars = array(
			'(' => ' ',
			')' => ' ',
			'[' => ' ',
			']' => ' ',
			'{' => ' ',
			'}' => ' ',
			'#' => ' ',
			'&' => ' and ',
			'+' => ' ',
			'~' => ' ',
			"'" => '',
			'"' => '',
			'%' => ' percent ',
			'@' => ' at ',
		);

		$text = str_replace( array_keys( $special_chars ), array_values( $special_chars ), $text );

		return $text;
	}

	/**
	 * Apply case formatting to cleaned text
	 *
	 * @param string $text Text to format.
	 * @return string Formatted text.
	 */
	private function apply_case_formatting( $text ) {
		// Get case formatting preference.
		$case_format = $this->settings ? $this->settings->get( 'rule_case_format', 'sentence' ) : 'sentence';

		switch ( $case_format ) {
			case 'lowercase':
				return strtolower( $text );

			case 'uppercase':
				return strtoupper( $text );

			case 'title':
				return ucwords( strtolower( $text ) );

			case 'sentence':
			default:
				return ucfirst( strtolower( $text ) );
		}
	}

	/**
	 * Clean generated alt text
	 *
	 * @param string $text Generated text.
	 * @return string Cleaned text.
	 */
	private function clean_generated_text( $text ) {
		// Remove empty placeholders.
		$text = preg_replace( '/\{\w+\}/', '', $text );

		// Clean up multiple spaces.
		$text = preg_replace( '/\s+/', ' ', $text );

		// Remove leading/trailing prepositions and conjunctions.
		$text = preg_replace( '/^(in|on|at|of|for|with|by|from|to|and|or|but)\s+/i', '', $text );
		$text = preg_replace( '/\s+(in|on|at|of|for|with|by|from|to|and|or|but)$/i', '', $text );

		// Trim.
		$text = trim( $text );

		return $text;
	}

	/**
	 * Get default template for generation
	 *
	 * @return string Default template.
	 */
	private function get_default_template() {
		// Try to get template from settings.
		if ( $this->settings ) {
			$template = $this->settings->get( 'rule_template', '' );
			if ( ! empty( $template ) ) {
				return $template;
			}
		}

		// Default template.
		return '{image_name}';
	}

	/**
	 * Get available dynamic values
	 *
	 * @return array Available placeholders and descriptions.
	 */
	public function get_available_placeholders() {
		return array(
			'image_name'    => __( 'Cleaned image filename', 'alt-audit' ),
			'image_title'   => __( 'Image title from media library', 'alt-audit' ),
			'image_desc'    => __( 'Image description from media library', 'alt-audit' ),
			'image_caption' => __( 'Image caption from media library', 'alt-audit' ),
			'post_title'    => __( 'Title of the post/page containing the image', 'alt-audit' ),
			'post_type'     => __( 'Type of post (page, post, etc.)', 'alt-audit' ),
			'page_title'    => __( 'Page title (same as post_title)', 'alt-audit' ),
			'site_name'     => __( 'Website name', 'alt-audit' ),
			'site_desc'     => __( 'Website tagline/description', 'alt-audit' ),
			'dimensions'    => __( 'Image dimensions (e.g., 800x600)', 'alt-audit' ),
			'file_name'     => __( 'Original filename with extension', 'alt-audit' ),
		);
	}

	/**
	 * Get template suggestions based on context
	 *
	 * @return array Template suggestions.
	 */
	public function get_template_suggestions() {
		return array(
			'basic'    => array(
				'label'    => __( 'Basic (Image Name)', 'alt-audit' ),
				'template' => '{image_name}',
			),
			'context'  => array(
				'label'    => __( 'With Context', 'alt-audit' ),
				'template' => '{image_name} in {post_title}',
			),
			'detailed' => array(
				'label'    => __( 'Detailed', 'alt-audit' ),
				'template' => '{image_name} - {post_title} | {site_name}',
			),
			'caption'  => array(
				'label'    => __( 'Caption First', 'alt-audit' ),
				'template' => '{image_caption} {image_name}',
			),
		);
	}

	/**
	 * Bulk generate alt text for multiple images
	 *
	 * @param array  $attachment_ids Array of attachment IDs.
	 * @param string $template       Template to use.
	 * @return array Results with success/error counts and details.
	 */
	public function bulk_generate( $attachment_ids, $template = null ) {
		$results = array(
			'success' => 0,
			'errors'  => 0,
			'details' => array(),
		);

		foreach ( $attachment_ids as $attachment_id ) {
			$attachment_id = intval( $attachment_id );

			if ( ! wp_attachment_is_image( $attachment_id ) ) {
				++$results['errors'];
				$results['details'][ $attachment_id ] = array(
					'status' => 'error',
					'error'  => __( 'Not an image attachment', 'alt-audit' ),
				);
				continue;
			}

			$generated_alt = $this->generate_for_image( $attachment_id, $template );

			if ( ! empty( $generated_alt ) ) {
				update_post_meta( $attachment_id, '_wp_attachment_image_alt', $generated_alt );
				++$results['success'];
				$results['details'][ $attachment_id ] = array(
					'status'   => 'success',
					'alt_text' => $generated_alt,
					'method'   => 'rule_based',
				);
			} else {
				++$results['errors'];
				$results['details'][ $attachment_id ] = array(
					'status' => 'error',
					'error'  => __( 'Failed to generate alt text', 'alt-audit' ),
				);
			}
		}

		return $results;
	}

	/**
	 * Validate template syntax
	 *
	 * @param string $template Template to validate.
	 * @return array Validation result with success status and message.
	 */
	public function validate_template( $template ) {
		if ( empty( $template ) ) {
			return array(
				'valid'   => false,
				'message' => __( 'Template cannot be empty', 'alt-audit' ),
			);
		}

		// Check for valid placeholders.
		$placeholders      = $this->get_available_placeholders();
		$used_placeholders = array();

		preg_match_all( '/\{(\w+)\}/', $template, $matches );

		if ( ! empty( $matches[1] ) ) {
			foreach ( $matches[1] as $placeholder ) {
				if ( ! array_key_exists( $placeholder, $placeholders ) ) {
					return array(
						'valid'   => false,
						'message' => sprintf(
							/* translators: %s: placeholder name */
							__( 'Invalid placeholder: {%s}', 'alt-audit' ),
							$placeholder
						),
					);
				}
				$used_placeholders[] = $placeholder;
			}
		}

		// Test template with sample data.
		$test_values = array(
			'image_name'    => 'Sample Image',
			'post_title'    => 'Sample Post',
			'site_name'     => 'Sample Site',
			'image_caption' => 'Sample caption',
		);

		$result = $this->parse_template( $template, $test_values );

		if ( empty( trim( $result ) ) ) {
			return array(
				'valid'   => false,
				'message' => __( 'Template produces empty result', 'alt-audit' ),
			);
		}

		return array(
			'valid'        => true,
			'message'      => __( 'Template is valid', 'alt-audit' ),
			'preview'      => $result,
			'placeholders' => $used_placeholders,
		);
	}

	/**
	 * Get filename cleaning settings with defaults
	 *
	 * @return array Cleaning settings.
	 */
	public function get_cleaning_settings() {
		return array(
			'rule_remove_hyphens'     => array(
				'label'   => __( 'Remove hyphens (-)', 'alt-audit' ),
				'default' => true,
				'example' => 'my-image.jpg → my image',
			),
			'rule_remove_underscores' => array(
				'label'   => __( 'Remove underscores (_)', 'alt-audit' ),
				'default' => true,
				'example' => 'my_image.jpg → my image',
			),
			'rule_remove_dots'        => array(
				'label'   => __( 'Remove dots (.)', 'alt-audit' ),
				'default' => true,
				'example' => 'my.image.jpg → my image',
			),
			'rule_remove_commas'      => array(
				'label'   => __( 'Remove commas (,)', 'alt-audit' ),
				'default' => true,
				'example' => 'my,image.jpg → my image',
			),
			'rule_remove_numbers'     => array(
				'label'   => __( 'Remove all numbers', 'alt-audit' ),
				'default' => false,
				'example' => 'image123.jpg → image',
			),
		);
	}

	/**
	 * Get case formatting options
	 *
	 * @return array Case formatting options.
	 */
	public function get_case_formatting_options() {
		return array(
			'sentence'  => array(
				'label'   => __( 'Sentence case', 'alt-audit' ),
				'example' => 'First word capitalized',
			),
			'title'     => array(
				'label'   => __( 'Title Case', 'alt-audit' ),
				'example' => 'Every Word Capitalized',
			),
			'lowercase' => array(
				'label'   => __( 'lowercase', 'alt-audit' ),
				'example' => 'all words lowercase',
			),
			'uppercase' => array(
				'label'   => __( 'UPPERCASE', 'alt-audit' ),
				'example' => 'ALL WORDS UPPERCASE',
			),
		);
	}

	/**
	 * Test filename cleaning with current settings
	 *
	 * @param string $filename Filename to test.
	 * @return string Cleaned result.
	 */
	public function test_filename_cleaning( $filename ) {
		return $this->clean_image_name( $filename );
	}
}
