<?php
/**
 * Quality Service
 *
 * @package AltAudit
 * @since 1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Quality Service class
 *
 * Main business logic service for quality assessment operations.
 * Coordinates between analyzers, validators, and data models.
 *
 * @since 1.0.0
 */
class Altaudit82ai_Quality_Service {


	/**
	 * Quality analyzer
	 *
	 * @var Altaudit82ai_Quality_Analyzer
	 */
	private $analyzer;

	/**
	 * Quality scorer
	 *
	 * @var Altaudit82ai_Quality_Scorer
	 */
	private $scorer;

	/**
	 * Quality criteria model
	 *
	 * @var Altaudit82ai_Quality_Criteria_Model
	 */
	private $criteria_model;

	/**
	 * Settings service
	 *
	 * @var Altaudit82ai_Settings
	 */
	private $settings;

	/**
	 * Container instance
	 *
	 * @var Altaudit82ai_Container
	 */
	private $container;

	/**
	 * Constructor
	 *
	 * @param Altaudit82ai_Container $container Container instance.
	 */
	public function __construct( ?Altaudit82ai_Container $container = null ) {
		$this->container = $container;

		if ( $container ) {
			$this->analyzer       = $container->get( 'quality_analyzer' );
			$this->scorer         = $container->get( 'quality_scorer' );
			$this->criteria_model = $container->get( 'quality_criteria_model' );
			$this->settings       = $container->get( 'settings' );
		}
	}

	/**
	 * Assess alt text quality
	 *
	 * @param string $alt_text Alt text to assess.
	 * @param array  $context  Additional context for assessment.
	 * @return array Quality assessment result.
	 */
	public function assess_quality( $alt_text, $context = array() ) {
		try {
			// Initialize result model.
			$result = new Altaudit82ai_Quality_Result_Model();

			// Basic text analysis.
			$basic_analysis = $this->analyzer->analyze_basic_properties( $alt_text );
			$result->update( $basic_analysis );

			// Determine quality status and run specific checker.
			$status = $this->determine_quality_status( $alt_text, $context );
			$result->set( 'status', $status );

			// Get checker-specific assessment.
			$checker_result = $this->get_checker_assessment( $status, $alt_text, $context );
			$result->update( $checker_result );

			// Calculate scores.
			$scores = $this->scorer->calculate_scores( $alt_text, $context, $result->to_array() );
			$result->update( $scores );

			// Check WCAG compliance.
			$wcag_compliance = $this->analyzer->check_wcag_compliance( $alt_text, $result->to_array() );
			$result->set( 'wcag_compliance', $wcag_compliance );

			// Generate suggestions if needed.
			if ( $result->get_accessibility_score() < 70 ) {
				$suggestions = $this->generate_suggestions( $alt_text, $result->to_array() );
				$result->set( 'suggestions', $suggestions );
			}

			// Save assessment if enabled.
			if ( $this->should_save_assessment( $context ) ) {
				$this->save_assessment( $result, $context );
			}

			return $result->to_array();

		} catch ( Exception $e ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Error logging for debugging.
			error_log( 'Alt Audit Quality Service Error: ' . $e->getMessage() );

			return array(
				'error'   => __( 'Quality assessment failed', 'alt-audit' ),
				'score'   => 0,
				'status'  => 'error',
				'message' => __( 'An error occurred during quality assessment', 'alt-audit' ),
			);
		}
	}

	/**
	 * Batch assess multiple alt texts
	 *
	 * @param array $alt_texts Array of alt texts to assess.
	 * @param array $context   Common context for all assessments.
	 * @return array Batch assessment results.
	 */
	public function batch_assess( $alt_texts, $context = array() ) {
		$results = array();
		$stats   = array(
			'total'      => count( $alt_texts ),
			'missing'    => 0,
			'decorative' => 0,
			'weak'       => 0,
			'good'       => 0,
			'excellent'  => 0,
			'avg_score'  => 0,
		);

		$total_score = 0;

		foreach ( $alt_texts as $index => $alt_text ) {
			$assessment        = $this->assess_quality( $alt_text, $context );
			$results[ $index ] = $assessment;

			// Update stats.
			$status = $assessment['status'];
			if ( isset( $stats[ $status ] ) ) {
				++$stats[ $status ];
			}

			$total_score += $assessment['accessibility_score'] ?? 0;
		}

		// Calculate average score.
		if ( $stats['total'] > 0 ) {
			$stats['avg_score'] = round( $total_score / $stats['total'], 2 );
		}

		return array(
			'results' => $results,
			'stats'   => $stats,
		);
	}

	/**
	 * Get quality suggestions for alt text
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @param array  $context  Additional context.
	 * @return array Quality suggestions.
	 */
	public function get_suggestions( $alt_text, $context = array() ) {
		$assessment  = $this->assess_quality( $alt_text, $context );
		$suggestions = $assessment['suggestions'] ?? array();

		// Add context-specific suggestions.
		$context_suggestions = $this->get_context_specific_suggestions( $alt_text, $context );
		$suggestions         = array_merge( $suggestions, $context_suggestions );

		// Add examples if helpful.
		if ( $assessment['score'] < 60 ) {
			$examples = $this->get_examples_for_improvement( $alt_text, $context );
			if ( ! empty( $examples ) ) {
				$suggestions[] = __( 'Examples of better alt text:', 'alt-audit' );
				$suggestions   = array_merge( $suggestions, $examples );
			}
		}

		return array_unique( $suggestions );
	}

	/**
	 * Get post quality statistics
	 *
	 * @param int $post_id Post ID.
	 * @return array Quality statistics for the post.
	 */
	public function get_post_quality_stats( $post_id ) {
		// Get real-time stats by analyzing current images.
		$current_stats = $this->analyze_post_images( $post_id );

		return array(
			'current' => $current_stats,
			'post_id' => $post_id,
		);
	}

	/**
	 * Compare two alt texts
	 *
	 * @param string $alt_text_1 First alt text.
	 * @param string $alt_text_2 Second alt text.
	 * @return array Comparison result.
	 */
	public function compare_alt_texts( $alt_text_1, $alt_text_2 ) {
		$result_1 = $this->assess_quality( $alt_text_1 );
		$result_2 = $this->assess_quality( $alt_text_2 );

		$improvement = $result_2['accessibility_score'] - $result_1['accessibility_score'];
		$better      = $improvement > 0 ? 2 : ( $improvement < 0 ? 1 : 0 );

		return array(
			'text_1'      => $result_1,
			'text_2'      => $result_2,
			'improvement' => $improvement,
			'better'      => $better,
			'summary'     => $this->generate_comparison_summary( $result_1, $result_2, $improvement ),
		);
	}

	/**
	 * Check if alt text meets quality threshold
	 *
	 * @param string $alt_text Alt text to check.
	 * @return bool True if meets threshold.
	 */
	public function meets_threshold( $alt_text ) {
		$threshold  = $this->get_quality_threshold();
		$assessment = $this->assess_quality( $alt_text );

		return ( $assessment['accessibility_score'] ?? 0 ) >= $threshold;
	}

	/**
	 * Determine quality status based on alt text
	 *
	 * WCAG-aligned: Checks decorative status BEFORE missing status to correctly
	 * identify empty alt text on decorative images as compliant (not missing).
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @param array  $context  Additional context.
	 * @return string Quality status.
	 */
	private function determine_quality_status( $alt_text, $context = array() ) {
		// Check if decorative first (handles both empty and non-empty decorative images).
		// Per WCAG 1.1.1, decorative images SHOULD have empty alt text - this is correct.
		if ( $this->analyzer->is_decorative( $alt_text, $context ) ) {
			return 'decorative';
		}

		// Now check if missing (non-decorative images with empty alt text).
		// Per WCAG 1.1.1, non-decorative images MUST have alt text - this is a violation.
		if ( empty( trim( $alt_text ) ) ) {
			return 'missing';
		}

		// Calculate preliminary score to determine status.
		$score = $this->scorer->calculate_preliminary_score( $alt_text );

		// Use criteria model to determine status by score.
		return $this->criteria_model->get_status_by_score( $score );
	}

	/**
	 * Get checker-specific assessment
	 *
	 * @param string $status   Quality status.
	 * @param string $alt_text Alt text.
	 * @param array  $context  Context data.
	 * @return array Checker assessment result.
	 */
	private function get_checker_assessment( $status, $alt_text, $context ) {
		$checker_class = 'Altaudit82ai_' . ucfirst( $status ) . '_Checker';

		if ( class_exists( $checker_class ) ) {
			$checker = new $checker_class();
			if ( method_exists( $checker, 'assess' ) ) {
				return $checker->assess( $alt_text, $context );
			}
		}

		// Fallback assessment.
		return array(
			'level'           => $this->criteria_model->get_quality_status( $status )['priority'] ?? 'medium',
			'issues'          => array(),
			'suggestions'     => array(),
			'action_required' => 'missing' === $status,
		);
	}

	/**
	 * Generate improvement suggestions
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @param array  $result   Assessment result.
	 * @return array Suggestions array.
	 */
	private function generate_suggestions( $alt_text, $result ) {
		$suggestions = array();

		// Length-based suggestions.
		if ( $result['length'] > 150 ) {
			$suggestions[] = __( 'Consider shortening the alt text to be more concise.', 'alt-audit' );
		} elseif ( $result['length'] < 25 && ! empty( $alt_text ) ) {
			$suggestions[] = __( 'Consider adding more descriptive details.', 'alt-audit' );
		}

		// Word count suggestions.
		if ( $result['word_count'] < 3 && ! empty( $alt_text ) ) {
			$suggestions[] = __( 'Add more descriptive words to better explain the image.', 'alt-audit' );
		}

		// Structure suggestions.
		if ( preg_match( '/^(image|picture|photo) (of|showing)/i', $alt_text ) ) {
			$suggestions[] = __( 'Remove redundant phrases like "image of" or "picture of".', 'alt-audit' );
		}

		return $suggestions;
	}

	/**
	 * Get context-specific suggestions
	 *
	 * @param string $alt_text Alt text.
	 * @param array  $context  Context data.
	 * @return array Context-specific suggestions.
	 */
	private function get_context_specific_suggestions( $alt_text, $context ) {
		$suggestions  = array();
		$context_type = $context['type'] ?? 'general';

		switch ( $context_type ) {
			case 'product':
				if ( ! preg_match( '/\b(color|size|material|brand)\b/i', $alt_text ) ) {
					$suggestions[] = __( 'Consider including product details like color, size, or material.', 'alt-audit' );
				}
				break;

			case 'navigation':
				if ( strlen( $alt_text ) > 60 ) {
					$suggestions[] = __( 'Navigation images should have concise alt text.', 'alt-audit' );
				}
				break;

			case 'article':
				if ( ! preg_match( '/\b(shows|displays|illustrates|demonstrates)\b/i', $alt_text ) ) {
					$suggestions[] = __( 'Explain how the image relates to the article content.', 'alt-audit' );
				}
				break;
		}

		return $suggestions;
	}

	/**
	 * Get examples for improvement
	 *
	 * @param string $alt_text Alt text.
	 * @param array  $context  Context data.
	 * @return array Example improvements.
	 */
	private function get_examples_for_improvement( $alt_text, $context ) {
		$examples     = array();
		$context_type = $context['type'] ?? 'general';

		// Context-specific examples.
		switch ( $context_type ) {
			case 'product':
				$examples = array(
					__( 'Blue wireless headphones with noise canceling feature', 'alt-audit' ),
					__( 'Stainless steel kitchen mixer with multiple attachments', 'alt-audit' ),
				);
				break;

			case 'article':
				$examples = array(
					__( 'Team meeting around conference table discussing project plans', 'alt-audit' ),
					/* translators: Placeholder for dynamic content */
					__( 'Graph showing 25% increase in website traffic over 6 months', 'alt-audit' ),
				);
				break;

			default:
				$examples = array(
					__( 'Person typing on laptop in modern coffee shop', 'alt-audit' ),
					__( 'Red sports car parked in front of glass office building', 'alt-audit' ),
				);
		}

		return array_slice( $examples, 0, 2 ); // Limit to 2 examples.
	}

	/**
	 * Analyze current images in a post
	 *
	 * @param int $post_id Post ID.
	 * @return array Current image analysis stats.
	 */
	private function analyze_post_images( $post_id ) {
		$post = get_post( $post_id );
		if ( ! $post ) {
			return array( 'error' => __( 'Post not found', 'alt-audit' ) );
		}

		// Extract images from post content.
		$content = $post->post_content;
		preg_match_all( '/<img[^>]+alt=(["\'])([^"\']*)\1[^>]*>/i', $content, $matches );

		$alt_texts = $matches[2] ?? array();

		// Also check featured image.
		$featured_id = get_post_thumbnail_id( $post_id );
		if ( $featured_id ) {
			$featured_alt = get_post_meta( $featured_id, '_wp_attachment_image_alt', true );
			if ( false !== $featured_alt ) {
				array_unshift( $alt_texts, $featured_alt );
			}
		}

		if ( empty( $alt_texts ) ) {
			return array(
				'total'   => 0,
				'message' => __( 'No images found in post', 'alt-audit' ),
			);
		}

		// Assess all alt texts.
		$batch_result = $this->batch_assess( $alt_texts );
		return $batch_result['stats'];
	}

	/**
	 * Generate comparison summary
	 *
	 * @param array $result_1    First assessment result.
	 * @param array $result_2    Second assessment result.
	 * @param int   $improvement Score improvement.
	 * @return string Comparison summary.
	 */
	private function generate_comparison_summary( $result_1, $result_2, $improvement ) {
		if ( $improvement > 10 ) {
			return __( 'Significant improvement in accessibility and quality.', 'alt-audit' );
		} elseif ( $improvement > 0 ) {
			return __( 'Some improvement in accessibility and quality.', 'alt-audit' );
		} elseif ( $improvement < -10 ) {
			return __( 'Significant decrease in accessibility and quality.', 'alt-audit' );
		} elseif ( $improvement < 0 ) {
			return __( 'Some decrease in accessibility and quality.', 'alt-audit' );
		} else {
			return __( 'No significant change in accessibility and quality.', 'alt-audit' );
		}
	}

	/**
	 * Save quality assessment
	 *
	 * @param Altaudit82ai_Quality_Result_Model $result  Assessment result.
	 * @param array                             $context Assessment context.
	 * @return void
	 */
	private function save_assessment( $result, $context ) {
		$image_id = absint( $context['image_id'] ?? 0 );
		if ( ! $image_id ) {
			return;
		}

		$data   = $result->to_array();
		$status = sanitize_key( $data['status'] ?? 'unknown' );

		// Save quality status as taxonomy term for efficient filtering.
		if ( in_array( $status, Altaudit82ai_Taxonomy::TERMS, true ) ) {
			Altaudit82ai_Taxonomy::set_status( $image_id, $status );
		}

		// Save quality score to post meta (numeric value, taxonomy not suitable).
		update_post_meta( $image_id, '_altaudit82ai_quality_score', absint( $data['accessibility_score'] ?? 0 ) );
	}

	/**
	 * Check if assessment should be saved
	 *
	 * @param array $context Assessment context.
	 * @return bool True if should save.
	 */
	private function should_save_assessment( $context ) {
		// Only save if we have image or post context and saving is enabled.
		$has_context    = ! empty( $context['image_id'] ) || ! empty( $context['post_id'] );
		$saving_enabled = $this->settings ? $this->settings->get( 'save_assessments', true ) : true;

		return $has_context && $saving_enabled;
	}

	/**
	 * Get quality threshold from settings
	 *
	 * @return int Quality threshold.
	 */
	private function get_quality_threshold() {
		return $this->settings ? $this->settings->get( 'quality_threshold', 70 ) : 70;
	}
}
