<?php
/**
 * Quality Scorer
 *
 * @package AltAudit
 * @since 1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Quality Scorer class
 *
 * Handles scoring algorithms for quality assessment.
 * Calculates various quality scores based on different criteria.
 *
 * @since 1.0.0
 */
class Altaudit82ai_Quality_Scorer {


	/**
	 * Quality criteria model
	 *
	 * @var Altaudit82ai_Quality_Criteria_Model
	 */
	private $criteria_model;

	/**
	 * Constructor
	 *
	 * @param Altaudit82ai_Quality_Criteria_Model $criteria_model Criteria model instance.
	 */
	public function __construct( ?Altaudit82ai_Quality_Criteria_Model $criteria_model = null ) {
		$this->criteria_model = $criteria_model ? $criteria_model : new Altaudit82ai_Quality_Criteria_Model();
	}

	/**
	 * Calculate comprehensive quality scores
	 *
	 * @param string $alt_text Alt text to score.
	 * @param array  $context  Context information.
	 * @param array  $analysis Analysis data from analyzer.
	 * @return array Calculated scores.
	 */
	public function calculate_scores( $alt_text, $context, $analysis ) {
		$base_score          = $this->calculate_base_score( $alt_text, $analysis );
		$context_score       = $this->calculate_context_score( $alt_text, $context );
		$accessibility_score = $this->calculate_accessibility_score( $alt_text, $analysis );

		// Combine scores with weights.
		$final_score = $this->combine_scores( $base_score, $context_score, $accessibility_score );

		return array(
			'score'               => $final_score,
			'base_score'          => $base_score,
			'context_score'       => $context_score,
			'accessibility_score' => $accessibility_score,
			'score_breakdown'     => $this->get_score_breakdown( $alt_text, $analysis, $context ),
		);
	}

	/**
	 * Calculate preliminary score for status determination
	 *
	 * @param string $alt_text Alt text to score.
	 * @return int Preliminary score (0-100).
	 */
	public function calculate_preliminary_score( $alt_text ) {
		if ( empty( trim( $alt_text ) ) ) {
			return 0;
		}

		$length     = strlen( $alt_text );
		$word_count = str_word_count( $alt_text );

		$score = 0;

		// Quick length scoring - more lenient ranges.
		if ( $length >= 20 && $length <= 200 ) {
			$score += 45;
		} elseif ( $length > 0 ) {
			$score += 25;
		}

		// Quick word count scoring - more lenient ranges.
		if ( $word_count >= 3 && $word_count <= 25 ) {
			$score += 35;
		} elseif ( $word_count > 0 ) {
			$score += 20;
		}

		// Basic structure check.
		if ( preg_match( '/^[A-Z]/', $alt_text ) ) {
			$score += 15;
		}

		// Check for obvious issues.
		if ( preg_match( '/\.(jpg|jpeg|png|gif)$/i', $alt_text ) ) {
			$score -= 15;
		}

		if ( preg_match( '/^(image|picture|photo) of/i', $alt_text ) ) {
			$score -= 5;
		}

		return max( 0, min( 100, $score ) );
	}

	/**
	 * Calculate base quality score
	 *
	 * @param string $alt_text Alt text to score.
	 * @param array  $analysis Analysis data.
	 * @return int Base score (0-100).
	 */
	private function calculate_base_score( $alt_text, $analysis ) {
		$score    = 0;
		$criteria = $this->criteria_model->get_scoring_criteria();

		// Length scoring.
		$length_score = $this->score_length( $analysis['length'], $criteria['length'] );
		$score       += $length_score;

		// Word count scoring.
		$word_score = $this->score_word_count( $analysis['word_count'], $criteria['word_count'] );
		$score     += $word_score;

		// Descriptiveness scoring.
		$descriptive_data  = $analysis['descriptive_words'];
		$descriptive_score = $this->score_descriptiveness( $descriptive_data, $criteria['descriptiveness'] );
		$score            += $descriptive_score;

		// Structure scoring.
		$structure_data  = $analysis['structure_score'];
		$structure_score = is_array( $structure_data ) ? $structure_data['score'] : $structure_data;
		$score          += min( $structure_score, $criteria['structure']['max_points'] );

		return min( $score, 100 );
	}

	/**
	 * Calculate context-specific score
	 *
	 * @param string $alt_text Alt text to score.
	 * @param array  $context  Context information.
	 * @return int Context score (0-100).
	 */
	private function calculate_context_score( $alt_text, $context ) {
		$context_type     = $context['type'] ?? 'general';
		$context_criteria = $this->criteria_model->get_context_criteria( $context_type );

		$score = 80; // Start with good base score.
		// Length preference for context.
		$length      = strlen( $alt_text );
		$length_pref = $context_criteria['length_preference'];

		if ( $length >= $length_pref['min'] && $length <= $length_pref['max'] ) {
			$score += 20;
		} elseif ( $length > 0 ) {
			// Deduct based on how far from optimal.
			$deviation = min(
				abs( $length - $length_pref['min'] ),
				abs( $length - $length_pref['max'] )
			);
			$score    -= min( 20, $deviation / 5 );
		}

		// Context-specific requirements.
		$required_elements = $context_criteria['required_elements'] ?? array();
		$found_elements    = $this->check_required_elements( $alt_text, $required_elements );
		$element_score     = ( count( $found_elements ) / max( 1, count( $required_elements ) ) ) * 20;
		$score             = ( $score * 0.8 ) + $element_score; // Weighted combination.
		return max( 0, min( 100, round( $score ) ) );
	}

	/**
	 * Calculate accessibility-focused score
	 *
	 * @param string $alt_text Alt text to score.
	 * @param array  $analysis Analysis data.
	 * @return int Accessibility score (0-100).
	 */
	private function calculate_accessibility_score( $alt_text, $analysis ) {
		$score = 0;

		// Start with base score.
		$base_score = $this->calculate_base_score( $alt_text, $analysis );
		$score     += $base_score * 0.7; // 70% weight for base score.
		// Quality tier bonus/penalty.
		// Level A/AA = real WCAG compliance, "AAA" = plugin's premium quality tier.
		$wcag_data = $analysis['wcag_compliance'] ?? array();
		if ( ! empty( $wcag_data ) ) {
			if ( $wcag_data['level_aaa'] ?? false ) {
				$score += 15; // Premium quality bonus.
			} elseif ( $wcag_data['level_aa'] ?? false ) {
				$score += 10; // WCAG AA compliant.
			} elseif ( $wcag_data['level_a'] ?? false ) {
				$score += 5;  // WCAG A compliant.
			} else {
				$score -= 20; // Missing required alt text.
			}
		}

		// Language quality bonus.
		$language_quality = $analysis['language_quality'] ?? array();
		if ( ! empty( $language_quality ) ) {
			$readability = $language_quality['readability_score'] ?? 0;
			$clarity     = $language_quality['clarity_score'] ?? 0;
			$specificity = $language_quality['specificity_score'] ?? 0;

			$language_score = ( $readability + $clarity + $specificity ) / 3;
			$score         += ( $language_score / 100 ) * 15; // Up to 15 bonus points.
		}

		return max( 0, min( 100, round( $score ) ) );
	}

	/**
	 * Combine multiple scores with appropriate weights
	 *
	 * @param int $base_score        Base quality score.
	 * @param int $context_score     Context-specific score.
	 * @param int $accessibility_score Accessibility score.
	 * @return int Combined final score.
	 */
	private function combine_scores( $base_score, $context_score, $accessibility_score ) {
		// Weighted combination: 40% base, 20% context, 40% accessibility.
		$final_score = ( $base_score * 0.4 ) + ( $context_score * 0.2 ) + ( $accessibility_score * 0.4 );

		return max( 0, min( 100, round( $final_score ) ) );
	}

	/**
	 * Score alt text length
	 *
	 * @param int   $length   Text length.
	 * @param array $criteria Length criteria.
	 * @return int Length score.
	 */
	private function score_length( $length, $criteria ) {
		$max_points = $criteria['max_points'];

		if ( $length >= $criteria['optimal_min'] && $length <= $criteria['optimal_max'] ) {
			return $max_points;
		} elseif ( $length >= $criteria['acceptable_min'] && $length <= $criteria['acceptable_max'] ) {
			return round( $max_points * 0.8 );
		} elseif ( $length > 0 ) {
			return round( $max_points * 0.4 );
		}

		return 0;
	}

	/**
	 * Score word count
	 *
	 * @param int   $word_count Word count.
	 * @param array $criteria   Word count criteria.
	 * @return int Word count score.
	 */
	private function score_word_count( $word_count, $criteria ) {
		$max_points = $criteria['max_points'];

		if ( $word_count >= $criteria['optimal_min'] && $word_count <= $criteria['optimal_max'] ) {
			return $max_points;
		} elseif ( $word_count >= $criteria['acceptable_min'] && $word_count <= $criteria['acceptable_max'] ) {
			return round( $max_points * 0.75 );
		} elseif ( $word_count > 0 ) {
			return round( $max_points * 0.3 );
		}

		return 0;
	}

	/**
	 * Score descriptiveness
	 *
	 * @param array $descriptive_data Descriptive words data.
	 * @param array $criteria         Descriptiveness criteria.
	 * @return int Descriptiveness score.
	 */
	private function score_descriptiveness( $descriptive_data, $criteria ) {
		$max_points = $criteria['max_points'];

		// Handle both array and integer formats.
		$count          = is_array( $descriptive_data ) ? $descriptive_data['count'] : $descriptive_data;
		$weighted_score = is_array( $descriptive_data ) ? ( $descriptive_data['weighted_score'] ?? $count ) : $count;

		if ( $weighted_score >= $criteria['min_descriptive_words'] ) {
			return $max_points;
		} elseif ( $count >= $criteria['good_descriptive_words'] ) {
			return round( $max_points * 0.75 );
		} elseif ( $count >= $criteria['basic_descriptive_words'] ) {
			return round( $max_points * 0.5 );
		}

		return 0;
	}

	/**
	 * Check for required elements in context
	 *
	 * @param string $alt_text         Alt text to check.
	 * @param array  $required_elements Required elements for context.
	 * @return array Found elements.
	 */
	private function check_required_elements( $alt_text, $required_elements ) {
		$found          = array();
		$alt_text_lower = strtolower( $alt_text );

		foreach ( $required_elements as $element ) {
			switch ( $element ) {
				case 'product_name':
					if ( preg_match( '/\b[A-Z][a-z]+ [A-Z][a-z]+\b/', $alt_text ) ) {
						$found[] = $element;
					}
					break;

				case 'key_features':
					if ( preg_match( '/\b(wireless|bluetooth|noise.canceling|waterproof|portable)\b/i', $alt_text ) ) {
						$found[] = $element;
					}
					break;

				case 'visual_appearance':
					$patterns = $this->criteria_model->get_descriptive_patterns();
					foreach ( array( 'colors', 'sizes', 'shapes' ) as $category ) {
						if ( isset( $patterns[ $category ] ) && preg_match( $patterns[ $category ]['pattern'], $alt_text ) ) {
							$found[] = $element;
							break;
						}
					}
					break;

				case 'relevance_to_content':
					if ( preg_match( '/\b(shows|displays|illustrates|demonstrates|example)\b/i', $alt_text ) ) {
						$found[] = $element;
					}
					break;

				case 'navigation_purpose':
					if ( preg_match( '/\b(menu|navigation|home|back|next|previous|search)\b/i', $alt_text ) ) {
						$found[] = $element;
					}
					break;

				case 'destination':
					if ( preg_match( '/\b(go to|visit|open|link to|page)\b/i', $alt_text ) ) {
						$found[] = $element;
					}
					break;
			}
		}

		return $found;
	}

	/**
	 * Get detailed score breakdown
	 *
	 * @param string $alt_text Alt text.
	 * @param array  $analysis Analysis data.
	 * @param array  $context  Context data (reserved for future use).
	 * @return array Score breakdown.
	 *
	 * @phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.FoundAfterLastUsed -- Reserved for future context-aware scoring.
	 */
	private function get_score_breakdown( $alt_text, $analysis, $context ) {
		$criteria = $this->criteria_model->get_scoring_criteria();

		return array(
			'length'          => array(
				'score'   => $this->score_length( $analysis['length'], $criteria['length'] ),
				'max'     => $criteria['length']['max_points'],
				'details' => sprintf( /* translators: Placeholder for dynamic content */ __( '%d characters', 'alt-audit' ), $analysis['length'] ),
			),
			'word_count'      => array(
				'score'   => $this->score_word_count( $analysis['word_count'], $criteria['word_count'] ),
				'max'     => $criteria['word_count']['max_points'],
				'details' => sprintf( /* translators: Placeholder for dynamic content */ __( '%d words', 'alt-audit' ), $analysis['word_count'] ),
			),
			'descriptiveness' => array(
				'score'   => $this->score_descriptiveness( $analysis['descriptive_words'], $criteria['descriptiveness'] ),
				'max'     => $criteria['descriptiveness']['max_points'],
				'details' => sprintf(
					/* translators: Placeholder for dynamic content */
					__( '%d descriptive words', 'alt-audit' ),
					is_array( $analysis['descriptive_words'] ) ? $analysis['descriptive_words']['count'] : $analysis['descriptive_words']
				),
			),
			'structure'       => array(
				'score'   => is_array( $analysis['structure_score'] ) ? $analysis['structure_score']['score'] : $analysis['structure_score'],
				'max'     => $criteria['structure']['max_points'],
				'details' => __( 'Grammar and structure', 'alt-audit' ),
			),
		);
	}
}
