<?php
/**
 * Legacy Quality Checker Wrapper
 *
 * @package AltAudit
 * @since 1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Legacy Quality Checker class
 *
 * Provides backward compatibility for the old Quality Checker API
 * while using the new MVC architecture under the hood.
 *
 * @since 1.0.0
 * @deprecated Use Altaudit82ai_Quality_Service instead
 */
class Altaudit82ai_Quality_Checker {


	/**
	 * Quality service instance
	 *
	 * @var Altaudit82ai_Quality_Service
	 */
	private $quality_service;

	/**
	 * Settings service
	 *
	 * @var Altaudit82ai_Settings
	 */
	private $settings;

	/**
	 * Container instance
	 *
	 * @var Altaudit82ai_Container
	 */
	private $container;

	/**
	 * Constructor
	 *
	 * @param Altaudit82ai_Settings $settings Settings service.
	 */
	public function __construct( ?Altaudit82ai_Settings $settings = null ) {
		$this->settings = $settings;

		// Get container instance from plugin singleton.
		$plugin = function_exists( 'altaudit82ai' ) ? altaudit82ai() : null;
		if ( $plugin && method_exists( $plugin, 'get_service' ) ) {
			$this->container       = $plugin->get_service( 'container' );
			$this->quality_service = $plugin->get_service( 'quality_service' );
		}

		// Fallback if container not available.
		if ( ! $this->quality_service ) {
			$this->init_fallback_service();
		}
	}

	/**
	 * Initialize fallback service if container not available
	 *
	 * @return void
	 */
	private function init_fallback_service() {
		// Create minimal service stack for compatibility.
		$criteria_model = new Altaudit82ai_Quality_Criteria_Model();
		$analyzer       = new Altaudit82ai_Quality_Analyzer( $criteria_model );
		$scorer         = new Altaudit82ai_Quality_Scorer( $criteria_model );
		$quality_model  = new Altaudit82ai_Quality_Model();

		// Create a basic container for the service.
		$container = new Altaudit82ai_Container();
		$container->register( 'quality_criteria_model', $criteria_model );
		$container->register( 'quality_analyzer', $analyzer );
		$container->register( 'quality_scorer', $scorer );
		$container->register( 'quality_model', $quality_model );
		$container->register( 'settings', $this->settings );

		$this->quality_service = new Altaudit82ai_Quality_Service( $container );
	}

	/**
	 * Check alt text quality (legacy method)
	 *
	 * @param string $alt_text Alt text to check.
	 * @param array  $context  Additional context for checking.
	 * @return array Quality assessment result.
	 */
	public function check_quality( $alt_text, $context = array() ) {
		if ( ! $this->quality_service ) {
			return $this->get_fallback_result( $alt_text );
		}

		$result = $this->quality_service->assess_quality( $alt_text, $context );

		// Convert new format to legacy format if needed.
		return $this->convert_to_legacy_format( $result );
	}

	/**
	 * Get quality statistics for multiple texts (legacy method)
	 *
	 * @param array $alt_texts Array of alt texts to analyze.
	 * @return array Quality statistics.
	 */
	public function get_batch_statistics( $alt_texts ) {
		if ( ! $this->quality_service ) {
			return $this->get_fallback_batch_stats( $alt_texts );
		}

		$batch_result = $this->quality_service->batch_assess( $alt_texts );
		return $batch_result['stats'] ?? array();
	}

	/**
	 * Compare two alt texts (legacy method)
	 *
	 * @param string $alt_text_1 First alt text.
	 * @param string $alt_text_2 Second alt text.
	 * @return array Comparison result.
	 */
	public function compare_alt_texts( $alt_text_1, $alt_text_2 ) {
		if ( ! $this->quality_service ) {
			return $this->get_fallback_comparison( $alt_text_1, $alt_text_2 );
		}

		return $this->quality_service->compare_alt_texts( $alt_text_1, $alt_text_2 );
	}

	/**
	 * Check if alt text meets quality threshold (legacy method)
	 *
	 * @param string $alt_text Alt text to check.
	 * @return bool True if meets threshold.
	 */
	public function meets_threshold( $alt_text ) {
		if ( ! $this->quality_service ) {
			$result = $this->get_fallback_result( $alt_text );
			return ( $result['accessibility_score'] ?? 0 ) >= 70;
		}

		return $this->quality_service->meets_threshold( $alt_text );
	}

	/**
	 * Convert new result format to legacy format
	 *
	 * @param array $result New format result.
	 * @return array Legacy format result.
	 */
	private function convert_to_legacy_format( $result ) {
		// Ensure legacy fields are present.
		$legacy_result = array_merge(
			array(
				'score'               => 0,
				'accessibility_score' => 0,
				'status'              => 'unknown',
				'length'              => 0,
				'word_count'          => 0,
				'descriptive_words'   => 0,
				'structure_score'     => 0,
				'issues'              => array(),
				'suggestions'         => array(),
				'wcag_compliance'     => array(),
				'message'             => '',
				'checker'             => '',
			),
			$result
		);

		// Handle descriptive_words format conversion.
		if ( is_array( $legacy_result['descriptive_words'] ) ) {
			$legacy_result['descriptive_words'] = $legacy_result['descriptive_words']['count'] ?? 0;
		}

		// Handle structure_score format conversion.
		if ( is_array( $legacy_result['structure_score'] ) ) {
			$legacy_result['structure_score'] = $legacy_result['structure_score']['score'] ?? 0;
		}

		// Set checker class name for legacy compatibility.
		if ( empty( $legacy_result['checker'] ) ) {
			$status                   = $legacy_result['status'];
			$legacy_result['checker'] = 'Altaudit82ai_Quality_' . ucfirst( $status );
		}

		return $legacy_result;
	}

	/**
	 * Get fallback result when service unavailable
	 *
	 * @param string $alt_text Alt text to check.
	 * @return array Fallback result.
	 */
	private function get_fallback_result( $alt_text ) {
		$length     = strlen( $alt_text );
		$word_count = str_word_count( $alt_text );

		// Basic scoring logic.
		$score = 0;
		if ( empty( trim( $alt_text ) ) ) {
			$status = 'missing';
			$score  = 0;
		} elseif ( $length < 10 ) {
			$status = 'weak';
			$score  = 25;
		} elseif ( $length > 150 ) {
			$status = 'weak';
			$score  = 40;
		} elseif ( $length >= 40 && $length <= 125 && $word_count >= 6 ) {
			$status = 'good';
			$score  = 75;
		} else {
			$status = 'weak';
			$score  = 50;
		}

		return array(
			'score'               => $score,
			'accessibility_score' => $score,
			'status'              => $status,
			'length'              => $length,
			'word_count'          => $word_count,
			'descriptive_words'   => 0,
			'structure_score'     => 0,
			'issues'              => empty( trim( $alt_text ) ) ? array( __( 'Alt text is missing', 'alt-audit' ) ) : array(),
			'suggestions'         => array( __( 'Quality service unavailable - using fallback assessment', 'alt-audit' ) ),
			'wcag_compliance'     => array(
				'level_a'   => ! empty( trim( $alt_text ) ),
				'level_aa'  => $length <= 150,
				'level_aaa' => $score >= 70,
				'issues'    => array(),
			),
			'message'             => __( 'Fallback quality assessment', 'alt-audit' ),
			'checker'             => 'Altaudit82ai_Quality_Checker_Legacy',
		);
	}

	/**
	 * Get fallback batch statistics
	 *
	 * @param array $alt_texts Alt texts to analyze.
	 * @return array Fallback statistics.
	 */
	private function get_fallback_batch_stats( $alt_texts ) {
		$stats = array(
			'total'      => count( $alt_texts ),
			'missing'    => 0,
			'decorative' => 0,
			'weak'       => 0,
			'good'       => 0,
			'excellent'  => 0,
			'avg_score'  => 0,
			'avg_length' => 0,
		);

		$total_score  = 0;
		$total_length = 0;

		foreach ( $alt_texts as $alt_text ) {
			$result = $this->get_fallback_result( $alt_text );
			++$stats[ $result['status'] ];
			$total_score  += $result['score'];
			$total_length += $result['length'];
		}

		$stats['avg_score']  = $stats['total'] > 0 ? round( $total_score / $stats['total'], 2 ) : 0;
		$stats['avg_length'] = $stats['total'] > 0 ? round( $total_length / $stats['total'], 2 ) : 0;

		return $stats;
	}

	/**
	 * Get fallback comparison
	 *
	 * @param string $alt_text_1 First alt text.
	 * @param string $alt_text_2 Second alt text.
	 * @return array Fallback comparison.
	 */
	private function get_fallback_comparison( $alt_text_1, $alt_text_2 ) {
		$result_1 = $this->get_fallback_result( $alt_text_1 );
		$result_2 = $this->get_fallback_result( $alt_text_2 );

		$improvement = $result_2['score'] - $result_1['score'];
		$better      = $improvement > 0 ? 2 : ( $improvement < 0 ? 1 : 0 );

		return array(
			'text_1'      => $result_1,
			'text_2'      => $result_2,
			'improvement' => $improvement,
			'better'      => $better,
			'summary'     => __( 'Fallback comparison - limited accuracy', 'alt-audit' ),
		);
	}

	/**
	 * Magic method to maintain compatibility with old method calls
	 *
	 * @param string $method Method name.
	 * @param array  $args   Method arguments.
	 * @return mixed Method result.
	 */
	public function __call( $method, $args ) {
		// Log deprecated method usage.
		if ( WP_DEBUG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only when WP_DEBUG is enabled.
			error_log( "Alt Audit: Deprecated method call - {$method}. Please update to use Altaudit82ai_Quality_Service." );
		}

		// Try to delegate to quality service if available.
		if ( $this->quality_service && method_exists( $this->quality_service, $method ) ) {
			return call_user_func_array( array( $this->quality_service, $method ), $args );
		}

		// Return null for unknown methods.
		return null;
	}
}
