<?php
/**
 * Quality Analyzer
 *
 * @package AltAudit
 * @since 1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Quality Analyzer class
 *
 * Analyzes alt text properties and determines quality characteristics.
 * Provides detailed analysis of text structure, content, and accessibility features.
 *
 * @since 1.0.0
 */
class Altaudit82ai_Quality_Analyzer {


	/**
	 * Quality criteria model
	 *
	 * @var Altaudit82ai_Quality_Criteria_Model
	 */
	private $criteria_model;

	/**
	 * Constructor
	 *
	 * @param Altaudit82ai_Quality_Criteria_Model $criteria_model Criteria model instance.
	 */
	public function __construct( ?Altaudit82ai_Quality_Criteria_Model $criteria_model = null ) {
		$this->criteria_model = $criteria_model ?? new Altaudit82ai_Quality_Criteria_Model();
	}

	/**
	 * Analyze basic properties of alt text
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return array Basic properties analysis.
	 */
	public function analyze_basic_properties( $alt_text ) {
		$length            = strlen( $alt_text );
		$word_count        = str_word_count( $alt_text );
		$descriptive_words = $this->count_descriptive_words( $alt_text );

		return array(
			'length'            => $length,
			'word_count'        => $word_count,
			'descriptive_words' => $descriptive_words,
			'structure_score'   => $this->analyze_structure( $alt_text ),
			'character_types'   => $this->analyze_character_types( $alt_text ),
			'language_quality'  => $this->analyze_language_quality( $alt_text ),
		);
	}

	/**
	 * Count descriptive words in alt text
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return int Number of descriptive words found.
	 */
	public function count_descriptive_words( $alt_text ) {
		$patterns       = $this->criteria_model->get_descriptive_patterns();
		$total_count    = 0;
		$weighted_score = 0;

		foreach ( $patterns as $category => $pattern_data ) {
			$matches = preg_match_all( $pattern_data['pattern'], $alt_text );
			$weight  = $pattern_data['weight'] ?? 1.0;

			$total_count    += $matches;
			$weighted_score += $matches * $weight;
		}

		return array(
			'count'          => $total_count,
			'weighted_score' => round( $weighted_score, 2 ),
			'categories'     => $this->get_descriptive_categories( $alt_text ),
		);
	}

	/**
	 * Analyze text structure and grammar
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return array Structure analysis with score.
	 */
	public function analyze_structure( $alt_text ) {
		$score        = 0;
		$issues       = array();
		$achievements = array();

		// Proper capitalization.
		if ( preg_match( '/^[A-Z]/', $alt_text ) ) {
			$score         += 5;
			$achievements[] = __( 'Proper capitalization', 'alt-audit' );
		} else {
			$issues[] = __( 'Should start with capital letter', 'alt-audit' );
		}

		// No file extensions.
		$file_pattern = $this->criteria_model->get_file_extension_pattern();
		if ( ! preg_match( $file_pattern, $alt_text ) ) {
			$score         += 5;
			$achievements[] = __( 'No file extensions', 'alt-audit' );
		} else {
			$issues[] = __( 'Contains file extension', 'alt-audit' );
		}

		// No redundant phrases.
		$redundant_phrases = $this->criteria_model->get_redundant_phrases();
		$has_redundant     = $this->check_redundant_phrases( $alt_text, $redundant_phrases );
		if ( ! $has_redundant ) {
			$score         += 5;
			$achievements[] = __( 'No redundant phrases', 'alt-audit' );
		} else {
			$issues[] = __( 'Contains redundant phrases like "image of"', 'alt-audit' );
		}

		// Proper sentence structure.
		$sentence_score = $this->analyze_sentence_structure( $alt_text );
		$score         += $sentence_score['score'];
		$issues         = array_merge( $issues, $sentence_score['issues'] );
		$achievements   = array_merge( $achievements, $sentence_score['achievements'] );

		// Punctuation analysis.
		$punctuation_score = $this->analyze_punctuation( $alt_text );
		$score            += $punctuation_score['score'];
		$issues            = array_merge( $issues, $punctuation_score['issues'] );
		$achievements      = array_merge( $achievements, $punctuation_score['achievements'] );

		return array(
			'score'        => min( $score, 25 ), // Max 25 points for structure.
			'issues'       => $issues,
			'achievements' => $achievements,
		);
	}

	/**
	 * Check if image appears to be decorative
	 *
	 * WCAG-aligned: Determines decorative status based on purpose and function
	 * (per WCAG 1.1.1), not text length or word count.
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @param array  $context  Additional context.
	 * @return bool True if appears decorative.
	 */
	public function is_decorative( $alt_text, $context = array() ) {
		$alt_text = trim( $alt_text );

		// Empty alt text might be decorative.
		if ( empty( $alt_text ) ) {
			return $this->analyze_decorative_context( $context );
		}

		// Check for explicit decorative terms only.
		$generic_terms = array( 'decoration', 'divider', 'spacer', 'border', 'line', 'dot', 'separator' );
		$trimmed_lower = strtolower( $alt_text );

		foreach ( $generic_terms as $term ) {
			if ( $trimmed_lower === $term || $trimmed_lower === $term . 's' ) {
				return true;
			}
		}

		// WCAG-aligned: Do NOT assume short or single-word alt text is decorative.
		// Valid examples: "Logo", "Menu", "Home", "Search", "Close" are all appropriate.
		// Alt text quality is determined by purpose, not length.

		// Context-based indicators.
		return $this->analyze_decorative_context( $context );
	}

	/**
	 * Check WCAG compliance
	 *
	 * @param string $alt_text Alt text to check.
	 * @param array  $result   Current assessment result.
	 * @return array WCAG compliance status.
	 */
	public function check_wcag_compliance( $alt_text, $result ) {
		$compliance = array(
			'level_a'   => true,
			'level_aa'  => true,
			'level_aaa' => true,
			'issues'    => array(),
			'rules'     => array(),
		);

		$wcag_rules = $this->criteria_model->get_wcag_rules();

		// Check WCAG 2.1 Level A - 1.1.1 Non-text Content.
		if ( empty( $alt_text ) && ( $result['status'] ?? '' ) !== 'decorative' ) {
			$compliance['level_a']        = false;
			$compliance['issues'][]       = __( 'Non-decorative images must have alt text (WCAG 1.1.1)', 'alt-audit' );
			$compliance['rules']['1.1.1'] = false;
		} else {
			$compliance['rules']['1.1.1'] = true;
		}

		// Best practice check (not a WCAG requirement, but recommended for screen readers).
		// Note: WCAG does not specify character limits for alt text.
		if ( strlen( $alt_text ) > 150 ) {
			// Don't fail AA compliance - this is just a recommendation.
			$compliance['issues'][] = __( 'Consider keeping alt text under 150 characters for better screen reader experience', 'alt-audit' );
		}

		// Context-based checks.
		if ( isset( $result['context'] ) && $this->is_link_image( $result['context'] ) ) {
			$link_compliance = $this->check_link_purpose_compliance( $alt_text, $result['context'] );
			if ( ! $link_compliance ) {
				$compliance['level_aa']       = false;
				$compliance['issues'][]       = __( 'Link images should describe link purpose (WCAG 2.4.4)', 'alt-audit' );
				$compliance['rules']['2.4.4'] = false;
			} else {
				$compliance['rules']['2.4.4'] = true;
			}
		}

		// Premium quality check (plugin-specific, not a WCAG requirement).
		// This is an additional quality standard beyond WCAG compliance.
		if ( ( $result['accessibility_score'] ?? 0 ) < 60 ) {
			$compliance['level_aaa'] = false;
			$compliance['issues'][]  = __( 'Alt text quality could be improved for optimal accessibility', 'alt-audit' );
		}

		// Ensure lower levels fail if higher levels fail.
		if ( ! $compliance['level_a'] ) {
			$compliance['level_aa']  = false;
			$compliance['level_aaa'] = false;
		} elseif ( ! $compliance['level_aa'] ) {
			$compliance['level_aaa'] = false;
		}

		return $compliance;
	}

	/**
	 * Analyze character types in alt text
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return array Character type analysis.
	 */
	private function analyze_character_types( $alt_text ) {
		return array(
			'has_numbers'        => preg_match( '/\d/', $alt_text ) ? true : false,
			'has_punctuation'    => preg_match( '/[.,!?;:]/', $alt_text ) ? true : false,
			'has_special'        => preg_match( '/[^a-zA-Z0-9\s.,!?;:]/', $alt_text ) ? true : false,
			'uppercase_ratio'    => $this->calculate_uppercase_ratio( $alt_text ),
			'alphanumeric_ratio' => $this->calculate_alphanumeric_ratio( $alt_text ),
		);
	}

	/**
	 * Analyze language quality
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return array Language quality metrics.
	 */
	private function analyze_language_quality( $alt_text ) {
		return array(
			'readability_score' => $this->calculate_readability_score( $alt_text ),
			'clarity_score'     => $this->calculate_clarity_score( $alt_text ),
			'specificity_score' => $this->calculate_specificity_score( $alt_text ),
		);
	}

	/**
	 * Get descriptive word categories found
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return array Categories of descriptive words found.
	 */
	private function get_descriptive_categories( $alt_text ) {
		$patterns         = $this->criteria_model->get_descriptive_patterns();
		$found_categories = array();

		foreach ( $patterns as $category => $pattern_data ) {
			if ( preg_match( $pattern_data['pattern'], $alt_text ) ) {
				$found_categories[] = $category;
			}
		}

		return $found_categories;
	}

	/**
	 * Check for redundant phrases
	 *
	 * @param string $alt_text        Alt text to check.
	 * @param array  $redundant_phrases Array of redundant phrases.
	 * @return bool True if redundant phrases found.
	 */
	private function check_redundant_phrases( $alt_text, $redundant_phrases ) {
		$alt_text_lower = strtolower( $alt_text );

		foreach ( $redundant_phrases as $phrase ) {
			if ( strpos( $alt_text_lower, strtolower( $phrase ) ) !== false ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Analyze sentence structure
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return array Sentence structure analysis.
	 */
	private function analyze_sentence_structure( $alt_text ) {
		$score        = 0;
		$issues       = array();
		$achievements = array();

		$length = strlen( $alt_text );

		// Complete sentences (for longer descriptions).
		if ( $length > 50 ) {
			if ( preg_match( '/[.!?]$/', $alt_text ) ) {
				$score         += 5;
				$achievements[] = __( 'Complete sentence structure', 'alt-audit' );
			} else {
				$issues[] = __( 'Consider ending with proper punctuation', 'alt-audit' );
			}
		} else {
			// Short descriptions don't need periods.
			$score         += 5;
			$achievements[] = __( 'Appropriate length for phrase structure', 'alt-audit' );
		}

		return array(
			'score'        => $score,
			'issues'       => $issues,
			'achievements' => $achievements,
		);
	}

	/**
	 * Analyze punctuation usage
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return array Punctuation analysis.
	 */
	private function analyze_punctuation( $alt_text ) {
		$score        = 0;
		$issues       = array();
		$achievements = array();

		// Check for excessive punctuation.
		if ( ! preg_match( '/[.!?]{2,}/', $alt_text ) ) {
			$score         += 5;
			$achievements[] = __( 'Appropriate punctuation usage', 'alt-audit' );
		} else {
			$issues[] = __( 'Avoid excessive punctuation', 'alt-audit' );
		}

		return array(
			'score'        => $score,
			'issues'       => $issues,
			'achievements' => $achievements,
		);
	}

	/**
	 * Analyze decorative context
	 *
	 * WCAG-aligned: Uses semantic indicators (CSS classes, file names, context type)
	 * to identify decorative images based on their purpose, not dimensions.
	 *
	 * @param array $context Context information.
	 * @return bool True if context suggests decorative image.
	 */
	private function analyze_decorative_context( $context ) {
		$context_type = strtolower( $context['type'] ?? '' );
		$css_classes  = strtolower( $context['css_classes'] ?? '' );
		$file_name    = strtolower( $context['src'] ?? '' );

		// Context type indicators.
		$decorative_types = array( 'decoration', 'border', 'spacer', 'divider', 'background' );
		if ( in_array( $context_type, $decorative_types, true ) ) {
			return true;
		}

		// CSS class indicators.
		$decorative_classes = array( 'decoration', 'decorative', 'spacer', 'divider', 'border', 'bg-' );
		foreach ( $decorative_classes as $class ) {
			if ( strpos( $css_classes, $class ) !== false ) {
				return true;
			}
		}

		// File name indicators.
		$decorative_file_patterns = array( 'decoration', 'divider', 'spacer', 'border', 'bg_', 'background' );
		foreach ( $decorative_file_patterns as $pattern ) {
			if ( strpos( $file_name, $pattern ) !== false ) {
				return true;
			}
		}

		// WCAG-aligned: Do NOT use size-based detection.
		// WCAG 1.1.1 determines decorative status by purpose/function, not dimensions.

		return false;
	}

	/**
	 * Check if image is used as a link
	 *
	 * @param array $context Context information.
	 * @return bool True if image is in a link.
	 */
	private function is_link_image( $context ) {
		return ! empty( $context['is_link'] ) || ! empty( $context['link_url'] );
	}

	/**
	 * Check link purpose compliance
	 *
	 * @param string $alt_text Alt text to check.
	 * @param array  $context  Link context.
	 * @return bool True if compliant with link purpose guidelines.
	 */
	private function check_link_purpose_compliance( $alt_text, $context ) {
		// Alt text should describe the link purpose, not just the image.
		$purpose_indicators = array( 'go to', 'visit', 'download', 'view', 'open', 'read more' );
		$alt_text_lower     = strtolower( $alt_text );

		foreach ( $purpose_indicators as $indicator ) {
			if ( strpos( $alt_text_lower, $indicator ) !== false ) {
				return true;
			}
		}

		// Check if alt text relates to destination.
		$destination = $context['link_url'] ?? '';
		if ( ! empty( $destination ) ) {
			// Simple check - could be enhanced with more sophisticated analysis.
			return strlen( $alt_text ) > 10; // Assume longer descriptions are more purposeful.
		}

		return strlen( $alt_text ) > 5; // Basic threshold for link purpose.
	}

	/**
	 * Calculate uppercase ratio
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return float Ratio of uppercase to total letters.
	 */
	private function calculate_uppercase_ratio( $alt_text ) {
		$letters = preg_replace( '/[^a-zA-Z]/', '', $alt_text );
		if ( empty( $letters ) ) {
			return 0;
		}

		$uppercase = preg_replace( '/[^A-Z]/', '', $alt_text );
		return strlen( $uppercase ) / strlen( $letters );
	}

	/**
	 * Calculate alphanumeric ratio
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return float Ratio of alphanumeric to total characters.
	 */
	private function calculate_alphanumeric_ratio( $alt_text ) {
		if ( empty( $alt_text ) ) {
			return 0;
		}

		$alphanumeric = preg_replace( '/[^a-zA-Z0-9]/', '', $alt_text );
		return strlen( $alphanumeric ) / strlen( $alt_text );
	}

	/**
	 * Calculate readability score
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return float Readability score (0-100).
	 */
	private function calculate_readability_score( $alt_text ) {
		$word_count     = str_word_count( $alt_text );
		$sentence_count = max( 1, preg_match_all( '/[.!?]+/', $alt_text ) );
		$syllable_count = $this->estimate_syllable_count( $alt_text );

		if ( 0 === $word_count ) {
			return 0;
		}

		// Simplified Flesch Reading Ease formula.
		$avg_sentence_length    = $word_count / $sentence_count;
		$avg_syllables_per_word = $syllable_count / $word_count;

		$score = 206.835 - ( 1.015 * $avg_sentence_length ) - ( 84.6 * $avg_syllables_per_word );

		return max( 0, min( 100, $score ) );
	}

	/**
	 * Calculate clarity score
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return float Clarity score (0-100).
	 */
	private function calculate_clarity_score( $alt_text ) {
		$score = 100;

		// Deduct for overly complex words.
		$complex_words = preg_match_all( '/\b\w{12,}\b/', $alt_text );
		$score        -= $complex_words * 10;

		// Deduct for jargon or technical terms (basic check).
		if ( preg_match( '/\b(API|HTML|CSS|JavaScript|jQuery)\b/i', $alt_text ) ) {
			$score -= 15;
		}

		return max( 0, $score );
	}

	/**
	 * Calculate specificity score
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return float Specificity score (0-100).
	 */
	private function calculate_specificity_score( $alt_text ) {
		// Start with a base score - well-structured text deserves a starting point.
		$word_count = str_word_count( $alt_text );
		$score      = 50; // Base score for having meaningful text.

		// Bonus for having adequate word count.
		if ( $word_count >= 3 ) {
			$score += 15;
		}
		if ( $word_count >= 5 ) {
			$score += 10;
		}

		// Reward specific descriptors.
		$descriptive_patterns = $this->criteria_model->get_descriptive_patterns();
		foreach ( $descriptive_patterns as $pattern_data ) {
			$matches = preg_match_all( $pattern_data['pattern'], $alt_text );
			$score  += $matches * 5; // Reduced from 10 to avoid over-rewarding.
		}

		// Reward numbers and measurements.
		if ( preg_match( '/\b\d+\b/', $alt_text ) ) {
			$score += 10;
		}

		// Penalize vague terms (reduced penalty).
		$vague_terms = array( 'thing', 'stuff', 'something' );
		foreach ( $vague_terms as $term ) {
			if ( stripos( $alt_text, $term ) !== false ) {
				$score -= 10;
			}
		}

		return max( 0, min( 100, $score ) );
	}

	/**
	 * Estimate syllable count for readability calculation
	 *
	 * @param string $text Text to analyze.
	 * @return int Estimated syllable count.
	 */
	private function estimate_syllable_count( $text ) {
		$words           = str_word_count( $text, 1 );
		$total_syllables = 0;

		foreach ( $words as $word ) {
			$word = strtolower( preg_replace( '/[^a-zA-Z]/', '', $word ) );
			if ( empty( $word ) ) {
				continue;
			}

			// Basic syllable counting.
			$vowels    = preg_match_all( '/[aeiouy]/', $word );
			$ending_e  = preg_match( '/e$/', $word ) ? 1 : 0;
			$syllables = max( 1, $vowels - $ending_e );

			$total_syllables += $syllables;
		}

		return max( 1, $total_syllables );
	}
}
