<?php
/**
 * Context Service
 *
 * @package AltAudit
 * @since 1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Context Service for gathering image context
 *
 * Collects relevant context data from WordPress to improve
 * alt text generation accuracy.
 *
 * @since 1.0.0
 */
class Altaudit82ai_Context_Service {


	/**
	 * Get context for an image
	 *
	 * @param int $attachment_id Attachment ID.
	 * @return array Context data.
	 */
	public function get_context( $attachment_id ) {
		$post_id = wp_get_post_parent_id( $attachment_id );

		$context = array(
			'image_caption' => wp_get_attachment_caption( $attachment_id ),
			'image_alt'     => get_post_meta( $attachment_id, '_wp_attachment_image_alt', true ),
			'site_name'     => get_bloginfo( 'name' ),
		);

		if ( $post_id ) {
			$context['page_title'] = get_the_title( $post_id );
			$context['page_url']   = get_permalink( $post_id );

			// SEO Context (Yoast).
			$yoast_kw = get_post_meta( $post_id, '_yoast_wpseo_focuskw', true );
			if ( $yoast_kw ) {
				$context['focus_keyword'] = $yoast_kw;
			}

			// SEO Context (Rank Math).
			$rank_math_kw = get_post_meta( $post_id, 'rank_math_focus_keyword', true );
			if ( $rank_math_kw ) {
				$context['focus_keyword'] = $rank_math_kw;
			}

			// Taxonomy.
			$categories = wp_get_post_categories( $post_id, array( 'fields' => 'names' ) );
			if ( ! empty( $categories ) && ! is_wp_error( $categories ) ) {
				$context['categories'] = $categories;
			}

			$tags = wp_get_post_tags( $post_id, array( 'fields' => 'names' ) );
			if ( ! empty( $tags ) && ! is_wp_error( $tags ) ) {
				$context['tags'] = $tags;
			}

			// Surrounding content (excerpt).
			$post = get_post( $post_id );
			if ( $post ) {
				$context['page_excerpt'] = wp_trim_words( $post->post_content, 50 );
			}
		}

		return array_filter( $context );
	}
}
