<?php
/**
 * API Client Service
 *
 * @package AltAudit
 * @since 1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * API Client for communicating with Alt Audit service
 *
 * Handles all communication with the Alt Audit API including
 * authentication, alt text generation, and error handling.
 *
 * @since 1.0.0
 */
class Altaudit82ai_Api_Client {



	/**
	 * API base URL
	 *
	 * @var string
	 */
	private $api_url;

	/**
	 * API key
	 *
	 * @var string
	 */
	private $api_key;

	/**
	 * Request timeout
	 *
	 * @var int
	 */
	private $timeout;

	/**
	 * Settings service
	 *
	 * @var Altaudit82ai_Settings
	 */
	private $settings;

	/**
	 * Constructor
	 *
	 * @param Altaudit82ai_Settings $settings Settings service.
	 */
	public function __construct( ?Altaudit82ai_Settings $settings = null ) {
		$this->settings = $settings;
		$this->api_url  = ALTAUDIT82AI_API_BASE_URL;
		$this->timeout  = ALTAUDIT82AI_API_TIMEOUT;

		if ( $this->settings ) {
			$this->api_key = $this->settings->get( 'api_key', '' );
		}
	}

	/**
	 * Set API key
	 *
	 * @param string $api_key API key.
	 * @return void
	 */
	public function set_api_key( $api_key ) {
		$this->api_key = sanitize_text_field( $api_key );
	}

	/**
	 * Get API key
	 *
	 * @return string
	 */
	public function get_api_key() {
		return $this->api_key;
	}

	/**
	 * Test API connection
	 *
	 * @return array Response array with success status and message.
	 */
	public function test_connection() {
		if ( empty( $this->api_key ) ) {
			return array(
				'success' => false,
				'message' => __( 'API key is required.', 'alt-audit' ),
				'code'    => 'missing_api_key',
			);
		}

		$response = $this->make_request( 'test-connection', array(), 'POST' );

		if ( is_wp_error( $response ) ) {
			return array(
				'success' => false,
				'message' => $response->get_error_message(),
				'code'    => $response->get_error_code(),
			);
		}

		return array(
			'success' => true,
			'message' => __( 'Connection successful!', 'alt-audit' ),
			'data'    => $response,
		);
	}

	/**
	 * Generate alt text for image
	 *
	 * @param int   $attachment_id Attachment ID.
	 * @param array $options       Generation options.
	 * @return array Response array.
	 */
	public function generate_alt_text( $attachment_id, $options = array() ) {
		if ( empty( $this->api_key ) ) {
			return array(
				'success' => false,
				'message' => __( 'API key is required.', 'alt-audit' ),
				'code'    => 'missing_api_key',
			);
		}

		$default_options = array(
			'style'      => $this->settings ? $this->settings->get( 'style', 'descriptive' ) : 'descriptive',
			'context'    => '',
			'max_length' => $this->settings ? $this->settings->get( 'max_length', 125 ) : 125,
		);

		$options = wp_parse_args( $options, $default_options );

		// Gather context if not provided.
		if ( empty( $options['context'] ) && class_exists( 'Altaudit82ai_Context_Service' ) ) {
			$context_service    = new Altaudit82ai_Context_Service();
			$options['context'] = $context_service->get_context( $attachment_id );
		}

		// Convert max_length to Laravel's 'length' enum.
		$max_length = intval( $options['max_length'] );
		$length     = (string) $max_length; // Default.

		// Determine if image is publicly accessible.
		$image_url = wp_get_attachment_url( $attachment_id );
		$is_public = $this->is_image_publicly_accessible( $image_url );

		if ( $is_public ) {
			// PUBLIC IMAGE: Send URL (more efficient - Google Vision fetches directly).
			$body = $this->prepare_public_image_request( $image_url, $options['style'], $options['context'], $length, $max_length );
		} else {
			// PRIVATE IMAGE: Send base64 data.
			$body = $this->prepare_private_image_request( $attachment_id, $options['style'], $options['context'], $length, $max_length );
			if ( isset( $body['error'] ) ) {
				return array(
					'success' => false,
					'message' => $body['message'],
					'code'    => $body['error'],
				);
			}
		}

		$response = $this->make_request( 'generate', $body, 'POST' );

		if ( is_wp_error( $response ) ) {
			return array(
				'success' => false,
				'message' => $response->get_error_message(),
				'code'    => $response->get_error_code(),
			);
		}

		// Check if Laravel returned an error.
		if ( isset( $response['success'] ) && false === $response['success'] ) {
			return array(
				'success' => false,
				'message' => $response['message'] ?? 'API request failed',
				'error'   => $response['error_code'] ?? 'unknown_error',
			);
		}

		// Extract alt_text from Laravel's response structure.
		$alt_text = '';
		if ( isset( $response['data']['alt_text'] ) ) {
			$alt_text = $response['data']['alt_text'];
		} elseif ( isset( $response['alt_text'] ) ) {
			$alt_text = $response['alt_text'];
		}

		$quality_score = 0;
		if ( isset( $response['data']['quality_score'] ) ) {
			$quality_score = $response['data']['quality_score'];
		} elseif ( isset( $response['quality_score'] ) ) {
			$quality_score = $response['quality_score'];
		}

		return array(
			'success'       => true,
			'alt_text'      => $alt_text,
			'quality_score' => $quality_score,
			'data'          => $response,
		);
	}

	/**
	 * Get user account information
	 *
	 * @return array Response array.
	 */
	public function get_user_info() {
		if ( empty( $this->api_key ) ) {
			return array(
				'success' => false,
				'message' => __( 'API key is required.', 'alt-audit' ),
				'code'    => 'missing_api_key',
			);
		}

		$response = $this->make_request( 'user/info', array(), 'GET' );

		if ( is_wp_error( $response ) ) {
			return array(
				'success' => false,
				'message' => $response->get_error_message(),
				'code'    => $response->get_error_code(),
			);
		}

		return array(
			'success' => true,
			'data'    => $response,
		);
	}

	/**
	 * Get queue status from API (Phase 1B)
	 *
	 * @return array Response array.
	 */
	public function get_queue_status() {
		if ( empty( $this->api_key ) ) {
			return array(
				'success' => false,
				'message' => __( 'API key is required.', 'alt-audit' ),
			);
		}

		$response = $this->make_request( 'queue/status', array(), 'GET' );

		// Handle WP_Error from make_request.
		if ( is_wp_error( $response ) ) {
			return array(
				'success' => false,
				'message' => $response->get_error_message(),
				'code'    => $response->get_error_code(),
			);
		}

		if ( ! isset( $response['success'] ) || ! $response['success'] ) {
			return array(
				'success' => false,
				'message' => $response['message'] ?? __( 'Failed to get queue status.', 'alt-audit' ),
			);
		}

		return array(
			'success' => true,
			'data'    => $response['data'],
		);
	}

	/**
	 * Batch generate alt text for multiple images
	 *
	 * @param array $attachment_ids Array of attachment IDs.
	 * @param array $options        Generation options.
	 * @return array Response array.
	 */
	public function batch_generate( $attachment_ids, $options = array() ) {
		if ( empty( $this->api_key ) ) {
			return array(
				'success' => false,
				'message' => __( 'API key is required.', 'alt-audit' ),
				'code'    => 'missing_api_key',
			);
		}

		$images = array();
		foreach ( $attachment_ids as $attachment_id ) {
			$image_url = wp_get_attachment_url( $attachment_id );
			if ( $image_url ) {
				$images[] = array(
					'attachment_id' => $attachment_id,
					'image_url'     => $image_url,
				);
			}
		}

		if ( empty( $images ) ) {
			return array(
				'success' => false,
				'message' => __( 'No valid images found.', 'alt-audit' ),
				'code'    => 'no_valid_images',
			);
		}

		$default_options = array(
			'style'      => $this->settings ? $this->settings->get( 'style', 'descriptive' ) : 'descriptive',
			'max_length' => $this->settings ? $this->settings->get( 'max_length', 125 ) : 125,
		);

		$options = wp_parse_args( $options, $default_options );

		// Convert max_length to Laravel's 'length' enum.
		$max_length = intval( $options['max_length'] );
		$length     = (string) $max_length; // Default.

		$body = array(
			'images'    => $images,
			'style'     => $options['style'],
			'length'    => $length,
			'max_chars' => $max_length,
			'source'    => 'wordpress-plugin',
		);

		$response = $this->make_request( 'generate/batch', $body, 'POST' );

		if ( is_wp_error( $response ) ) {
			return array(
				'success' => false,
				'message' => $response->get_error_message(),
				'code'    => $response->get_error_code(),
			);
		}

		return array(
			'success' => true,
			'results' => $response['results'] ?? array(),
			'data'    => $response,
		);
	}

	/**
	 * Make HTTP request to API
	 *
	 * @param string $endpoint    API endpoint.
	 * @param array  $body        Request body.
	 * @param string $method      HTTP method.
	 * @param int    $retry_count Current retry attempt.
	 * @return array|WP_Error Response or error.
	 */
	private function make_request( $endpoint, $body = array(), $method = 'POST', $retry_count = 0 ) {
		$url = trailingslashit( $this->api_url ) . $endpoint;

		$headers = array(
			'Content-Type'  => 'application/json',
			'Authorization' => 'Bearer ' . $this->api_key,
			'User-Agent'    => 'AltAudit-WordPress-Plugin/' . ALTAUDIT82AI_VERSION,
		);

		$args = array(
			'method'  => $method,
			'headers' => $headers,
			'timeout' => $this->timeout,
		);

		if ( 'POST' === $method && ! empty( $body ) ) {
			$args['body'] = wp_json_encode( $body );
		} elseif ( 'GET' === $method && ! empty( $body ) ) {
			$url = add_query_arg( $body, $url );
		}

		$response = wp_remote_request( $url, $args );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		$response_code = wp_remote_retrieve_response_code( $response );
		$response_body = wp_remote_retrieve_body( $response );

		// Log request if debug mode is enabled.
		if ( $this->settings && $this->settings->get( 'debug_mode', false ) ) {
			$this->log_request( $endpoint, $args, $response_code, $response_body );
		}

		// Handle rate limiting with retry.
		if ( 429 === $response_code || 500 === $response_code ) {
			$max_retries = 2; // Reduced from 3 for faster processing.

			if ( $retry_count < $max_retries ) {
				// Faster backoff: 1s, 2s (reduced from 2s, 4s, 8s).
				$delay = $retry_count + 1;

				// Check for Retry-After header.
				$retry_after = wp_remote_retrieve_header( $response, 'retry-after' );
				if ( $retry_after && is_numeric( $retry_after ) ) {
					$delay = min( (int) $retry_after, 10 ); // Max 10 seconds.
				}

				// Log retry attempt.
				if ( $this->settings && $this->settings->get( 'debug_mode', false ) ) {
					// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only when debug mode is enabled.
					error_log(
						sprintf(
							'Alt Audit: API error %d. Retrying in %d seconds (attempt %d/%d)',
							$response_code,
							$delay,
							$retry_count + 1,
							$max_retries
						)
					);
				}

				// Wait before retrying.
				sleep( $delay );

				// Retry the request.
				return $this->make_request( $endpoint, $body, $method, $retry_count + 1 );
			}

			// Max retries reached - return detailed error.
			$error_data = json_decode( $response_body, true );
			$error_type = 429 === $response_code ? 'rate_limit' : 'server_error';

			// Build user-friendly error message.
			$user_message = '';
			if ( 429 === $response_code ) {
				$user_message = __( 'Rate limit exceeded. Please wait a few minutes and try again.', 'alt-audit' );
			} elseif ( 500 === $response_code ) {
				$user_message = __( 'The Alt Audit service is temporarily unavailable. Please try again in a few minutes.', 'alt-audit' );
			}

			return new WP_Error(
				$error_type,
				$error_data['message'] ?? $user_message,
				array(
					'status'      => $response_code,
					'error_type'  => $error_type,
					'retry_count' => $retry_count,
					'response'    => $error_data,
				)
			);
		}

		if ( $response_code >= 400 ) {
			$error_data = json_decode( $response_body, true );

			// Determine error type and build user-friendly message.
			$error_type    = 'api_error';
			$error_message = '';

			if ( 401 === $response_code || 403 === $response_code ) {
				$error_type    = 'auth_error';
				$error_message = __( 'Authentication failed. Please check your API key in plugin settings.', 'alt-audit' );
			} elseif ( 404 === $response_code ) {
				$error_type    = 'not_found';
				$error_message = __( 'API endpoint not found. Please update the plugin or check your API URL settings.', 'alt-audit' );
			} elseif ( 422 === $response_code ) {
				$error_type    = 'validation_error';
				$error_message = $error_data['message'] ?? __( 'Invalid request data.', 'alt-audit' );
			} elseif ( $response_code >= 500 ) {
				$error_type    = 'server_error';
				$error_message = __( 'Server error. The Alt Audit service may be experiencing issues. Please try again later.', 'alt-audit' );
			} else {
				$error_message = $error_data['message'] ?? __( 'API request failed.', 'alt-audit' );
			}

			return new WP_Error(
				$error_type,
				$error_message,
				array(
					'status'   => $response_code,
					'response' => $error_data,
				)
			);
		}

		$data = json_decode( $response_body, true );

		if ( null === $data ) {
			return new WP_Error(
				'invalid_response',
				__( 'Invalid API response.', 'alt-audit' )
			);
		}

		return $data;
	}

	/**
	 * Log API request for debugging with comprehensive details
	 *
	 * @param string $endpoint      API endpoint.
	 * @param array  $request_args  Request arguments.
	 * @param int    $response_code Response code.
	 * @param string $response_body Response body.
	 * @param array  $extra_context Optional extra context (image details, etc.).
	 * @return void
	 */
	private function log_request( $endpoint, $request_args, $response_code, $response_body, $extra_context = array() ) {
		// Parse response body for detailed logging.
		$response_data = json_decode( $response_body, true );

		$log_entry = array(
			'timestamp'     => current_time( 'mysql' ),
			'endpoint'      => sanitize_text_field( $endpoint ),
			'method'        => isset( $request_args['method'] ) ? sanitize_text_field( $request_args['method'] ) : 'GET',
			'response_code' => absint( $response_code ),
			'response_size' => strlen( $response_body ),
			'success'       => 200 <= $response_code && 300 > $response_code,
		);

		// Add response details.
		if ( is_array( $response_data ) ) {
			$log_entry['response'] = array(
				'success'    => isset( $response_data['success'] ) ? (bool) $response_data['success'] : null,
				'message'    => isset( $response_data['message'] ) ? sanitize_text_field( $response_data['message'] ) : null,
				'error'      => isset( $response_data['error'] ) ? sanitize_text_field( $response_data['error'] ) : null,
				'error_code' => isset( $response_data['error_code'] ) ? sanitize_key( $response_data['error_code'] ) : null,
			);

			// Include alt_text if present (successful generation).
			if ( isset( $response_data['data']['alt_text'] ) ) {
				$log_entry['response']['alt_text'] = sanitize_text_field( $response_data['data']['alt_text'] );
			} elseif ( isset( $response_data['alt_text'] ) ) {
				$log_entry['response']['alt_text'] = sanitize_text_field( $response_data['alt_text'] );
			}

			// Include quality score if present.
			if ( isset( $response_data['data']['quality_score'] ) ) {
				$log_entry['response']['quality_score'] = absint( $response_data['data']['quality_score'] );
			}

			// Include credits info if present.
			if ( isset( $response_data['data']['credits_remaining'] ) ) {
				$log_entry['response']['credits_remaining'] = absint( $response_data['data']['credits_remaining'] );
			}
		} else {
			// Raw response if not JSON.
			$log_entry['response_raw'] = sanitize_text_field( substr( $response_body, 0, 500 ) );
		}

		// Add extra context (image details, etc.).
		if ( ! empty( $extra_context ) ) {
			$log_entry['context'] = $extra_context;
		}

		// Extract image info from request body.
		if ( isset( $request_args['body'] ) ) {
			$request_body = json_decode( $request_args['body'], true );
			if ( is_array( $request_body ) ) {
				$log_entry['request_details'] = array(
					'style'     => isset( $request_body['style'] ) ? sanitize_key( $request_body['style'] ) : null,
					'length'    => isset( $request_body['length'] ) ? sanitize_key( $request_body['length'] ) : null,
					'max_chars' => isset( $request_body['max_chars'] ) ? absint( $request_body['max_chars'] ) : null,
				);

				if ( isset( $request_body['image_url'] ) ) {
					$log_entry['request_details']['image_url'] = esc_url_raw( $request_body['image_url'] );
				}

				if ( isset( $request_body['image_data'] ) ) {
					// Log base64 size, not content.
					$log_entry['request_details']['image_data_size'] = strlen( $request_body['image_data'] );
					$log_entry['request_details']['image_sent_as']   = 'base64';
				} else {
					$log_entry['request_details']['image_sent_as'] = 'url';
				}
			}
		}

		// Don't log sensitive data.
		if ( isset( $request_args['headers']['Authorization'] ) ) {
			$request_args['headers']['Authorization'] = '[REDACTED]';
		}

		// Log to WordPress debug.log if WP_DEBUG_LOG is enabled.
		if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only when WP_DEBUG_LOG is enabled.
			error_log( 'Alt Audit API: ' . wp_json_encode( $log_entry, JSON_UNESCAPED_SLASHES ) );
		}
	}

	/**
	 * Log detailed image processing info
	 *
	 * @param int    $attachment_id Attachment ID.
	 * @param string $status        Processing status (start, success, error).
	 * @param array  $details       Additional details.
	 * @return void
	 */
	public function log_image_processing( $attachment_id, $status, $details = array() ) {
		if ( ! $this->settings || ! $this->settings->get( 'debug_mode', false ) ) {
			return;
		}

		$attachment_id = absint( $attachment_id );
		$image_path    = get_attached_file( $attachment_id );
		$image_url     = wp_get_attachment_url( $attachment_id );

		$log_entry = array(
			'timestamp'     => current_time( 'mysql' ),
			'attachment_id' => $attachment_id,
			'status'        => sanitize_key( $status ),
			'filename'      => $image_path ? sanitize_file_name( basename( $image_path ) ) : 'unknown',
			'image_url'     => $image_url ? esc_url_raw( $image_url ) : '',
		);

		// Add file size if file exists.
		if ( $image_path && file_exists( $image_path ) ) {
			$file_size                    = filesize( $image_path );
			$log_entry['file_size']       = absint( $file_size );
			$log_entry['file_size_human'] = size_format( $file_size, 2 );

			// Get image dimensions.
			$image_info = wp_getimagesize( $image_path );
			if ( is_array( $image_info ) ) {
				$log_entry['dimensions'] = array(
					'width'  => absint( $image_info[0] ),
					'height' => absint( $image_info[1] ),
				);
				$log_entry['mime_type']  = isset( $image_info['mime'] ) ? sanitize_mime_type( $image_info['mime'] ) : '';
			}
		}

		// Add any extra details.
		if ( ! empty( $details ) ) {
			$log_entry = array_merge( $log_entry, $details );
		}

		if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only when WP_DEBUG_LOG is enabled.
			error_log( 'Alt Audit Image: ' . wp_json_encode( $log_entry, JSON_UNESCAPED_SLASHES ) );
		}
	}

	/**
	 * Validate API key format
	 *
	 * @param string $api_key API key to validate.
	 * @return bool
	 */
	public function validate_api_key_format( $api_key ) {
		// Basic validation - adjust based on actual API key format.
		return ! empty( $api_key ) && strlen( $api_key ) >= 20;
	}

	/**
	 * Check if image is publicly accessible
	 *
	 * @param string $image_url Image URL.
	 * @return bool True if public, false if private.
	 */
	private function is_image_publicly_accessible( $image_url ) {
		// Parse the URL to get the host.
		$parsed_url = wp_parse_url( $image_url );
		if ( empty( $parsed_url['host'] ) ) {
			return false;
		}

		$host = $parsed_url['host'];

		// Check for local/development domains that won't be accessible from the API server.
		$local_patterns = array(
			'.test',       // Laravel Herd, Valet (local development).
			'.local',      // Local development environments.
			'.localhost',  // Localhost variations (testing).
			'.example',    // Example domains (reserved).
			'.invalid',    // Invalid TLD (reserved).
			'localhost',   // Plain localhost (local server).
			'127.0.0.1',   // IPv4 loopback (local).
			'::1',         // IPv6 loopback (local).
		);

		foreach ( $local_patterns as $pattern ) {
			if ( $host === $pattern || str_ends_with( $host, $pattern ) ) {
				// Local URL - must send base64.
				return false;
			}
		}

		// Check for private IP ranges.
		$ip = gethostbyname( $host );
		if ( $ip !== $host ) {
			// Check private IP ranges.
			if ( false === filter_var( $ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE ) ) {
				// Private or reserved IP - must send base64.
				return false;
			}
		}

		// Public URL - can send image_url.
		return true;
	}

	/**
	 * Prepare request body for public image (send URL)
	 *
	 * @param string $image_url Image URL.
	 * @param string $style     Generation style.
	 * @param string $context    Additional context.
	 * @param string $length     Alt text length.
	 * @param int    $max_length Maximum character length.
	 * @return array Request body.
	 */
	private function prepare_public_image_request( $image_url, $style, $context, $length, $max_length ) {
		return array(
			'image_url' => $image_url,
			'style'     => $style,
			'context'   => $context,
			'length'    => $length,
			'max_chars' => $max_length,
			'source'    => 'wordpress-plugin',
		);
	}

	/**
	 * Prepare request body for private image (send base64)
	 *
	 * @param int    $attachment_id Attachment ID.
	 * @param string $style         Generation style.
	 * @param string $context       Additional context.
	 * @param string $length        Alt text length.
	 * @param int    $max_length    Maximum character length.
	 * @return array Request body or error.
	 */
	private function prepare_private_image_request( $attachment_id, $style, $context, $length, $max_length ) {
		// Get image file path.
		$image_path = get_attached_file( $attachment_id );
		if ( ! $image_path || ! file_exists( $image_path ) ) {
			return array(
				'error'   => 'image_not_found',
				'message' => __( 'Image file not found.', 'alt-audit' ),
			);
		}

		// Convert image to base64 for API transmission.
		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents, WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode -- Local file read and base64 encoding required for image data transmission to API.
		$image_data = base64_encode( file_get_contents( $image_path ) );
		if ( ! $image_data ) {
			return array(
				'error'   => 'image_read_error',
				'message' => __( 'Failed to read image file.', 'alt-audit' ),
			);
		}

		// Get the public URL for reference (even though image is sent as base64).
		$image_url = wp_get_attachment_url( $attachment_id );

		return array(
			'image_data' => $image_data,
			// Include URL for reference in SaaS database.
			'image_url'  => $image_url,
			'filename'   => basename( $image_path ),
			'style'      => $style,
			'context'    => $context,
			'length'     => $length,
			'max_chars'  => $max_length,
			'source'     => 'wordpress-plugin',
		);
	}
}
