<?php
/**
 * Length Validator
 *
 * @package AltAudit
 * @since 1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Length Validator class
 *
 * Validates alt text length according to accessibility guidelines
 * and best practices for different contexts.
 *
 * @since 1.0.0
 */
class Altaudit82ai_Length_Validator {

	/**
	 * Length thresholds
	 *
	 * @var array
	 */
	private $thresholds = array(
		'minimum'     => 1,   // Absolute minimum (non-empty).
		'very_short'  => 10,  // Too short threshold.
		'short'       => 25,  // Minimum acceptable.
		'optimal_min' => 40,  // Optimal range start.
		'optimal_max' => 125, // Optimal range end.
		'long'        => 150, // Getting long.
		'very_long'   => 200, // Too long threshold.
		'maximum'     => 250, // Absolute maximum.
	);

	/**
	 * Context-specific thresholds
	 *
	 * @var array
	 */
	private $context_thresholds = array(
		'navigation' => array(
			'optimal_min' => 5,
			'optimal_max' => 50,
			'maximum'     => 80,
		),
		'decorative' => array(
			'optimal_min' => 0,
			'optimal_max' => 0,
			'maximum'     => 10,
		),
		'product'    => array(
			'optimal_min' => 30,
			'optimal_max' => 100,
			'maximum'     => 150,
		),
		'article'    => array(
			'optimal_min' => 40,
			'optimal_max' => 125,
			'maximum'     => 180,
		),
		'complex'    => array(
			'optimal_min' => 60,
			'optimal_max' => 150,
			'maximum'     => 200,
		),
	);

	/**
	 * Validate alt text length
	 *
	 * @param string $alt_text Alt text to validate.
	 * @param array  $context  Context information.
	 * @return array Validation result.
	 */
	public function validate( $alt_text, $context = array() ) {
		$length       = strlen( $alt_text );
		$context_type = $context['type'] ?? 'general';
		$thresholds   = $this->get_context_thresholds( $context_type );

		return array(
			'length'          => $length,
			'is_valid'        => $this->is_valid_length( $length, $thresholds ),
			'category'        => $this->categorize_length( $length, $thresholds ),
			'issues'          => $this->identify_length_issues( $length, $thresholds ),
			'recommendations' => $this->get_length_recommendations( $length, $thresholds, $context_type ),
			'thresholds'      => $thresholds,
			'score'           => $this->calculate_length_score( $length, $thresholds ),
		);
	}

	/**
	 * Validate multiple alt texts
	 *
	 * @param array $alt_texts Array of alt texts to validate.
	 * @param array $context   Common context.
	 * @return array Batch validation results.
	 */
	public function validate_batch( $alt_texts, $context = array() ) {
		$results = array();
		$stats   = array(
			'total'      => count( $alt_texts ),
			'valid'      => 0,
			'too_short'  => 0,
			'too_long'   => 0,
			'optimal'    => 0,
			'avg_length' => 0,
		);

		$total_length = 0;

		foreach ( $alt_texts as $index => $alt_text ) {
			$result            = $this->validate( $alt_text, $context );
			$results[ $index ] = $result;

			// Update stats.
			if ( $result['is_valid'] ) {
				++$stats['valid'];
			}

			switch ( $result['category'] ) {
				case 'optimal':
					++$stats['optimal'];
					break;
				case 'too_short':
				case 'very_short':
					++$stats['too_short'];
					break;
				case 'too_long':
				case 'very_long':
					++$stats['too_long'];
					break;
			}

			$total_length += $result['length'];
		}

		$stats['avg_length'] = $stats['total'] > 0 ? round( $total_length / $stats['total'], 1 ) : 0;

		return array(
			'results' => $results,
			'stats'   => $stats,
		);
	}

	/**
	 * Get context-specific thresholds
	 *
	 * @param string $context_type Context type.
	 * @return array Context thresholds.
	 */
	private function get_context_thresholds( $context_type ) {
		$base_thresholds   = $this->thresholds;
		$context_overrides = $this->context_thresholds[ $context_type ] ?? array();

		return array_merge( $base_thresholds, $context_overrides );
	}

	/**
	 * Check if length is valid
	 *
	 * @param int   $length     Text length.
	 * @param array $thresholds Length thresholds.
	 * @return bool True if valid length.
	 */
	private function is_valid_length( $length, $thresholds ) {
		return $length >= $thresholds['minimum'] && $length <= $thresholds['maximum'];
	}

	/**
	 * Categorize length
	 *
	 * @param int   $length     Text length.
	 * @param array $thresholds Length thresholds.
	 * @return string Length category.
	 */
	private function categorize_length( $length, $thresholds ) {
		if ( 0 === $length ) {
			return 'empty';
		} elseif ( $length < $thresholds['very_short'] ) {
			return 'very_short';
		} elseif ( $length < $thresholds['short'] ) {
			return 'too_short';
		} elseif ( $length >= $thresholds['optimal_min'] && $length <= $thresholds['optimal_max'] ) {
			return 'optimal';
		} elseif ( $length < $thresholds['optimal_min'] ) {
			return 'short';
		} elseif ( $length <= $thresholds['long'] ) {
			return 'acceptable';
		} elseif ( $length <= $thresholds['very_long'] ) {
			return 'too_long';
		} else {
			return 'very_long';
		}
	}

	/**
	 * Identify length issues
	 *
	 * @param int   $length     Text length.
	 * @param array $thresholds Length thresholds.
	 * @return array Length issues.
	 */
	private function identify_length_issues( $length, $thresholds ) {
		$issues = array();

		if ( 0 === $length ) {
			$issues[] = array(
				'type'     => 'empty',
				'severity' => 'critical',
				'message'  => __( 'Alt text is empty', 'alt-audit' ),
				'impact'   => __( 'Screen readers cannot describe the image', 'alt-audit' ),
			);
		} elseif ( $length < $thresholds['very_short'] ) {
			$issues[] = array(
				'type'     => 'very_short',
				'severity' => 'high',
				'message'  => sprintf( /* translators: Placeholder for dynamic content */ __( 'Alt text is very short (%d characters)', 'alt-audit' ), $length ),
				'impact'   => __( 'May not provide sufficient information', 'alt-audit' ),
			);
		} elseif ( $length < $thresholds['short'] ) {
			$issues[] = array(
				'type'     => 'too_short',
				'severity' => 'medium',
				'message'  => sprintf( /* translators: Placeholder for dynamic content */ __( 'Alt text is too short (%d characters)', 'alt-audit' ), $length ),
				'impact'   => __( 'Could benefit from more detail', 'alt-audit' ),
			);
		} elseif ( $length > $thresholds['very_long'] ) {
			$issues[] = array(
				'type'     => 'very_long',
				'severity' => 'high',
				'message'  => sprintf( /* translators: Placeholder for dynamic content */ __( 'Alt text is very long (%d characters)', 'alt-audit' ), $length ),
				'impact'   => __( 'May overwhelm screen reader users', 'alt-audit' ),
			);
		} elseif ( $length > $thresholds['long'] ) {
			$issues[] = array(
				'type'     => 'too_long',
				'severity' => 'medium',
				'message'  => sprintf( /* translators: Placeholder for dynamic content */ __( 'Alt text is too long (%d characters)', 'alt-audit' ), $length ),
				'impact'   => __( 'Consider making it more concise', 'alt-audit' ),
			);
		}

		return $issues;
	}

	/**
	 * Get length recommendations
	 *
	 * @param int    $length       Text length.
	 * @param array  $thresholds   Length thresholds.
	 * @param string $context_type Context type.
	 * @return array Recommendations.
	 */
	private function get_length_recommendations( $length, $thresholds, $context_type ) {
		$recommendations = array();

		if ( 0 === $length ) {
			$recommendations[] = __( 'Add descriptive alt text that explains the image content', 'alt-audit' );
			$recommendations[] = sprintf(
				/* translators: Placeholder for dynamic content */
				__( 'Aim for %1$d-%2$d characters for optimal accessibility', 'alt-audit' ),
				$thresholds['optimal_min'],
				$thresholds['optimal_max']
			);
		} elseif ( $length < $thresholds['short'] ) {
			$recommendations[] = __( 'Add more descriptive details to help users understand the image', 'alt-audit' );
			$recommendations[] = $this->get_context_specific_expansion_tips( $context_type );
		} elseif ( $length < $thresholds['optimal_min'] ) {
			$recommendations[] = sprintf(
				/* translators: Placeholder for dynamic content */
				__( 'Consider adding a bit more detail. Current: %1$d characters, optimal range: %2$d-%3$d', 'alt-audit' ),
				$length,
				$thresholds['optimal_min'],
				$thresholds['optimal_max']
			);
		} elseif ( $length > $thresholds['very_long'] ) {
			$recommendations[] = __( 'Significantly shorten the alt text by focusing on essential information', 'alt-audit' );
			$recommendations[] = __( 'Consider using longdesc attribute for detailed descriptions', 'alt-audit' );
		} elseif ( $length > $thresholds['long'] ) {
			$recommendations[] = __( 'Consider shortening by removing less essential details', 'alt-audit' );
			$recommendations[] = sprintf(
				/* translators: Placeholder for dynamic content */
				__( 'Target %1$d-%2$d characters for optimal screen reader experience', 'alt-audit' ),
				$thresholds['optimal_min'],
				$thresholds['optimal_max']
			);
		}

		// Add context-specific recommendations.
		if ( empty( $recommendations ) ) {
			$recommendations[] = $this->get_context_specific_recommendations( $length, $context_type, $thresholds );
		}

		return array_filter( $recommendations );
	}

	/**
	 * Calculate length score
	 *
	 * @param int   $length     Text length.
	 * @param array $thresholds Length thresholds.
	 * @return int Length score (0-100).
	 */
	private function calculate_length_score( $length, $thresholds ) {
		if ( 0 === $length ) {
			return 0;
		}

		$optimal_min = $thresholds['optimal_min'];
		$optimal_max = $thresholds['optimal_max'];

		// Perfect score for optimal range.
		if ( $length >= $optimal_min && $length <= $optimal_max ) {
			return 100;
		}

		// Calculate score based on distance from optimal range.
		if ( $length < $optimal_min ) {
			// Too short.
			$distance_from_optimal = $optimal_min - $length;
			$max_penalty           = $optimal_min - $thresholds['minimum'];
			$penalty_ratio         = min( 1, $distance_from_optimal / $max_penalty );
			return max( 20, 100 - ( $penalty_ratio * 60 ) );
		} else {
			// Too long.
			$distance_from_optimal = $length - $optimal_max;
			$max_penalty           = $thresholds['maximum'] - $optimal_max;
			$penalty_ratio         = min( 1, $distance_from_optimal / $max_penalty );
			return max( 10, 100 - ( $penalty_ratio * 70 ) );
		}
	}

	/**
	 * Get context-specific expansion tips
	 *
	 * @param string $context_type Context type.
	 * @return string Expansion tips.
	 */
	private function get_context_specific_expansion_tips( $context_type ) {
		$tips = array(
			'product'    => __( 'Include product features, colors, or materials', 'alt-audit' ),
			'article'    => __( 'Explain how the image relates to the article content', 'alt-audit' ),
			'navigation' => __( 'Describe the destination or purpose of the link', 'alt-audit' ),
			'decorative' => __( 'Consider if this image truly needs description', 'alt-audit' ),
			'complex'    => __( 'Break down the complex information into key points', 'alt-audit' ),
		);

		return $tips[ $context_type ] ?? __( 'Add more specific descriptive details', 'alt-audit' );
	}

	/**
	 * Get context-specific recommendations
	 *
	 * @param int    $length       Current length.
	 * @param string $context_type Context type.
	 * @param array  $thresholds   Length thresholds.
	 * @return string Context recommendation.
	 */
	private function get_context_specific_recommendations( $length, $context_type, $thresholds ) {
		$recommendations = array(
			'navigation' => sprintf(
				/* translators: Placeholder for dynamic content */                __( 'Navigation alt text is good at %d characters. Keep it concise and purposeful.', 'alt-audit' ),
				$length
			),
			'decorative' => __( 'For decorative images, consider using empty alt="" instead.', 'alt-audit' ),
			'product'    => sprintf(
				/* translators: Placeholder for dynamic content */                __( 'Product alt text at %1$d characters. Consider adding key features if under %2$d characters.', 'alt-audit' ),
				$length,
				$thresholds['optimal_min']
			),
			'article'    => sprintf(
				/* translators: Placeholder for dynamic content */                __( 'Article image alt text at %d characters. Ensure it complements the content.', 'alt-audit' ),
				$length
			),
			'complex'    => sprintf(
				/* translators: Placeholder for dynamic content */                __( 'Complex image alt text at %d characters. Consider if longdesc is needed for full details.', 'alt-audit' ),
				$length
			),
		);

		return $recommendations[ $context_type ] ?? sprintf(
			/* translators: Placeholder for dynamic content */            __( 'Alt text length of %d characters is within acceptable range.', 'alt-audit' ),
			$length
		);
	}

	/**
	 * Get length guidelines
	 *
	 * @param string $context_type Context type.
	 * @return array Length guidelines.
	 */
	public function get_length_guidelines( $context_type = 'general' ) {
		$thresholds = $this->get_context_thresholds( $context_type );

		return array(
			'context'          => $context_type,
			'optimal_range'    => array(
				'min' => $thresholds['optimal_min'],
				'max' => $thresholds['optimal_max'],
			),
			'acceptable_range' => array(
				'min' => $thresholds['short'],
				'max' => $thresholds['long'],
			),
			'guidelines'       => array(
				sprintf( /* translators: Placeholder for dynamic content */ __( 'Optimal: %1$d-%2$d characters', 'alt-audit' ), $thresholds['optimal_min'], $thresholds['optimal_max'] ),
				sprintf( /* translators: Placeholder for dynamic content */ __( 'Acceptable: %1$d-%2$d characters', 'alt-audit' ), $thresholds['short'], $thresholds['long'] ),
				sprintf( /* translators: Placeholder for dynamic content */ __( 'Maximum: %d characters', 'alt-audit' ), $thresholds['maximum'] ),
			),
			'tips'             => $this->get_length_tips( $context_type ),
		);
	}

	/**
	 * Get length tips for context
	 *
	 * @param string $context_type Context type.
	 * @return array Length tips.
	 */
	private function get_length_tips( $context_type ) {
		$tips = array(
			'general'    => array(
				__( 'Focus on the most important visual information', 'alt-audit' ),
				__( 'Be descriptive but concise', 'alt-audit' ),
				__( 'Consider the context and purpose', 'alt-audit' ),
			),
			'navigation' => array(
				__( 'Keep it short and purposeful', 'alt-audit' ),
				__( 'Describe the destination, not the image', 'alt-audit' ),
				__( 'Use action words when appropriate', 'alt-audit' ),
			),
			'product'    => array(
				__( 'Include key identifying features', 'alt-audit' ),
				__( 'Mention color, brand, or unique characteristics', 'alt-audit' ),
				__( 'Keep marketing language minimal', 'alt-audit' ),
			),
			'article'    => array(
				__( 'Explain relevance to the content', 'alt-audit' ),
				__( 'Include data or key information from charts', 'alt-audit' ),
				__( 'Describe the main subject and context', 'alt-audit' ),
			),
		);

		return $tips[ $context_type ] ?? $tips['general'];
	}

	/**
	 * Check if length is in optimal range
	 *
	 * @param string $alt_text     Alt text to check.
	 * @param string $context_type Context type.
	 * @return bool True if in optimal range.
	 */
	public function is_optimal_length( $alt_text, $context_type = 'general' ) {
		$length     = strlen( $alt_text );
		$thresholds = $this->get_context_thresholds( $context_type );

		return $length >= $thresholds['optimal_min'] && $length <= $thresholds['optimal_max'];
	}

	/**
	 * Get length category description
	 *
	 * @param string $category Length category.
	 * @return string Category description.
	 */
	public function get_category_description( $category ) {
		$descriptions = array(
			'empty'      => __( 'No alt text provided', 'alt-audit' ),
			'very_short' => __( 'Very short - needs more detail', 'alt-audit' ),
			'too_short'  => __( 'Too short - could use more description', 'alt-audit' ),
			'short'      => __( 'Short but acceptable', 'alt-audit' ),
			'optimal'    => __( 'Optimal length for accessibility', 'alt-audit' ),
			'acceptable' => __( 'Acceptable length', 'alt-audit' ),
			'too_long'   => __( 'Too long - consider shortening', 'alt-audit' ),
			'very_long'  => __( 'Very long - needs significant shortening', 'alt-audit' ),
		);

		return $descriptions[ $category ] ?? __( 'Unknown length category', 'alt-audit' );
	}
}
