<?php
/**
 * Weak Alt Text Checker
 *
 * @package AltAudit
 * @since 1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Weak Alt Text Checker class
 *
 * WCAG 2.1 compliant: Handles assessment of alt text that is present but contains
 * quality issues. Focuses on WCAG 2.1 Level A (1.1.1 Non-text Content) requirements.
 *
 * @since 1.0.0
 */
class Altaudit82ai_Weak_Checker {

	/**
	 * Weak quality indicators
	 *
	 * WCAG 2.1 compliant: No minimum length/word requirements.
	 * Focus on actual quality issues like redundancy, vagueness, and structure.
	 *
	 * @var array
	 */
	private $weak_indicators = array(
		'too_long'  => 200, // Practical limit - screen readers may clip at 200-300 chars.
		'max_words' => 30,  // Practical limit for conciseness.
	);

	/**
	 * Problematic patterns
	 *
	 * WCAG 2.1 compliant: Identifies actual quality issues, not valid contextual terms.
	 *
	 * @var array
	 */
	private $problematic_patterns = array(
		'redundant_prefixes' => array(
			'image of',
			'picture of',
			'photo of',
			'graphic of',
			'screenshot of',
			'image shows',
			'picture shows',
			'photo shows',
		),
		'vague_terms'        => array(
			'thing',
			'stuff',
			'something',
			'here',
			'this',
			'that',
			'click here',
			'see below',
			'check this out',
		),
		'file_extensions'    => array(
			'.jpg',
			'.jpeg',
			'.png',
			'.gif',
			'.webp',
			'.svg',
			'.bmp',
		),
	);

	/**
	 * Check if this checker matches the alt text
	 *
	 * @param string $alt_text Alt text to check.
	 * @param array  $context  Additional context.
	 * @return bool True if matches this quality level.
	 */
	public function matches( $alt_text, $context = array() ) {
		if ( empty( trim( $alt_text ) ) ) {
			return false; // Missing alt text is handled by Missing checker.
		}

		$issues = $this->identify_issues( $alt_text );
		$score  = $this->calculate_preliminary_score( $alt_text );

		// Weak if has significant issues or low score.
		return ( count( $issues ) >= 2 ) || ( $score >= 20 && $score <= 60 );
	}

	/**
	 * Assess the quality of weak alt text
	 *
	 * @param string $alt_text Alt text to assess.
	 * @param array  $context  Additional context.
	 * @return array Assessment result.
	 */
	public function assess( $alt_text, $context = array() ) {
		$issues      = $this->identify_issues( $alt_text );
		$suggestions = $this->generate_suggestions( $alt_text, $issues );
		$score       = $this->calculate_detailed_score( $alt_text, $issues );

		return array(
			'score'           => $score,
			'level'           => 'weak',
			'priority'        => 'medium',
			'issues'          => $issues,
			'suggestions'     => $suggestions,
			'improvements'    => $this->get_specific_improvements( $alt_text, $issues ),
			'examples'        => $this->get_improvement_examples( $alt_text, $context ),
			'action_required' => true,
			'auto_fixable'    => $this->is_auto_fixable( $issues ),
			'quick_fixes'     => $this->get_quick_fixes( $alt_text, $issues ),
		);
	}

	/**
	 * Identify issues in alt text
	 *
	 * WCAG 2.1 compliant: Identifies actual quality issues without enforcing
	 * minimum length/word count requirements. Focuses on problematic patterns
	 * and structural issues that genuinely impair accessibility.
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return array Identified issues.
	 */
	private function identify_issues( $alt_text ) {
		$issues     = array();
		$length     = strlen( $alt_text );
		$word_count = str_word_count( $alt_text );

		// WCAG 2.1 compliant: Only flag if too long (practical screen reader limit).
		// No minimum length check - "Home", "Logo", "Search" are valid.
		if ( $length > $this->weak_indicators['too_long'] ) {
			$issues[] = __( 'Alt text exceeds 200 characters - screen readers may clip content', 'alt-audit' );
		}

		// WCAG 2.1 compliant: Only flag if too wordy (practical limit).
		// No minimum word check - single words like "Logo" are valid.
		if ( $word_count > $this->weak_indicators['max_words'] ) {
			$issues[] = __( 'Alt text is too wordy and should be more concise', 'alt-audit' );
		}

		// Pattern-based issues (redundancy, vagueness, file extensions).
		$issues = array_merge( $issues, $this->check_problematic_patterns( $alt_text ) );

		// Structure issues (capitalization, excessive punctuation).
		$issues = array_merge( $issues, $this->check_structure_issues( $alt_text ) );

		// WCAG 2.1 compliant: Removed descriptive words requirement.
		// Functional images like "Search" don't need descriptive words.

		return $issues;
	}

	/**
	 * Check for problematic patterns
	 *
	 * WCAG 2.1 compliant: Identifies genuine quality issues without flagging
	 * context-appropriate terms like "Logo", "Menu", "Home".
	 *
	 * @param string $alt_text Alt text to check.
	 * @return array Issues found.
	 */
	private function check_problematic_patterns( $alt_text ) {
		$issues         = array();
		$alt_text_lower = strtolower( $alt_text );

		// Redundant prefixes (genuinely problematic).
		foreach ( $this->problematic_patterns['redundant_prefixes'] as $prefix ) {
			if ( strpos( $alt_text_lower, strtolower( $prefix ) ) === 0 ) {
				/* translators: %s: The redundant phrase found in the alt text */
				$issues[] = sprintf( __( 'Remove redundant phrase "%s"', 'alt-audit' ), $prefix );
				break;
			}
		}

		// Vague terms (genuinely unhelpful).
		foreach ( $this->problematic_patterns['vague_terms'] as $term ) {
			if ( strpos( $alt_text_lower, strtolower( $term ) ) !== false ) {
				/* translators: %s: The vague term found in the alt text */
				$issues[] = sprintf( __( 'Replace vague term "%s" with specific description', 'alt-audit' ), $term );
			}
		}

		// File extensions (always problematic).
		foreach ( $this->problematic_patterns['file_extensions'] as $ext ) {
			if ( strpos( $alt_text_lower, $ext ) !== false ) {
				$issues[] = __( 'Remove file extension from alt text', 'alt-audit' );
				break;
			}
		}

		// WCAG 2.1 compliant: Removed generic-only check.
		// Terms like "Logo", "Menu", "Home", "Search" are valid and context-appropriate.

		return $issues;
	}

	/**
	 * Check for structure issues
	 *
	 * @param string $alt_text Alt text to check.
	 * @return array Structure issues found.
	 */
	private function check_structure_issues( $alt_text ) {
		$issues = array();

		// Capitalization.
		if ( ! preg_match( '/^[A-Z]/', $alt_text ) ) {
			$issues[] = __( 'Alt text should start with a capital letter', 'alt-audit' );
		}

		// Excessive punctuation.
		if ( preg_match( '/[.!?]{2,}/', $alt_text ) ) {
			$issues[] = __( 'Remove excessive punctuation', 'alt-audit' );
		}

		// ALL CAPS.
		if ( preg_match( '/^[A-Z\s]+$/', $alt_text ) && strlen( $alt_text ) > 10 ) {
			$issues[] = __( 'Avoid writing in ALL CAPS', 'alt-audit' );
		}

		return $issues;
	}


	/**
	 * Count descriptive words
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return int Number of descriptive words.
	 */
	private function count_descriptive_words( $alt_text ) {
		$descriptive_patterns = array(
			'/\b(red|blue|green|yellow|orange|purple|black|white|gray|grey|brown|pink)\b/i',
			'/\b(large|small|big|tiny|huge|massive|little|medium|tall|short|wide|narrow)\b/i',
			'/\b(round|square|circular|rectangular|triangular|oval|curved|straight)\b/i',
			'/\b(bright|dark|light|shadowy|sunny|cloudy|clear|blurry|sharp|soft)\b/i',
			'/\b(happy|sad|smiling|frowning|serious|excited|calm|angry|peaceful)\b/i',
		);

		$count = 0;
		foreach ( $descriptive_patterns as $pattern ) {
			$count += preg_match_all( $pattern, $alt_text );
		}

		return $count;
	}

	/**
	 * Calculate preliminary score
	 *
	 * @param string $alt_text Alt text to score.
	 * @return int Score (0-100).
	 */
	private function calculate_preliminary_score( $alt_text ) {
		$score  = 50; // Start with medium score.
		$issues = $this->identify_issues( $alt_text );

		// Deduct for each issue.
		$score -= count( $issues ) * 8;

		// Basic length bonus.
		$length = strlen( $alt_text );
		if ( $length >= 25 && $length <= 100 ) {
			$score += 15;
		}

		// Descriptive words bonus.
		$descriptive_count = $this->count_descriptive_words( $alt_text );
		$score            += $descriptive_count * 5;

		return max( 10, min( 85, $score ) );
	}

	/**
	 * Calculate detailed score
	 *
	 * WCAG 2.1 compliant: Applies penalties only for genuine quality issues,
	 * not for valid contextual alt text.
	 *
	 * @param string $alt_text Alt text to score.
	 * @param array  $issues   Identified issues.
	 * @return int Detailed score (0-100).
	 */
	private function calculate_detailed_score( $alt_text, $issues ) {
		$base_score = $this->calculate_preliminary_score( $alt_text );

		// Penalty for genuine quality issues only.
		$critical_penalties = array(
			'exceeds 200'      => 10,  // Too long for screen readers.
			'too wordy'        => 8,   // Excessive word count.
			'redundant phrase' => 8,   // Unnecessary prefixes.
			'vague term'       => 6,   // Unhelpful descriptions.
			'file extension'   => 10,  // Technical artifacts.
		);

		foreach ( $issues as $issue ) {
			foreach ( $critical_penalties as $pattern => $penalty ) {
				if ( stripos( $issue, $pattern ) !== false ) {
					$base_score -= $penalty;
					break;
				}
			}
		}

		return max( 15, min( 60, $base_score ) ); // Weak range: 15-60.
	}

	/**
	 * Generate improvement suggestions
	 *
	 * WCAG 2.1 compliant: Provides suggestions for genuine quality issues
	 * without enforcing arbitrary length or descriptiveness requirements.
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @param array  $issues   Identified issues.
	 * @return array Suggestions for improvement.
	 */
	private function generate_suggestions( $alt_text, $issues ) {
		$suggestions = array();

		// Issue-specific suggestions.
		foreach ( $issues as $issue ) {
			if ( stripos( $issue, 'exceeds 200' ) !== false || stripos( $issue, 'too long' ) !== false ) {
				$suggestions[] = __( 'Focus on the most important visual information', 'alt-audit' );
			} elseif ( stripos( $issue, 'redundant' ) !== false ) {
				$suggestions[] = __( 'Start directly with the description, skip "image of"', 'alt-audit' );
			} elseif ( stripos( $issue, 'vague' ) !== false ) {
				$suggestions[] = __( 'Replace vague terms with specific descriptions', 'alt-audit' );
			} elseif ( stripos( $issue, 'file extension' ) !== false ) {
				$suggestions[] = __( 'Remove technical file extensions from alt text', 'alt-audit' );
			}
		}

		// General improvement suggestions only if we have issues.
		if ( ! empty( $issues ) && empty( $suggestions ) ) {
			$suggestions[] = __( 'Review alt text for clarity and accessibility', 'alt-audit' );
		}

		return array_unique( $suggestions );
	}

	/**
	 * Get specific improvements
	 *
	 * @param string $alt_text Alt text to improve.
	 * @param array  $issues   Identified issues.
	 * @return array Specific improvement actions.
	 */
	private function get_specific_improvements( $alt_text, $issues ) {
		$improvements = array();

		// Check for auto-fixable issues.
		foreach ( $this->problematic_patterns['redundant_prefixes'] as $prefix ) {
			if ( stripos( $alt_text, $prefix ) === 0 ) {
				$improved = trim( substr( $alt_text, strlen( $prefix ) ) );
				if ( ! empty( $improved ) ) {
					$improvements[] = array(
						'type'        => 'remove_prefix',
						'current'     => $alt_text,
						'improved'    => ucfirst( $improved ),
						/* translators: %s: The redundant prefix to be removed from alt text */
						'description' => sprintf( __( 'Remove "%s" prefix', 'alt-audit' ), $prefix ),
					);
				}
			}
		}

		return $improvements;
	}

	/**
	 * Get improvement examples
	 *
	 * @param string $alt_text Alt text to improve.
	 * @param array  $context  Image context.
	 * @return array Example improvements.
	 */
	private function get_improvement_examples( $alt_text, $context ) {
		$examples     = array();
		$context_type = $context['type'] ?? 'general';

		// Context-specific examples.
		switch ( $context_type ) {
			case 'product':
				$examples = array(
					__( 'Blue wireless headphones with noise canceling feature', 'alt-audit' ),
					__( 'Red leather handbag with gold chain strap', 'alt-audit' ),
				);
				break;

			case 'article':
				$examples = array(
					/* translators: Placeholder for dynamic content */
					__( 'Graph showing 25% increase in sales over 6 months', 'alt-audit' ),
					__( 'Team of five people collaborating around a conference table', 'alt-audit' ),
				);
				break;

			default:
				$examples = array(
					__( 'Person typing on laptop in bright modern office', 'alt-audit' ),
					__( 'Golden retriever playing with tennis ball in park', 'alt-audit' ),
				);
		}

		return $examples;
	}

	/**
	 * Check if issues are auto-fixable
	 *
	 * @param array $issues Identified issues.
	 * @return bool True if auto-fixable.
	 */
	private function is_auto_fixable( $issues ) {
		$auto_fixable_patterns = array( 'redundant phrase', 'file extension', 'capital letter' );

		foreach ( $issues as $issue ) {
			foreach ( $auto_fixable_patterns as $pattern ) {
				if ( stripos( $issue, $pattern ) !== false ) {
					return true;
				}
			}
		}

		return false;
	}

	/**
	 * Get quick fixes
	 *
	 * @param string $alt_text Alt text to fix.
	 * @param array  $issues   Issues to fix.
	 * @return array Quick fix suggestions.
	 */
	private function get_quick_fixes( $alt_text, $issues ) {
		$fixes = array();

		// Remove redundant prefixes.
		foreach ( $this->problematic_patterns['redundant_prefixes'] as $prefix ) {
			if ( stripos( $alt_text, $prefix ) === 0 ) {
				$fixed = trim( substr( $alt_text, strlen( $prefix ) ) );
				if ( ! empty( $fixed ) ) {
					$fixes[] = array(
						'action'  => 'remove_prefix',
						'current' => $alt_text,
						'fixed'   => ucfirst( $fixed ),
					);
				}
			}
		}

		// Remove file extensions.
		foreach ( $this->problematic_patterns['file_extensions'] as $ext ) {
			if ( stripos( $alt_text, $ext ) !== false ) {
				$fixes[] = array(
					'action'  => 'remove_extension',
					'current' => $alt_text,
					'fixed'   => str_ireplace( $ext, '', $alt_text ),
				);
			}
		}

		// Fix capitalization.
		if ( ! preg_match( '/^[A-Z]/', $alt_text ) ) {
			$fixes[] = array(
				'action'  => 'capitalize',
				'current' => $alt_text,
				'fixed'   => ucfirst( $alt_text ),
			);
		}

		return $fixes;
	}
}
