<?php
/**
 * Missing Alt Text Checker
 *
 * @package AltAudit
 * @since 1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Missing Alt Text Checker class
 *
 * Handles assessment of images that are missing alt text entirely.
 * This represents the most critical accessibility issue.
 *
 * @since 1.0.0
 */
class Altaudit82ai_Missing_Checker {

	/**
	 * Check if this checker matches the alt text
	 *
	 * @param string $alt_text Alt text to check.
	 * @param array  $context  Additional context.
	 * @return bool True if matches this quality level.
	 */
	public function matches( $alt_text, $context = array() ) {
		return empty( trim( $alt_text ) );
	}

	/**
	 * Assess the quality of missing alt text
	 *
	 * @param string $alt_text Alt text to assess (empty).
	 * @param array  $context  Additional context.
	 * @return array Assessment result.
	 */
	public function assess( $alt_text, $context = array() ) {
		return array(
			'score'           => 0,
			'level'           => 'critical',
			'priority'        => 'high',
			'issues'          => array(
				__( 'Alt text is completely missing', 'alt-audit' ),
				__( 'This creates a significant accessibility barrier', 'alt-audit' ),
				__( 'Screen readers cannot describe this image to users', 'alt-audit' ),
			),
			'suggestions'     => array(
				__( 'Add descriptive alt text that explains what the image shows', 'alt-audit' ),
				__( 'Focus on the most important visual information', 'alt-audit' ),
				__( 'Keep it concise but informative (50-125 characters recommended)', 'alt-audit' ),
				__( 'Consider the context in which the image appears', 'alt-audit' ),
			),
			'wcag_violations' => array(
				'1.1.1' => __( 'Non-text Content (Level A)', 'alt-audit' ),
			),
			'action_required' => true,
			'auto_fixable'    => false,
			'examples'        => $this->get_examples( $context ),
		);
	}

	/**
	 * Get examples of good alt text for missing images
	 *
	 * @param array $context Image context.
	 * @return array Examples.
	 */
	public function get_examples( $context = array() ) {
		$context_type = $context['type'] ?? 'general';

		$examples = array(
			'general'    => array(
				__( 'Person typing on a laptop in a coffee shop', 'alt-audit' ),
				__( 'Red sports car parked in front of a modern building', 'alt-audit' ),
				__( 'Graph showing website traffic increase over 6 months', 'alt-audit' ),
			),
			'product'    => array(
				__( 'Blue wireless headphones with noise canceling feature', 'alt-audit' ),
				__( 'Stainless steel kitchen mixer with multiple attachments', 'alt-audit' ),
				__( 'Black leather wallet with multiple card slots', 'alt-audit' ),
			),
			'article'    => array(
				__( 'Team meeting around a conference table discussing project plans', 'alt-audit' ),
				__( 'Close-up of hands writing notes during a lecture', 'alt-audit' ),
				__( 'City skyline at sunset with orange and purple sky', 'alt-audit' ),
			),
			'navigation' => array(
				__( 'Home', 'alt-audit' ),
				__( 'Menu', 'alt-audit' ),
				__( 'Search', 'alt-audit' ),
			),
		);

		return $examples[ $context_type ] ?? $examples['general'];
	}

	/**
	 * Get severity level
	 *
	 * @return string Severity level.
	 */
	public function get_severity() {
		return 'critical';
	}

	/**
	 * Get WCAG impact
	 *
	 * @return array WCAG impact details.
	 */
	public function get_wcag_impact() {
		return array(
			'level'          => 'A',
			'guideline'      => '1.1.1',
			'title'          => __( 'Non-text Content', 'alt-audit' ),
			'impact'         => __( 'Critical - Prevents screen reader users from understanding image content', 'alt-audit' ),
			'users_affected' => array(
				__( 'Blind users using screen readers', 'alt-audit' ),
				__( 'Users with low vision using screen readers', 'alt-audit' ),
				__( 'Users in low bandwidth situations', 'alt-audit' ),
			),
		);
	}

	/**
	 * Get repair suggestions
	 *
	 * @param array $context Image context.
	 * @return array Specific repair suggestions.
	 */
	public function get_repair_suggestions( $context = array() ) {
		$suggestions = array(
			__( 'Identify the main purpose of the image', 'alt-audit' ),
			__( 'Describe the essential visual information', 'alt-audit' ),
			__( 'Keep the description concise and focused', 'alt-audit' ),
		);

		// Add context-specific suggestions.
		$context_type = $context['type'] ?? 'general';

		switch ( $context_type ) {
			case 'product':
				$suggestions[] = __( 'Include key product features like color, size, or material', 'alt-audit' );
				$suggestions[] = __( 'Mention brand name if visible and relevant', 'alt-audit' );
				break;

			case 'navigation':
				$suggestions[] = __( 'Use the link destination or purpose as alt text', 'alt-audit' );
				$suggestions[] = __( 'Keep it short - navigation should be concise', 'alt-audit' );
				break;

			case 'article':
				$suggestions[] = __( 'Explain how the image relates to the article content', 'alt-audit' );
				$suggestions[] = __( 'Include data or key information if it\'s an infographic', 'alt-audit' );
				break;

			case 'decorative':
				$suggestions[] = __( 'Consider if this image is purely decorative', 'alt-audit' );
				$suggestions[] = __( 'If decorative, use empty alt="" instead of missing alt', 'alt-audit' );
				break;
		}

		return $suggestions;
	}
}
