<?php
/**
 * Good Alt Text Checker
 *
 * @package AltAudit
 * @since 1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Good Alt Text Checker class
 *
 * Handles assessment of alt text that meets good accessibility
 * standards with solid description and proper structure.
 *
 * @since 1.0.0
 */
class Altaudit82ai_Good_Checker {

	/**
	 * Good quality criteria
	 *
	 * @var array
	 */
	private $good_criteria = array(
		'min_length'      => 25,  // Minimum character count.
		'max_length'      => 150, // Maximum character count.
		'min_words'       => 4,   // Minimum word count.
		'max_words'       => 20,  // Maximum word count.
		'min_descriptive' => 2,   // Minimum descriptive words.
		'structure_score' => 15,  // Minimum structure score.
	);

	/**
	 * Quality indicators
	 *
	 * @var array
	 */
	private $quality_indicators = array(
		'good_descriptors' => array(
			'colors'    => array( 'red', 'blue', 'green', 'yellow', 'orange', 'purple', 'black', 'white', 'gray' ),
			'sizes'     => array( 'large', 'small', 'big', 'tiny', 'huge', 'medium', 'tall', 'short', 'wide' ),
			'emotions'  => array( 'happy', 'sad', 'smiling', 'serious', 'excited', 'calm', 'focused' ),
			'materials' => array( 'wooden', 'metal', 'glass', 'plastic', 'fabric', 'leather', 'stone' ),
			'actions'   => array( 'running', 'sitting', 'standing', 'walking', 'working', 'playing' ),
		),
		'contextual_words' => array(
			'location'      => array( 'in', 'on', 'at', 'near', 'beside', 'behind', 'front', 'inside', 'outside' ),
			'relationships' => array( 'with', 'and', 'between', 'among', 'surrounded by', 'next to' ),
		),
	);

	/**
	 * Check if this checker matches the alt text
	 *
	 * @param string $alt_text Alt text to check.
	 * @param array  $context  Additional context (unused, for interface compatibility).
	 * @return bool True if matches this quality level.
	 *
	 * phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.FoundAfterLastUsed
	 */
	public function matches( $alt_text, $context = array() ) {
		if ( empty( trim( $alt_text ) ) ) {
			return false;
		}

		$score = $this->calculate_good_score( $alt_text );
		return $score >= 61 && $score <= 85;
	}

	/**
	 * Assess the quality of good alt text
	 *
	 * @param string $alt_text Alt text to assess.
	 * @param array  $context  Additional context (unused, for interface compatibility).
	 * @return array Assessment result.
	 *
	 * phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.FoundAfterLastUsed
	 */
	public function assess( $alt_text, $context = array() ) {
		$score              = $this->calculate_good_score( $alt_text );
		$strengths          = $this->identify_strengths( $alt_text );
		$minor_improvements = $this->identify_minor_improvements( $alt_text );

		return array(
			'score'              => $score,
			'level'              => 'good',
			'priority'           => 'maintain',
			'strengths'          => $strengths,
			'minor_improvements' => $minor_improvements,
			'achievements'       => $this->get_achievements( $alt_text ),
			'suggestions'        => $this->get_optimization_suggestions( $alt_text ),
			'action_required'    => false,
			'auto_fixable'       => false,
			'quality_factors'    => $this->analyze_quality_factors( $alt_text ),
		);
	}

	/**
	 * Calculate good quality score
	 *
	 * @param string $alt_text Alt text to score.
	 * @return int Score from 0-100.
	 */
	private function calculate_good_score( $alt_text ) {
		$score      = 0;
		$length     = strlen( $alt_text );
		$word_count = str_word_count( $alt_text );

		// Length scoring (optimal: 25-125 characters).
		if ( $length >= 40 && $length <= 100 ) {
			$score += 25;
		} elseif ( $length >= 25 && $length <= 150 ) {
			$score += 20;
		} elseif ( $length > 0 ) {
			$score += 10;
		}

		// Word count scoring (optimal: 6-15 words).
		if ( $word_count >= 6 && $word_count <= 12 ) {
			$score += 20;
		} elseif ( $word_count >= 4 && $word_count <= 20 ) {
			$score += 15;
		}

		// Descriptiveness scoring.
		$descriptive_score = $this->assess_descriptiveness( $alt_text );
		$score            += $descriptive_score; // Up to 20 points.
		// Structure scoring.
		$structure_score = $this->assess_structure( $alt_text );
		$score          += $structure_score; // Up to 20 points.
		// Context and clarity scoring.
		$clarity_score = $this->assess_clarity( $alt_text );
		$score        += $clarity_score; // Up to 15 points.
		return min( $score, 100 );
	}

	/**
	 * Assess descriptiveness of alt text
	 *
	 * @param string $alt_text Alt text to assess.
	 * @return int Descriptiveness score (0-20).
	 */
	private function assess_descriptiveness( $alt_text ) {
		$score            = 0;
		$found_categories = 0;

		foreach ( $this->quality_indicators['good_descriptors'] as $category => $words ) {
			foreach ( $words as $word ) {
				if ( stripos( $alt_text, $word ) !== false ) {
					++$found_categories;
					break; // Only count each category once.
				}
			}
		}

		// Score based on variety of descriptive categories.
		if ( $found_categories >= 3 ) {
			$score = 20;
		} elseif ( $found_categories >= 2 ) {
			$score = 15;
		} elseif ( $found_categories >= 1 ) {
			$score = 10;
		}

		return $score;
	}

	/**
	 * Assess text structure
	 *
	 * @param string $alt_text Alt text to assess.
	 * @return int Structure score (0-20).
	 */
	private function assess_structure( $alt_text ) {
		$score = 0;

		// Proper capitalization.
		if ( preg_match( '/^[A-Z]/', $alt_text ) ) {
			$score += 4;
		}

		// No file extensions.
		if ( ! preg_match( '/\.(jpg|jpeg|png|gif|webp|svg)$/i', $alt_text ) ) {
			$score += 4;
		}

		// No redundant phrases.
		$redundant_phrases = array( 'image of', 'picture of', 'photo of' );
		$has_redundant     = false;
		foreach ( $redundant_phrases as $phrase ) {
			if ( stripos( $alt_text, $phrase ) !== false ) {
				$has_redundant = true;
				break;
			}
		}
		if ( ! $has_redundant ) {
			$score += 4;
		}

		// Proper punctuation for longer descriptions.
		if ( strlen( $alt_text ) > 50 ) {
			if ( preg_match( '/[.!?]$/', $alt_text ) ) {
				$score += 4;
			}
		} else {
			$score += 4; // Short descriptions don't need periods.
		}

		// No excessive punctuation.
		if ( ! preg_match( '/[.!?]{2,}/', $alt_text ) ) {
			$score += 4;
		}

		return $score;
	}

	/**
	 * Assess clarity and readability
	 *
	 * @param string $alt_text Alt text to assess.
	 * @return int Clarity score (0-15).
	 */
	private function assess_clarity( $alt_text ) {
		$score = 0;

		// Clear, specific language.
		$vague_terms = array( 'thing', 'stuff', 'something', 'here', 'this', 'that' );
		$has_vague   = false;
		foreach ( $vague_terms as $term ) {
			if ( stripos( $alt_text, $term ) !== false ) {
				$has_vague = true;
				break;
			}
		}
		if ( ! $has_vague ) {
			$score += 5;
		}

		// Contextual relationships.
		$has_context = false;
		foreach ( $this->quality_indicators['contextual_words'] as $category => $words ) {
			foreach ( $words as $word ) {
				if ( stripos( $alt_text, $word ) !== false ) {
					$has_context = true;
					break 2;
				}
			}
		}
		if ( $has_context ) {
			$score += 5;
		}

		// Logical flow and completeness.
		if ( $this->has_logical_flow( $alt_text ) ) {
			$score += 5;
		}

		return $score;
	}

	/**
	 * Check if alt text has logical flow
	 *
	 * @param string $alt_text Alt text to check.
	 * @return bool True if has logical flow.
	 */
	private function has_logical_flow( $alt_text ) {
		// Basic check for logical sentence structure.
		$words = str_word_count( $alt_text, 1 );

		// Should have subject and action/description.
		$has_subject               = false;
		$has_action_or_description = false;

		$subjects     = array( 'person', 'man', 'woman', 'child', 'people', 'group', 'team' );
		$actions      = array( 'sitting', 'standing', 'walking', 'working', 'holding', 'wearing' );
		$descriptions = array( 'showing', 'displaying', 'featuring', 'with', 'in' );

		foreach ( $words as $word ) {
			$word_lower = strtolower( $word );

			if ( in_array( $word_lower, $subjects, true ) ) {
				$has_subject = true;
			}

			if ( in_array( $word_lower, array_merge( $actions, $descriptions ), true ) ) {
				$has_action_or_description = true;
			}
		}

		return $has_subject && $has_action_or_description;
	}

	/**
	 * Identify strengths in the alt text
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return array Identified strengths.
	 */
	private function identify_strengths( $alt_text ) {
		$strengths  = array();
		$length     = strlen( $alt_text );
		$word_count = str_word_count( $alt_text );

		// Length strengths.
		if ( $length >= 40 && $length <= 100 ) {
			$strengths[] = __( 'Optimal length for accessibility', 'alt-audit' );
		}

		// Structure strengths.
		if ( preg_match( '/^[A-Z]/', $alt_text ) ) {
			$strengths[] = __( 'Proper capitalization', 'alt-audit' );
		}

		if ( ! preg_match( '/\b(image|picture|photo) of\b/i', $alt_text ) ) {
			$strengths[] = __( 'No redundant phrases', 'alt-audit' );
		}

		// Descriptiveness strengths.
		$descriptive_categories = $this->count_descriptive_categories( $alt_text );
		if ( $descriptive_categories >= 2 ) {
			$strengths[] = __( 'Good variety of descriptive details', 'alt-audit' );
		}

		// Context strengths.
		if ( $this->has_contextual_information( $alt_text ) ) {
			$strengths[] = __( 'Provides contextual information', 'alt-audit' );
		}

		return $strengths;
	}

	/**
	 * Identify minor improvements
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return array Minor improvement suggestions.
	 */
	private function identify_minor_improvements( $alt_text ) {
		$improvements = array();
		$length       = strlen( $alt_text );

		// Length improvements.
		if ( $length < 40 ) {
			$improvements[] = __( 'Could include slightly more detail', 'alt-audit' );
		} elseif ( $length > 125 ) {
			$improvements[] = __( 'Could be slightly more concise', 'alt-audit' );
		}

		// Descriptiveness improvements.
		$descriptive_categories = $this->count_descriptive_categories( $alt_text );
		if ( $descriptive_categories < 2 ) {
			$improvements[] = __( 'Could include more descriptive details', 'alt-audit' );
		}

		// Context improvements.
		if ( ! $this->has_contextual_information( $alt_text ) ) {
			$improvements[] = __( 'Could better explain the image context', 'alt-audit' );
		}

		return $improvements;
	}

	/**
	 * Get achievements for good alt text
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return array Achievements.
	 */
	private function get_achievements( $alt_text ) {
		$achievements = array();

		$achievements[] = __( 'Meets accessibility standards', 'alt-audit' );
		$achievements[] = __( 'Provides useful description', 'alt-audit' );

		if ( strlen( $alt_text ) >= 40 && strlen( $alt_text ) <= 100 ) {
			$achievements[] = __( 'Optimal length for screen readers', 'alt-audit' );
		}

		if ( $this->count_descriptive_categories( $alt_text ) >= 2 ) {
			$achievements[] = __( 'Rich descriptive content', 'alt-audit' );
		}

		return $achievements;
	}

	/**
	 * Get optimization suggestions for excellent quality
	 *
	 * @param string $alt_text Alt text to optimize.
	 * @return array Optimization suggestions.
	 */
	private function get_optimization_suggestions( $alt_text ) {
		$suggestions = array();

		// Suggest moving to excellent.
		$descriptive_categories = $this->count_descriptive_categories( $alt_text );
		if ( $descriptive_categories < 3 ) {
			$suggestions[] = __( 'To reach excellent quality, add more varied descriptive details', 'alt-audit' );
		}

		if ( ! $this->has_emotional_context( $alt_text ) ) {
			$suggestions[] = __( 'Consider adding emotional or atmospheric context', 'alt-audit' );
		}

		if ( strlen( $alt_text ) < 50 ) {
			$suggestions[] = __( 'Could expand slightly with more specific details', 'alt-audit' );
		}

		return $suggestions;
	}

	/**
	 * Analyze quality factors
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return array Quality factor breakdown.
	 */
	private function analyze_quality_factors( $alt_text ) {
		return array(
			'length'                 => strlen( $alt_text ),
			'word_count'             => str_word_count( $alt_text ),
			'descriptive_categories' => $this->count_descriptive_categories( $alt_text ),
			'has_context'            => $this->has_contextual_information( $alt_text ),
			'structure_score'        => $this->assess_structure( $alt_text ),
			'clarity_indicators'     => $this->get_clarity_indicators( $alt_text ),
		);
	}

	/**
	 * Count descriptive categories found
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return int Number of descriptive categories.
	 */
	private function count_descriptive_categories( $alt_text ) {
		$found_categories = 0;

		foreach ( $this->quality_indicators['good_descriptors'] as $category => $words ) {
			foreach ( $words as $word ) {
				if ( stripos( $alt_text, $word ) !== false ) {
					++$found_categories;
					break; // Only count each category once.
				}
			}
		}

		return $found_categories;
	}

	/**
	 * Check if alt text has contextual information
	 *
	 * @param string $alt_text Alt text to check.
	 * @return bool True if has contextual information.
	 */
	private function has_contextual_information( $alt_text ) {
		foreach ( $this->quality_indicators['contextual_words'] as $category => $words ) {
			foreach ( $words as $word ) {
				if ( stripos( $alt_text, $word ) !== false ) {
					return true;
				}
			}
		}

		return false;
	}

	/**
	 * Check if alt text has emotional context
	 *
	 * @param string $alt_text Alt text to check.
	 * @return bool True if has emotional context.
	 */
	private function has_emotional_context( $alt_text ) {
		$emotional_words = $this->quality_indicators['good_descriptors']['emotions'];

		foreach ( $emotional_words as $word ) {
			if ( stripos( $alt_text, $word ) !== false ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Get clarity indicators
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return array Clarity indicators.
	 */
	private function get_clarity_indicators( $alt_text ) {
		return array(
			'no_vague_terms'       => ! $this->has_vague_terms( $alt_text ),
			'has_specific_details' => $this->count_descriptive_categories( $alt_text ) >= 2,
			'logical_structure'    => $this->has_logical_flow( $alt_text ),
			'appropriate_length'   => strlen( $alt_text ) >= 25 && strlen( $alt_text ) <= 150,
		);
	}

	/**
	 * Check for vague terms
	 *
	 * @param string $alt_text Alt text to check.
	 * @return bool True if has vague terms.
	 */
	private function has_vague_terms( $alt_text ) {
		$vague_terms = array( 'thing', 'stuff', 'something', 'here', 'this', 'that' );

		foreach ( $vague_terms as $term ) {
			if ( stripos( $alt_text, $term ) !== false ) {
				return true;
			}
		}

		return false;
	}
}
