<?php
/**
 * Excellent Alt Text Checker
 *
 * @package AltAudit
 * @since 1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Excellent Alt Text Checker class
 *
 * Handles assessment of alt text that meets the highest accessibility
 * standards with exceptional description, context, and user experience.
 *
 * @since 1.0.0
 */
class Altaudit82ai_Excellent_Checker {

	/**
	 * Excellence criteria
	 *
	 * @var array
	 */
	private $excellence_criteria = array(
		'min_length'         => 40,  // Minimum character count.
		'optimal_length_min' => 50,  // Optimal minimum.
		'optimal_length_max' => 125, // Optimal maximum.
		'max_length'         => 150, // Maximum acceptable.
		'min_words'          => 6,   // Minimum word count.
		'optimal_words_min'  => 8,   // Optimal minimum words.
		'optimal_words_max'  => 15,  // Optimal maximum words.
		'min_descriptive'    => 3,   // Minimum descriptive categories.
		'min_context_score'  => 85,  // Minimum context relevance.
	);

	/**
	 * Excellence indicators
	 *
	 * @var array
	 */
	private $excellence_indicators = array(
		'advanced_descriptors'  => array(
			'emotions'        => array( 'joyful', 'contemplative', 'determined', 'serene', 'confident', 'focused', 'vibrant' ),
			'atmospherics'    => array( 'bustling', 'tranquil', 'dramatic', 'intimate', 'spacious', 'cozy', 'elegant' ),
			'textures'        => array( 'smooth', 'rough', 'glossy', 'matte', 'textured', 'polished', 'weathered' ),
			'lighting'        => array( 'bright', 'dim', 'natural', 'warm', 'cool', 'backlit', 'shadowy', 'golden' ),
			'compositions'    => array( 'centered', 'balanced', 'dynamic', 'symmetrical', 'layered', 'framed' ),
			'advanced_colors' => array( 'crimson', 'azure', 'emerald', 'amber', 'violet', 'turquoise', 'burgundy' ),
		),
		'contextual_excellence' => array(
			'purpose_clarity'    => array( 'demonstrating', 'illustrating', 'highlighting', 'showcasing', 'exemplifying' ),
			'relationships'      => array( 'surrounded by', 'contrasting with', 'complementing', 'harmonizing with' ),
			'narrative_elements' => array( 'capturing', 'revealing', 'depicting', 'conveying', 'expressing' ),
		),
	);

	/**
	 * Check if this checker matches the alt text
	 *
	 * @param string $alt_text Alt text to check.
	 * @param array  $context  Additional context.
	 * @return bool True if matches this quality level.
	 */
	public function matches( $alt_text, $context = array() ) {
		if ( empty( trim( $alt_text ) ) ) {
			return false;
		}

		$score = $this->calculate_excellence_score( $alt_text );
		return $score >= 86;
	}

	/**
	 * Assess the quality of excellent alt text
	 *
	 * @param string $alt_text Alt text to assess.
	 * @param array  $context  Additional context.
	 * @return array Assessment result.
	 */
	public function assess( $alt_text, $context = array() ) {
		$score              = $this->calculate_excellence_score( $alt_text );
		$excellence_factors = $this->identify_excellence_factors( $alt_text );
		$mastery_elements   = $this->identify_mastery_elements( $alt_text );

		return array(
			'score'                => $score,
			'level'                => 'excellent',
			'priority'             => 'maintain',
			'excellence_factors'   => $excellence_factors,
			'mastery_elements'     => $mastery_elements,
			'achievements'         => $this->identify_achievements( $alt_text ),
			'suggestions'          => $this->get_perfection_suggestions( $alt_text ),
			'user_experience'      => $this->assess_user_experience( $alt_text ),
			'accessibility_impact' => $this->assess_accessibility_impact( $alt_text ),
			'action_required'      => false,
			'auto_fixable'         => false,
			'exemplary_qualities'  => $this->get_exemplary_qualities( $alt_text ),
		);
	}

	/**
	 * Calculate excellence score
	 *
	 * @param string $alt_text Alt text to score.
	 * @return int Score from 0-100.
	 */
	private function calculate_excellence_score( $alt_text ) {
		$score      = 0;
		$length     = strlen( $alt_text );
		$word_count = str_word_count( $alt_text );

		// Perfect length scoring (50-125 characters is optimal).
		if ( $length >= $this->excellence_criteria['optimal_length_min'] &&
			$length <= $this->excellence_criteria['optimal_length_max'] ) {
			$score += 25;
		} elseif ( $length >= $this->excellence_criteria['min_length'] &&
					$length <= $this->excellence_criteria['max_length'] ) {
			$score += 20;
		}

		// Optimal word count (8-15 words).
		if ( $word_count >= $this->excellence_criteria['optimal_words_min'] &&
			$word_count <= $this->excellence_criteria['optimal_words_max'] ) {
			$score += 20;
		} elseif ( $word_count >= $this->excellence_criteria['min_words'] ) {
			$score += 15;
		}

		// Advanced descriptiveness.
		$descriptive_score = $this->assess_advanced_descriptiveness( $alt_text );
		$score            += $descriptive_score; // Up to 25 points.
		// Perfect structure and grammar.
		$structure_score = $this->assess_perfect_structure( $alt_text );
		$score          += $structure_score; // Up to 15 points.
		// Contextual mastery.
		$context_score = $this->assess_contextual_mastery( $alt_text );
		$score        += $context_score; // Up to 15 points.
		return min( $score, 100 );
	}

	/**
	 * Assess advanced descriptiveness
	 *
	 * @param string $alt_text Alt text to assess.
	 * @return int Descriptiveness score (0-25).
	 */
	private function assess_advanced_descriptiveness( $alt_text ) {
		$score          = 0;
		$found_advanced = 0;
		$variety_bonus  = 0;

		foreach ( $this->excellence_indicators['advanced_descriptors'] as $category => $words ) {
			$category_found = false;
			foreach ( $words as $word ) {
				if ( stripos( $alt_text, $word ) !== false ) {
					++$found_advanced;
					if ( ! $category_found ) {
						++$variety_bonus;
						$category_found = true;
					}
				}
			}
		}

		// Score based on number and variety of advanced descriptors.
		if ( $found_advanced >= 3 && $variety_bonus >= 3 ) {
			$score = 25;
		} elseif ( $found_advanced >= 2 && $variety_bonus >= 2 ) {
			$score = 20;
		} elseif ( $found_advanced >= 1 ) {
			$score = 15;
		}

		// Bonus for exceptional variety.
		if ( $variety_bonus >= 4 ) {
			$score += 3;
		}

		return min( $score, 25 );
	}

	/**
	 * Assess perfect structure
	 *
	 * @param string $alt_text Alt text to assess.
	 * @return int Structure score (0-15).
	 */
	private function assess_perfect_structure( $alt_text ) {
		$score = 0;

		// All basic structure requirements.
		if ( preg_match( '/^[A-Z]/', $alt_text ) ) {
			$score += 3;
		}

		if ( ! preg_match( '/\.(jpg|jpeg|png|gif|webp|svg)$/i', $alt_text ) ) {
			$score += 3;
		}

		if ( ! preg_match( '/\b(image|picture|photo) of\b/i', $alt_text ) ) {
			$score += 3;
		}

		// Advanced structure elements.
		if ( $this->has_sophisticated_grammar( $alt_text ) ) {
			$score += 3;
		}

		if ( $this->has_natural_flow( $alt_text ) ) {
			$score += 3;
		}

		return $score;
	}

	/**
	 * Assess contextual mastery
	 *
	 * @param string $alt_text Alt text to assess.
	 * @return int Context score (0-15).
	 */
	private function assess_contextual_mastery( $alt_text ) {
		$score               = 0;
		$contextual_elements = 0;

		foreach ( $this->excellence_indicators['contextual_excellence'] as $category => $words ) {
			foreach ( $words as $word ) {
				if ( stripos( $alt_text, $word ) !== false ) {
					++$contextual_elements;
					break; // Count each category once.
				}
			}
		}

		// Score based on contextual sophistication.
		if ( $contextual_elements >= 2 ) {
			$score = 15;
		} elseif ( $contextual_elements >= 1 ) {
			$score = 10;
		}

		// Bonus for narrative quality.
		if ( $this->has_narrative_quality( $alt_text ) ) {
			$score += 3;
		}

		return min( $score, 15 );
	}

	/**
	 * Check for sophisticated grammar
	 *
	 * @param string $alt_text Alt text to check.
	 * @return bool True if has sophisticated grammar.
	 */
	private function has_sophisticated_grammar( $alt_text ) {
		// Check for complex sentence structures.
		$complex_patterns = array(
			'/\b(while|during|showing|displaying|featuring)\b/i',
			'/\b(with|wearing|holding|surrounded by)\b/i',
			'/,\s+\w+/i', // Comma-separated descriptors.
		);

		foreach ( $complex_patterns as $pattern ) {
			if ( preg_match( $pattern, $alt_text ) ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Check for natural flow
	 *
	 * @param string $alt_text Alt text to check.
	 * @return bool True if has natural flow.
	 */
	private function has_natural_flow( $alt_text ) {
		// Check for logical progression and readability.
		$words = str_word_count( $alt_text, 1 );

		// Should read naturally without forced structure.
		$transition_words = array( 'with', 'in', 'on', 'while', 'during', 'showing', 'wearing' );
		$has_transitions  = false;

		foreach ( $words as $word ) {
			if ( in_array( strtolower( $word ), $transition_words, true ) ) {
				$has_transitions = true;
				break;
			}
		}

		return $has_transitions && str_word_count( $alt_text ) >= 8;
	}

	/**
	 * Check for narrative quality
	 *
	 * @param string $alt_text Alt text to check.
	 * @return bool True if has narrative quality.
	 */
	private function has_narrative_quality( $alt_text ) {
		$narrative_indicators = $this->excellence_indicators['contextual_excellence']['narrative_elements'];

		foreach ( $narrative_indicators as $indicator ) {
			if ( stripos( $alt_text, $indicator ) !== false ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Identify excellence factors
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return array Excellence factors found.
	 */
	private function identify_excellence_factors( $alt_text ) {
		$factors = array();

		// Length excellence.
		$length = strlen( $alt_text );
		if ( $length >= 50 && $length <= 125 ) {
			$factors[] = __( 'Optimal length for comprehensive description', 'alt-audit' );
		}

		// Descriptive excellence.
		$advanced_count = $this->count_advanced_descriptors( $alt_text );
		if ( $advanced_count >= 3 ) {
			$factors[] = __( 'Rich variety of advanced descriptive elements', 'alt-audit' );
		}

		// Structural excellence.
		if ( $this->has_sophisticated_grammar( $alt_text ) ) {
			$factors[] = __( 'Sophisticated grammatical structure', 'alt-audit' );
		}

		// Contextual excellence.
		if ( $this->has_narrative_quality( $alt_text ) ) {
			$factors[] = __( 'Narrative quality that engages users', 'alt-audit' );
		}

		return $factors;
	}

	/**
	 * Identify mastery elements
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return array Mastery elements identified.
	 */
	private function identify_mastery_elements( $alt_text ) {
		$elements = array();

		if ( $this->has_emotional_intelligence( $alt_text ) ) {
			$elements[] = __( 'Emotional intelligence in description', 'alt-audit' );
		}

		if ( $this->has_technical_precision( $alt_text ) ) {
			$elements[] = __( 'Technical precision without jargon', 'alt-audit' );
		}

		if ( $this->has_inclusive_language( $alt_text ) ) {
			$elements[] = __( 'Inclusive and accessible language', 'alt-audit' );
		}

		if ( $this->has_purposeful_detail( $alt_text ) ) {
			$elements[] = __( 'Every detail serves a purpose', 'alt-audit' );
		}

		return $elements;
	}

	/**
	 * Identify achievements
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return array Achievements unlocked.
	 */
	private function identify_achievements( $alt_text ) {
		$achievements = array(
			__( 'Accessibility Champion - Exceeds WCAG AAA standards', 'alt-audit' ),
			__( 'Master Descriptor - Rich, varied descriptive language', 'alt-audit' ),
			__( 'User Experience Expert - Optimal for screen reader users', 'alt-audit' ),
		);

		if ( $this->has_artistic_quality( $alt_text ) ) {
			$achievements[] = __( 'Artistic Vision - Captures aesthetic and emotional elements', 'alt-audit' );
		}

		if ( $this->count_advanced_descriptors( $alt_text ) >= 4 ) {
			$achievements[] = __( 'Descriptive Virtuoso - Exceptional variety of details', 'alt-audit' );
		}

		return $achievements;
	}

	/**
	 * Get perfection suggestions
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return array Suggestions for perfection.
	 */
	private function get_perfection_suggestions( $alt_text ) {
		$suggestions = array();

		// Only suggest if there's room for improvement to perfect score.
		$score = $this->calculate_excellence_score( $alt_text );

		if ( $score < 95 ) {
			if ( ! $this->has_emotional_intelligence( $alt_text ) ) {
				$suggestions[] = __( 'Consider adding subtle emotional or atmospheric context', 'alt-audit' );
			}

			if ( $this->count_advanced_descriptors( $alt_text ) < 4 ) {
				$suggestions[] = __( 'Could include additional advanced descriptive elements', 'alt-audit' );
			}

			if ( ! $this->has_narrative_quality( $alt_text ) ) {
				$suggestions[] = __( 'Consider adding narrative elements for engagement', 'alt-audit' );
			}
		} else {
			$suggestions[] = __( 'This alt text represents exceptional accessibility practice', 'alt-audit' );
		}

		return $suggestions;
	}

	/**
	 * Assess user experience impact
	 *
	 * @param string $alt_text Alt text to assess.
	 * @return array User experience assessment.
	 */
	private function assess_user_experience( $alt_text ) {
		return array(
			'screen_reader_friendly' => strlen( $alt_text ) <= 125,
			'cognitive_load'         => $this->assess_cognitive_load( $alt_text ),
			'engagement_level'       => $this->assess_engagement_level( $alt_text ),
			'comprehension_ease'     => $this->assess_comprehension_ease( $alt_text ),
		);
	}

	/**
	 * Assess accessibility impact
	 *
	 * @param string $alt_text Alt text to assess.
	 * @return array Accessibility impact assessment.
	 */
	private function assess_accessibility_impact( $alt_text ) {
		return array(
			'wcag_level'       => 'AAA',
			'barrier_removal'  => 'Complete',
			'inclusion_score'  => $this->calculate_inclusion_score( $alt_text ),
			'universal_design' => $this->assess_universal_design( $alt_text ),
		);
	}

	/**
	 * Get exemplary qualities
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return array Exemplary qualities identified.
	 */
	private function get_exemplary_qualities( $alt_text ) {
		$qualities = array();

		if ( $this->has_precision_without_verbosity( $alt_text ) ) {
			$qualities[] = __( 'Precise yet concise description', 'alt-audit' );
		}

		if ( $this->has_contextual_awareness( $alt_text ) ) {
			$qualities[] = __( 'Strong contextual awareness', 'alt-audit' );
		}

		if ( $this->has_emotional_intelligence( $alt_text ) ) {
			$qualities[] = __( 'Emotional intelligence in description', 'alt-audit' );
		}

		return $qualities;
	}

	/**
	 * Count advanced descriptors
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return int Number of advanced descriptors.
	 */
	private function count_advanced_descriptors( $alt_text ) {
		$count = 0;

		foreach ( $this->excellence_indicators['advanced_descriptors'] as $category => $words ) {
			foreach ( $words as $word ) {
				if ( stripos( $alt_text, $word ) !== false ) {
					++$count;
				}
			}
		}

		return $count;
	}

	/**
	 * Check for emotional intelligence
	 *
	 * @param string $alt_text Alt text to check.
	 * @return bool True if has emotional intelligence.
	 */
	private function has_emotional_intelligence( $alt_text ) {
		$emotional_words   = $this->excellence_indicators['advanced_descriptors']['emotions'];
		$atmospheric_words = $this->excellence_indicators['advanced_descriptors']['atmospherics'];

		$all_emotional = array_merge( $emotional_words, $atmospheric_words );

		foreach ( $all_emotional as $word ) {
			if ( stripos( $alt_text, $word ) !== false ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Check for technical precision
	 *
	 * @param string $alt_text Alt text to check.
	 * @return bool True if has technical precision.
	 */
	private function has_technical_precision( $alt_text ) {
		// Check for specific, technical details without jargon.
		return preg_match( '/\b\d+\b/', $alt_text ) && ! preg_match( '/\b(API|HTML|CSS|JavaScript)\b/i', $alt_text );
	}

	/**
	 * Check for inclusive language
	 *
	 * @param string $alt_text Alt text to check.
	 * @return bool True if uses inclusive language.
	 */
	private function has_inclusive_language( $alt_text ) {
		// Basic check - could be expanded with more sophisticated analysis.
		$exclusive_terms = array( 'normal', 'weird', 'strange', 'abnormal' );

		foreach ( $exclusive_terms as $term ) {
			if ( stripos( $alt_text, $term ) !== false ) {
				return false;
			}
		}

		return true;
	}

	/**
	 * Check for purposeful detail
	 *
	 * @param string $alt_text Alt text to check.
	 * @return bool True if every detail is purposeful.
	 */
	private function has_purposeful_detail( $alt_text ) {
		// Check that the description is focused and not rambling.
		$word_count        = str_word_count( $alt_text );
		$descriptive_count = $this->count_advanced_descriptors( $alt_text );

		// Good ratio of descriptive to total words.
		return $word_count <= 20 && $descriptive_count >= 2;
	}

	/**
	 * Check for artistic quality
	 *
	 * @param string $alt_text Alt text to check.
	 * @return bool True if has artistic quality.
	 */
	private function has_artistic_quality( $alt_text ) {
		$artistic_indicators = array_merge(
			$this->excellence_indicators['advanced_descriptors']['lighting'],
			$this->excellence_indicators['advanced_descriptors']['compositions'],
			$this->excellence_indicators['advanced_descriptors']['atmospherics']
		);

		$found_artistic = 0;
		foreach ( $artistic_indicators as $indicator ) {
			if ( stripos( $alt_text, $indicator ) !== false ) {
				++$found_artistic;
			}
		}

		return $found_artistic >= 2;
	}

	/**
	 * Assess cognitive load
	 *
	 * @param string $alt_text Alt text to assess.
	 * @return string Cognitive load level.
	 */
	private function assess_cognitive_load( $alt_text ) {
		$word_count    = str_word_count( $alt_text );
		$complex_words = preg_match_all( '/\b\w{8,}\b/', $alt_text );

		if ( $word_count <= 15 && $complex_words <= 2 ) {
			return 'Low - Easy to process';
		} elseif ( $word_count <= 20 && $complex_words <= 4 ) {
			return 'Moderate - Well-balanced';
		} else {
			return 'High - May be challenging';
		}
	}

	/**
	 * Assess engagement level
	 *
	 * @param string $alt_text Alt text to assess.
	 * @return string Engagement level.
	 */
	private function assess_engagement_level( $alt_text ) {
		$has_emotion         = $this->has_emotional_intelligence( $alt_text );
		$has_narrative       = $this->has_narrative_quality( $alt_text );
		$descriptive_variety = $this->count_advanced_descriptors( $alt_text );

		if ( $has_emotion && $has_narrative && $descriptive_variety >= 3 ) {
			return 'Highly engaging';
		} elseif ( ( $has_emotion || $has_narrative ) && $descriptive_variety >= 2 ) {
			return 'Moderately engaging';
		} else {
			return 'Functional';
		}
	}

	/**
	 * Assess comprehension ease
	 *
	 * @param string $alt_text Alt text to assess.
	 * @return string Comprehension ease level.
	 */
	private function assess_comprehension_ease( $alt_text ) {
		$length     = strlen( $alt_text );
		$word_count = str_word_count( $alt_text );
		$has_flow   = $this->has_natural_flow( $alt_text );

		if ( $length <= 100 && $word_count <= 15 && $has_flow ) {
			return 'Very easy';
		} elseif ( $length <= 125 && $word_count <= 18 ) {
			return 'Easy';
		} else {
			return 'Moderate';
		}
	}

	/**
	 * Calculate inclusion score
	 *
	 * @param string $alt_text Alt text to assess.
	 * @return int Inclusion score (0-100).
	 */
	private function calculate_inclusion_score( $alt_text ) {
		$score = 85; // Start high for excellent level.
		if ( $this->has_inclusive_language( $alt_text ) ) {
			$score += 5;
		}

		if ( $this->has_purposeful_detail( $alt_text ) ) {
			$score += 5;
		}

		if ( strlen( $alt_text ) <= 125 ) {
			$score += 5;
		}

		return min( $score, 100 );
	}

	/**
	 * Assess universal design principles
	 *
	 * @param string $alt_text Alt text to assess.
	 * @return string Universal design assessment.
	 */
	private function assess_universal_design( $alt_text ) {
		$principles_met = 0;

		// Equitable use.
		if ( $this->has_inclusive_language( $alt_text ) ) {
			++$principles_met;
		}

		// Flexibility in use.
		if ( strlen( $alt_text ) >= 40 && strlen( $alt_text ) <= 125 ) {
			++$principles_met;
		}

		// Simple and intuitive use.
		if ( $this->has_natural_flow( $alt_text ) ) {
			++$principles_met;
		}

		// Perceptible information.
		if ( $this->count_advanced_descriptors( $alt_text ) >= 3 ) {
			++$principles_met;
		}

		if ( $principles_met >= 3 ) {
			return 'Excellent adherence to universal design';
		} elseif ( $principles_met >= 2 ) {
			return 'Good universal design principles';
		} else {
			return 'Basic universal design compliance';
		}
	}

	/**
	 * Check for precision without verbosity
	 *
	 * @param string $alt_text Alt text to check.
	 * @return bool True if precise but not verbose.
	 */
	private function has_precision_without_verbosity( $alt_text ) {
		$word_count        = str_word_count( $alt_text );
		$descriptive_count = $this->count_advanced_descriptors( $alt_text );

		// High descriptive density without excessive length.
		return $word_count <= 18 && $descriptive_count >= 3;
	}

	/**
	 * Check for contextual awareness
	 *
	 * @param string $alt_text Alt text to check.
	 * @return bool True if shows contextual awareness.
	 */
	private function has_contextual_awareness( $alt_text ) {
		$contextual_words = array_merge(
			$this->excellence_indicators['contextual_excellence']['purpose_clarity'],
			$this->excellence_indicators['contextual_excellence']['relationships']
		);

		foreach ( $contextual_words as $word ) {
			if ( stripos( $alt_text, $word ) !== false ) {
				return true;
			}
		}

		return false;
	}
}
