<?php
/**
 * Decorative Alt Text Checker
 *
 * @package AltAudit
 * @since 1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Decorative Alt Text Checker class
 *
 * Handles assessment of images that appear to be decorative
 * and may not need descriptive alt text.
 *
 * WCAG-aligned: Determines decorative status based on purpose and function
 * (per WCAG 1.1.1), not image dimensions. Uses context clues like CSS classes,
 * file names, and alt text content to identify decorative images.
 *
 * @since 1.0.0
 */
class Altaudit82ai_Decorative_Checker {

	/**
	 * Decorative indicators
	 *
	 * WCAG-compliant indicators based on purpose/function, not size or length.
	 *
	 * @var array
	 */
	private $decorative_indicators = array(
		'empty_alt'     => true,
		'generic_terms' => array( 'decoration', 'divider', 'spacer', 'border', 'line', 'dot', 'separator' ),
	);

	/**
	 * Check if this checker matches the alt text
	 *
	 * @param string $alt_text Alt text to check.
	 * @param array  $context  Additional context.
	 * @return bool True if matches this quality level.
	 */
	public function matches( $alt_text, $context = array() ) {
		// Empty alt text may be intentionally decorative.
		if ( empty( trim( $alt_text ) ) ) {
			return $this->analyze_decorative_context( $context );
		}

		// Check for decorative indicators.
		return $this->has_decorative_indicators( $alt_text, $context );
	}

	/**
	 * Assess the quality of decorative alt text
	 *
	 * @param string $alt_text Alt text to assess.
	 * @param array  $context  Additional context.
	 * @return array Assessment result.
	 */
	public function assess( $alt_text, $context = array() ) {
		$is_empty = empty( trim( $alt_text ) );

		if ( $is_empty ) {
			return $this->assess_empty_decorative( $context );
		}

		return $this->assess_described_decorative( $alt_text, $context );
	}

	/**
	 * Assess empty decorative alt text
	 *
	 * @param array $context Image context.
	 * @return array Assessment result.
	 */
	private function assess_empty_decorative( $context ) {
		$context_suggests_decorative = $this->analyze_decorative_context( $context );

		if ( $context_suggests_decorative ) {
			return array(
				'score'           => 85,
				'level'           => 'appropriate',
				'priority'        => 'maintain',
				'issues'          => array(),
				'suggestions'     => array(
					__( 'Good - decorative images should have empty alt text', 'alt-audit' ),
					__( 'Consider adding role="presentation" for better semantics', 'alt-audit' ),
				),
				'achievements'    => array(
					__( 'Correctly marked as decorative', 'alt-audit' ),
					__( 'Follows WCAG best practices for decorative images', 'alt-audit' ),
				),
				'action_required' => false,
				'auto_fixable'    => false,
			);
		}

		return array(
			'score'           => 15,
			'level'           => 'questionable',
			'priority'        => 'review',
			'issues'          => array(
				__( 'Empty alt text - verify if image is truly decorative', 'alt-audit' ),
				__( 'May need descriptive alt text if image conveys information', 'alt-audit' ),
			),
			'suggestions'     => array(
				__( 'Review if this image serves a functional purpose', 'alt-audit' ),
				__( 'Add descriptive alt text if image is informative', 'alt-audit' ),
				__( 'Keep empty alt if image is purely decorative', 'alt-audit' ),
			),
			'action_required' => true,
			'auto_fixable'    => false,
		);
	}

	/**
	 * Assess decorative image with description
	 *
	 * @param string $alt_text Alt text to assess.
	 * @param array  $context  Image context.
	 * @return array Assessment result.
	 */
	private function assess_described_decorative( $alt_text, $context ) {
		$decorative_score = $this->calculate_decorative_appropriateness( $alt_text, $context );

		if ( $decorative_score > 70 ) {
			return array(
				'score'           => 60,
				'level'           => 'adequate',
				'priority'        => 'review',
				'issues'          => array(
					__( 'Image appears decorative but has descriptive alt text', 'alt-audit' ),
				),
				'suggestions'     => array(
					__( 'Consider if this image truly needs description', 'alt-audit' ),
					__( 'Use empty alt="" if image is purely decorative', 'alt-audit' ),
					__( 'Keep description if image provides meaningful context', 'alt-audit' ),
				),
				'action_required' => false,
				'auto_fixable'    => false,
			);
		}

		return array(
			'score'           => 40,
			'level'           => 'unclear',
			'priority'        => 'review',
			'issues'          => array(
				__( 'Unclear if image is decorative or informative', 'alt-audit' ),
				__( 'Alt text may be too generic for informative content', 'alt-audit' ),
			),
			'suggestions'     => array(
				__( 'Clarify the purpose of this image', 'alt-audit' ),
				__( 'Provide more specific description if informative', 'alt-audit' ),
				__( 'Use empty alt if purely decorative', 'alt-audit' ),
			),
			'action_required' => true,
			'auto_fixable'    => false,
		);
	}

	/**
	 * Check for decorative indicators in alt text
	 *
	 * WCAG-aligned: Checks for explicit decorative terms only.
	 * Does NOT assume short or single-word alt text is decorative.
	 *
	 * @param string $alt_text Alt text to check.
	 * @param array  $context  Additional context.
	 * @return bool True if has decorative indicators.
	 */
	private function has_decorative_indicators( $alt_text, $context ) {
		$alt_text_lower = strtolower( trim( $alt_text ) );

		// Check for explicit decorative terms only.
		foreach ( $this->decorative_indicators['generic_terms'] as $term ) {
			if ( $alt_text_lower === $term || $alt_text_lower === $term . 's' ) {
				return true;
			}
		}

		// WCAG-aligned: Do NOT assume short or single-word alt text is decorative.
		// Valid examples: "Logo", "Menu", "Home", "Search", "Close" are all appropriate.
		// Alt text quality is determined by purpose, not length.

		// Context-based indicators.
		return $this->analyze_decorative_context( $context );
	}

	/**
	 * Analyze context for decorative indicators
	 *
	 * WCAG-aligned: Uses semantic indicators (CSS classes, file names, context type)
	 * to identify decorative images based on their purpose, not dimensions.
	 *
	 * @param array $context Image context.
	 * @return bool True if context suggests decorative image.
	 */
	private function analyze_decorative_context( $context ) {
		$context_type = strtolower( $context['type'] ?? '' );
		$css_classes  = strtolower( $context['css_classes'] ?? '' );
		$file_name    = strtolower( $context['src'] ?? '' );

		// Context type indicators.
		$decorative_types = array( 'decoration', 'border', 'spacer', 'divider', 'background' );
		if ( in_array( $context_type, $decorative_types, true ) ) {
			return true;
		}

		// CSS class indicators.
		$decorative_classes = array( 'decoration', 'decorative', 'spacer', 'divider', 'border', 'bg-' );
		foreach ( $decorative_classes as $class ) {
			if ( strpos( $css_classes, $class ) !== false ) {
				return true;
			}
		}

		// File name indicators.
		$decorative_file_patterns = array( 'decoration', 'divider', 'spacer', 'border', 'bg_', 'background' );
		foreach ( $decorative_file_patterns as $pattern ) {
			if ( strpos( $file_name, $pattern ) !== false ) {
				return true;
			}
		}

		// WCAG-aligned: Do NOT use size-based detection.
		// WCAG 1.1.1 determines decorative status by purpose/function, not dimensions.
		// Size alone should not determine if an image is decorative.

		return false;
	}

	/**
	 * Calculate decorative appropriateness score
	 *
	 * WCAG-aligned: Scores based on semantic indicators only, not text length.
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @param array  $context  Image context.
	 * @return int Appropriateness score (0-100).
	 */
	private function calculate_decorative_appropriateness( $alt_text, $context ) {
		$score = 0;

		// Generic decorative terms suggest decorative purpose.
		$generic_terms = $this->decorative_indicators['generic_terms'];
		foreach ( $generic_terms as $term ) {
			if ( stripos( $alt_text, $term ) !== false ) {
				$score += 50; // Higher weight since this is semantic indicator.
			}
		}

		// WCAG-aligned: Do NOT penalize short or single-word alt text.
		// Valid examples: "Logo", "Menu", "Home" are appropriate despite being short.
		// Quality is determined by purpose, not length.

		// Context indicators (CSS classes, file names, context type).
		if ( $this->analyze_decorative_context( $context ) ) {
			$score += 50;
		}

		return min( 100, $score );
	}

	/**
	 * Get decorative guidelines
	 *
	 * @return array Guidelines for decorative images.
	 */
	public function get_decorative_guidelines() {
		return array(
			'when_to_use_empty' => array(
				__( 'Image is purely decorative with no informational value', 'alt-audit' ),
				__( 'Image is used only for visual design or layout', 'alt-audit' ),
				__( 'Information is already provided in surrounding text', 'alt-audit' ),
				__( 'Image is a spacer, divider, or border element', 'alt-audit' ),
			),
			'when_to_describe'  => array(
				__( 'Image provides information not available in text', 'alt-audit' ),
				__( 'Image is part of a link or button', 'alt-audit' ),
				__( 'Image illustrates or enhances content understanding', 'alt-audit' ),
				__( 'Image contains text or data', 'alt-audit' ),
			),
			'best_practices'    => array(
				__( 'Use empty alt="" for decorative images, not missing alt', 'alt-audit' ),
				__( 'Consider adding role="presentation" for decorative images', 'alt-audit' ),
				__( 'When in doubt, provide a brief description', 'alt-audit' ),
				__( 'Test with screen readers to verify appropriateness', 'alt-audit' ),
			),
		);
	}

	/**
	 * Get WCAG compliance for decorative images
	 *
	 * @return array WCAG compliance information.
	 */
	public function get_wcag_compliance() {
		return array(
			'guideline' => '1.1.1',
			'level'     => 'A',
			'title'     => __( 'Non-text Content', 'alt-audit' ),
			'rule'      => __( 'Decorative images should have empty alt text (alt="")', 'alt-audit' ),
			'rationale' => __( 'Screen readers skip decorative images with empty alt, preventing unnecessary noise', 'alt-audit' ),
		);
	}
}
