<?php
/**
 * WCAG Analyzer
 *
 * @package AltAudit
 * @since 1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * WCAG Analyzer class
 *
 * Analyzes alt text compliance with WCAG 2.1 Level A (1.1.1 Non-text Content).
 * Provides detailed compliance checking and recommendations for image accessibility.
 *
 * @since 1.0.0
 */
class Altaudit82ai_WCAG_Analyzer {

	/**
	 * WCAG 2.1 guidelines for image accessibility
	 *
	 * Alt Audit focuses on WCAG 2.1 Level A (1.1.1 Non-text Content) compliance.
	 * Additional guidelines are included for reference and quality improvement.
	 *
	 * @var array
	 */
	private $wcag_guidelines = array(
		'1.1.1' => array(
			'level'        => 'A',
			'title'        => 'Non-text Content',
			'description'  => 'All non-text content that is presented to the user has a text alternative that serves the equivalent purpose',
			'intent'       => 'Ensure that all non-text content is also available in text form for people who cannot see images',
			'wcag_version' => '2.1',
		),
		'2.4.4' => array(
			'level'        => 'AA',
			'title'        => 'Link Purpose (In Context)',
			'description'  => 'The purpose of each link can be determined from the link text alone',
			'intent'       => 'Help users understand the destination of links without relying on visual context',
			'wcag_version' => '2.1',
		),
		'1.4.5' => array(
			'level'        => 'AA',
			'title'        => 'Images of Text',
			'description'  => 'If the technologies being used can achieve the same visual presentation, text is used',
			'intent'       => 'Ensure text in images is accessible to assistive technologies',
			'wcag_version' => '2.1',
		),
		'2.4.9' => array(
			'level'        => 'AAA',
			'title'        => 'Link Purpose (Link Only)',
			'description'  => 'A mechanism is available to allow the purpose of each link to be identified from link text alone',
			'intent'       => 'Enable users to understand link purpose without additional context',
			'wcag_version' => '2.1',
		),
	);

	/**
	 * Analyze WCAG compliance
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @param array  $context  Image context information.
	 * @return array WCAG compliance analysis.
	 */
	public function analyze_compliance( $alt_text, $context = array() ) {
		$compliance = array(
			'level_a'            => $this->check_level_a_compliance( $alt_text, $context ),
			'level_aa'           => $this->check_level_aa_compliance( $alt_text, $context ),
			'level_aaa'          => $this->check_level_aaa_compliance( $alt_text, $context ),
			'overall'            => 'unknown',
			'violations'         => array(),
			'recommendations'    => array(),
			'guidelines_checked' => array(),
		);

		// Determine overall compliance level.
		if ( $compliance['level_aaa']['compliant'] ) {
			$compliance['overall'] = 'AAA';
		} elseif ( $compliance['level_aa']['compliant'] ) {
			$compliance['overall'] = 'AA';
		} elseif ( $compliance['level_a']['compliant'] ) {
			$compliance['overall'] = 'A';
		} else {
			$compliance['overall'] = 'Non-compliant';
		}

		// Collect all violations and recommendations.
		$compliance['violations'] = array_merge(
			$compliance['level_a']['violations'] ?? array(),
			$compliance['level_aa']['violations'] ?? array(),
			$compliance['level_aaa']['violations'] ?? array()
		);

		$compliance['recommendations'] = array_merge(
			$compliance['level_a']['recommendations'] ?? array(),
			$compliance['level_aa']['recommendations'] ?? array(),
			$compliance['level_aaa']['recommendations'] ?? array()
		);

		$compliance['guidelines_checked'] = array_keys( $this->wcag_guidelines );

		return $compliance;
	}

	/**
	 * Check WCAG Level A compliance
	 *
	 * @param string $alt_text Alt text to check.
	 * @param array  $context  Context information.
	 * @return array Level A compliance result.
	 */
	private function check_level_a_compliance( $alt_text, $context ) {
		$result = array(
			'compliant'       => true,
			'violations'      => array(),
			'recommendations' => array(),
			'guidelines'      => array(),
		);

		// WCAG 1.1.1 - Non-text Content.
		$guideline_111                 = $this->check_guideline_111( $alt_text, $context );
		$result['guidelines']['1.1.1'] = $guideline_111;

		if ( ! $guideline_111['compliant'] ) {
			$result['compliant']  = false;
			$result['violations'] = array_merge( $result['violations'], $guideline_111['violations'] );
		}

		$result['recommendations'] = array_merge( $result['recommendations'], $guideline_111['recommendations'] );

		return $result;
	}

	/**
	 * Check WCAG Level AA compliance
	 *
	 * @param string $alt_text Alt text to check.
	 * @param array  $context  Context information.
	 * @return array Level AA compliance result.
	 */
	private function check_level_aa_compliance( $alt_text, $context ) {
		$result = array(
			'compliant'       => true,
			'violations'      => array(),
			'recommendations' => array(),
			'guidelines'      => array(),
		);

		// Must first pass Level A.
		$level_a = $this->check_level_a_compliance( $alt_text, $context );
		if ( ! $level_a['compliant'] ) {
			$result['compliant']  = false;
			$result['violations'] = array_merge( $result['violations'], $level_a['violations'] );
		}

		// WCAG 2.4.4 - Link Purpose (In Context).
		if ( $this->is_link_image( $context ) ) {
			$guideline_244                 = $this->check_guideline_244( $alt_text, $context );
			$result['guidelines']['2.4.4'] = $guideline_244;

			if ( ! $guideline_244['compliant'] ) {
				$result['compliant']  = false;
				$result['violations'] = array_merge( $result['violations'], $guideline_244['violations'] );
			}

			$result['recommendations'] = array_merge( $result['recommendations'], $guideline_244['recommendations'] );
		}

		// WCAG 1.4.5 - Images of Text.
		$guideline_145                 = $this->check_guideline_145( $alt_text, $context );
		$result['guidelines']['1.4.5'] = $guideline_145;

		if ( ! $guideline_145['compliant'] ) {
			$result['compliant']  = false;
			$result['violations'] = array_merge( $result['violations'], $guideline_145['violations'] );
		}

		$result['recommendations'] = array_merge( $result['recommendations'], $guideline_145['recommendations'] );

		// Length recommendations for AA.
		if ( strlen( $alt_text ) > 150 ) {
			$result['recommendations'][] = __( 'Consider shortening alt text to under 150 characters for optimal screen reader experience', 'alt-audit' );
		}

		return $result;
	}

	/**
	 * Check WCAG Level AAA compliance
	 *
	 * @param string $alt_text Alt text to check.
	 * @param array  $context  Context information.
	 * @return array Level AAA compliance result.
	 */
	private function check_level_aaa_compliance( $alt_text, $context ) {
		$result = array(
			'compliant'       => true,
			'violations'      => array(),
			'recommendations' => array(),
			'guidelines'      => array(),
		);

		// Must first pass Level AA.
		$level_aa = $this->check_level_aa_compliance( $alt_text, $context );
		if ( ! $level_aa['compliant'] ) {
			$result['compliant']  = false;
			$result['violations'] = array_merge( $result['violations'], $level_aa['violations'] );
		}

		// WCAG 2.4.9 - Link Purpose (Link Only).
		if ( $this->is_link_image( $context ) ) {
			$guideline_249                 = $this->check_guideline_249( $alt_text, $context );
			$result['guidelines']['2.4.9'] = $guideline_249;

			if ( ! $guideline_249['compliant'] ) {
				$result['compliant']  = false;
				$result['violations'] = array_merge( $result['violations'], $guideline_249['violations'] );
			}

			$result['recommendations'] = array_merge( $result['recommendations'], $guideline_249['recommendations'] );
		}

		// AAA quality standards.
		$quality_check = $this->check_aaa_quality_standards( $alt_text, $context );
		if ( ! $quality_check['compliant'] ) {
			$result['compliant']  = false;
			$result['violations'] = array_merge( $result['violations'], $quality_check['violations'] );
		}

		$result['recommendations'] = array_merge( $result['recommendations'], $quality_check['recommendations'] );

		return $result;
	}

	/**
	 * Check WCAG 1.1.1 - Non-text Content
	 *
	 * @param string $alt_text Alt text to check.
	 * @param array  $context  Context information.
	 * @return array Guideline compliance result.
	 */
	private function check_guideline_111( $alt_text, $context ) {
		$result = array(
			'compliant'       => true,
			'violations'      => array(),
			'recommendations' => array(),
		);

		$is_decorative = $this->is_decorative_image( $context );

		// Check for missing alt text on non-decorative images.
		if ( empty( trim( $alt_text ) ) && ! $is_decorative ) {
			$result['compliant']         = false;
			$result['violations'][]      = __( 'Non-decorative images must have alt text (WCAG 1.1.1)', 'alt-audit' );
			$result['recommendations'][] = __( 'Add descriptive alt text that conveys the purpose and content of the image', 'alt-audit' );
		}

		// Check for appropriate empty alt on decorative images.
		if ( ! empty( trim( $alt_text ) ) && $is_decorative ) {
			$result['recommendations'][] = __( 'Consider using empty alt="" for decorative images (WCAG 1.1.1)', 'alt-audit' );
		}

		// Check for alt text that just repeats filename.
		if ( $this->is_filename_repetition( $alt_text, $context ) ) {
			$result['violations'][]      = __( 'Alt text should not repeat the filename (WCAG 1.1.1)', 'alt-audit' );
			$result['recommendations'][] = __( 'Replace filename with meaningful description of image content', 'alt-audit' );
		}

		return $result;
	}

	/**
	 * Check WCAG 2.4.4 - Link Purpose (In Context)
	 *
	 * @param string $alt_text Alt text to check.
	 * @param array  $context  Context information.
	 * @return array Guideline compliance result.
	 */
	private function check_guideline_244( $alt_text, $context ) {
		$result = array(
			'compliant'       => true,
			'violations'      => array(),
			'recommendations' => array(),
		);

		if ( empty( trim( $alt_text ) ) ) {
			$result['compliant']         = false;
			$result['violations'][]      = __( 'Link images must have alt text describing link purpose (WCAG 2.4.4)', 'alt-audit' );
			$result['recommendations'][] = __( 'Add alt text that describes where the link goes or what it does', 'alt-audit' );
		} elseif ( $this->describes_image_not_purpose( $alt_text, $context ) ) {
			// Check if alt text describes the link purpose vs just the image.
			$result['recommendations'][] = __( 'Alt text should describe link purpose, not just image appearance (WCAG 2.4.4)', 'alt-audit' );
		}

		return $result;
	}

	/**
	 * Check WCAG 1.4.5 - Images of Text
	 *
	 * @param string $alt_text Alt text to check.
	 * @param array  $context  Context information.
	 * @return array Guideline compliance result.
	 */
	private function check_guideline_145( $alt_text, $context ) {
		$result = array(
			'compliant'       => true,
			'violations'      => array(),
			'recommendations' => array(),
		);

		if ( $this->is_text_image( $context ) ) {
			if ( ! $this->has_essential_text_purpose( $context ) ) {
				$result['violations'][]      = __( 'Images of text should be avoided unless essential (WCAG 1.4.5)', 'alt-audit' );
				$result['recommendations'][] = __( 'Consider using actual text instead of an image of text', 'alt-audit' );
			}

			// If it's an image of text, alt text should include the text.
			if ( ! $this->alt_includes_text_content( $alt_text, $context ) ) {
				$result['recommendations'][] = __( 'Alt text should include the text content shown in the image (WCAG 1.4.5)', 'alt-audit' );
			}
		}

		return $result;
	}

	/**
	 * Check WCAG 2.4.9 - Link Purpose (Link Only)
	 *
	 * @param string $alt_text Alt text to check.
	 * @param array  $context  Context information.
	 * @return array Guideline compliance result.
	 */
	private function check_guideline_249( $alt_text, $context ) {
		$result = array(
			'compliant'       => true,
			'violations'      => array(),
			'recommendations' => array(),
		);

		// For AAA, link purpose must be clear from alt text alone.
		if ( ! $this->is_link_purpose_clear( $alt_text, $context ) ) {
			$result['compliant']         = false;
			$result['violations'][]      = __( 'Link purpose must be clear from alt text alone (WCAG 2.4.9)', 'alt-audit' );
			$result['recommendations'][] = __( 'Make alt text more specific about the link destination or action', 'alt-audit' );
		}

		return $result;
	}

	/**
	 * Check AAA quality standards
	 *
	 * @param string $alt_text Alt text to check.
	 * @param array  $context  Context information.
	 * @return array Quality standards result.
	 */
	private function check_aaa_quality_standards( $alt_text, $context ) {
		$result = array(
			'compliant'       => true,
			'violations'      => array(),
			'recommendations' => array(),
		);

		// Length standards for AAA.
		if ( strlen( $alt_text ) > 125 ) {
			$result['recommendations'][] = __( 'For optimal accessibility, keep alt text under 125 characters', 'alt-audit' );
		}

		// Quality standards for AAA.
		if ( ! empty( $alt_text ) && $this->calculate_quality_score( $alt_text ) < 80 ) {
			$result['compliant']         = false;
			$result['violations'][]      = __( 'Alt text quality should be high for AAA compliance', 'alt-audit' );
			$result['recommendations'][] = __( 'Improve alt text with more specific, descriptive language', 'alt-audit' );
		}

		return $result;
	}

	/**
	 * Check if image is in a link
	 *
	 * @param array $context Context information.
	 * @return bool True if image is in a link.
	 */
	private function is_link_image( $context ) {
		return ! empty( $context['is_link'] ) || ! empty( $context['link_url'] ) || ! empty( $context['parent_link'] );
	}

	/**
	 * Check if image is decorative
	 *
	 * @param array $context Context information.
	 * @return bool True if image is decorative.
	 */
	private function is_decorative_image( $context ) {
		$decorative_indicators = array(
			'role'      => 'presentation',
			'type'      => 'decorative',
			'css_class' => array( 'decoration', 'decorative', 'spacer', 'divider' ),
		);

		// Check role attribute.
		if ( ! empty( $context['role'] ) && 'presentation' === $context['role'] ) {
			return true;
		}

		// Check explicit type.
		if ( ! empty( $context['type'] ) && 'decorative' === $context['type'] ) {
			return true;
		}

		// Check CSS classes.
		$css_classes = strtolower( $context['css_classes'] ?? $context['class'] ?? '' );
		foreach ( $decorative_indicators['css_class'] as $class ) {
			if ( strpos( $css_classes, $class ) !== false ) {
				return true;
			}
		}

		// Check image dimensions (very small images often decorative).
		$width  = intval( $context['width'] ?? 0 );
		$height = intval( $context['height'] ?? 0 );
		if ( ( $width > 0 && $width <= 20 ) || ( $height > 0 && $height <= 20 ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Check if alt text is just repeating filename
	 *
	 * @param string $alt_text Alt text to check.
	 * @param array  $context  Context information.
	 * @return bool True if repeating filename.
	 */
	private function is_filename_repetition( $alt_text, $context ) {
		$src = $context['src'] ?? '';
		if ( empty( $src ) || empty( $alt_text ) ) {
			return false;
		}

		$filename       = pathinfo( $src, PATHINFO_FILENAME );
		$alt_text_clean = preg_replace( '/[^a-zA-Z0-9]/', '', strtolower( $alt_text ) );
		$filename_clean = preg_replace( '/[^a-zA-Z0-9]/', '', strtolower( $filename ) );

		return $alt_text_clean === $filename_clean;
	}

	/**
	 * Check if alt text describes image appearance rather than link purpose
	 *
	 * @param string $alt_text Alt text to check.
	 * @param array  $context  Context information.
	 * @return bool True if describes image not purpose.
	 */
	private function describes_image_not_purpose( $alt_text, $context ) {
		// Look for visual descriptors that don't relate to purpose.
		$visual_only_patterns = array(
			'/\b(red|blue|green|yellow|orange|purple|black|white)\s+(button|icon|image)\b/i',
			'/\b(small|large|big|tiny)\s+(button|icon|image)\b/i',
			'/\b(round|square|circular)\s+(button|icon|image)\b/i',
		);

		foreach ( $visual_only_patterns as $pattern ) {
			if ( preg_match( $pattern, $alt_text ) ) {
				return true;
			}
		}

		// Check if it has purpose indicators.
		$purpose_indicators = array( 'go to', 'visit', 'download', 'view', 'open', 'read more', 'learn more', 'contact' );
		foreach ( $purpose_indicators as $indicator ) {
			if ( stripos( $alt_text, $indicator ) !== false ) {
				return false;
			}
		}

		// If alt text is very generic without purpose context, it likely describes appearance.
		$generic_terms  = array( 'button', 'icon', 'image', 'link', 'logo' );
		$alt_text_lower = strtolower( trim( $alt_text ) );

		return in_array( $alt_text_lower, $generic_terms, true );
	}

	/**
	 * Check if image contains text
	 *
	 * @param array $context Context information.
	 * @return bool True if image contains text.
	 */
	private function is_text_image( $context ) {
		// Check explicit indicators.
		if ( ! empty( $context['contains_text'] ) ) {
			return true;
		}

		// Check file naming patterns.
		$src             = strtolower( $context['src'] ?? '' );
		$text_indicators = array( 'logo', 'banner', 'header', 'title', 'text', 'quote', 'sign' );

		foreach ( $text_indicators as $indicator ) {
			if ( strpos( $src, $indicator ) !== false ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Check if text image has essential purpose
	 *
	 * @param array $context Context information.
	 * @return bool True if text is essential.
	 */
	private function has_essential_text_purpose( $context ) {
		$essential_types = array( 'logo', 'signature', 'handwriting', 'artistic', 'brand' );
		$type            = strtolower( $context['type'] ?? '' );

		return in_array( $type, $essential_types, true );
	}

	/**
	 * Check if alt text includes the text content from image
	 *
	 * @param string $alt_text Alt text to check.
	 * @param array  $context  Context information.
	 * @return bool True if includes text content.
	 */
	private function alt_includes_text_content( $alt_text, $context ) {
		$image_text = $context['text_content'] ?? '';
		if ( empty( $image_text ) ) {
			return true; // Can't verify if we don't know the text content.
		}

		// Check if alt text contains the image text.
		return stripos( $alt_text, trim( $image_text ) ) !== false;
	}

	/**
	 * Check if link purpose is clear from alt text alone
	 *
	 * @param string $alt_text Alt text to check.
	 * @param array  $context  Context information.
	 * @return bool True if purpose is clear.
	 */
	private function is_link_purpose_clear( $alt_text, $context ) {
		if ( empty( trim( $alt_text ) ) ) {
			return false;
		}

		// Check for clear action words.
		$action_words = array( 'go to', 'visit', 'download', 'view', 'open', 'read', 'learn', 'contact', 'call', 'email' );
		foreach ( $action_words as $action ) {
			if ( stripos( $alt_text, $action ) !== false ) {
				return true;
			}
		}

		// Check for destination indicators.
		$destination_words = array( 'page', 'site', 'website', 'article', 'section', 'home', 'about', 'contact' );
		foreach ( $destination_words as $destination ) {
			if ( stripos( $alt_text, $destination ) !== false ) {
				return true;
			}
		}

		// Alt text should be reasonably descriptive for purpose.
		return strlen( $alt_text ) >= 10 && str_word_count( $alt_text ) >= 2;
	}

	/**
	 * Calculate basic quality score for WCAG assessment
	 *
	 * @param string $alt_text Alt text to score.
	 * @return int Quality score (0-100).
	 */
	private function calculate_quality_score( $alt_text ) {
		$score      = 50; // Base score.
		$length     = strlen( $alt_text );
		$word_count = str_word_count( $alt_text );

		// Length scoring.
		if ( $length >= 25 && $length <= 125 ) {
			$score += 20;
		} elseif ( $length > 0 ) {
			$score += 10;
		}

		// Word count scoring.
		if ( $word_count >= 4 && $word_count <= 20 ) {
			$score += 15;
		}

		// Structure scoring.
		if ( preg_match( '/^[A-Z]/', $alt_text ) ) {
			$score += 5;
		}

		if ( ! preg_match( '/\.(jpg|jpeg|png|gif)$/i', $alt_text ) ) {
			$score += 5;
		}

		// Descriptiveness (basic check).
		if ( preg_match_all( '/\b(red|blue|green|large|small|happy|sad)\b/i', $alt_text ) ) {
			$score += 10;
		}

		return max( 0, min( 100, $score ) );
	}

	/**
	 * Get WCAG guideline information
	 *
	 * @param string $guideline_id Guideline ID (e.g., '1.1.1').
	 * @return array|null Guideline information.
	 */
	public function get_guideline_info( $guideline_id ) {
		return $this->wcag_guidelines[ $guideline_id ] ?? null;
	}

	/**
	 * Get all WCAG guidelines
	 *
	 * @return array All guidelines information.
	 */
	public function get_all_guidelines() {
		return $this->wcag_guidelines;
	}
}
