<?php
/**
 * Content Analyzer
 *
 * @package AltAudit
 * @since 1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Content Analyzer class
 *
 * Analyzes alt text content for quality, readability, and effectiveness.
 * Provides detailed content analysis beyond basic structure checks.
 *
 * @since 1.0.0
 */
class Altaudit82ai_Content_Analyzer {

	/**
	 * Content analysis patterns
	 *
	 * @var array
	 */
	private $analysis_patterns = array(
		'descriptive_categories' => array(
			'colors'    => '/\b(red|blue|green|yellow|orange|purple|black|white|gray|grey|brown|pink|violet|indigo|cyan|magenta|crimson|azure|emerald|amber|turquoise|burgundy)\b/i',
			'sizes'     => '/\b(large|small|big|tiny|huge|massive|little|medium|tall|short|wide|narrow|long|thick|thin|enormous|gigantic|minuscule)\b/i',
			'shapes'    => '/\b(round|square|circular|rectangular|triangular|oval|curved|straight|angular|spherical|cylindrical|conical)\b/i',
			'materials' => '/\b(wooden|metal|glass|plastic|fabric|leather|stone|brick|concrete|ceramic|paper|cardboard|rubber|steel|aluminum)\b/i',
			'textures'  => '/\b(smooth|rough|glossy|matte|textured|polished|bumpy|silky|coarse|fine|soft|hard|scratchy|velvety)\b/i',
			'emotions'  => '/\b(happy|sad|smiling|frowning|serious|excited|calm|angry|peaceful|joyful|content|worried|surprised|focused|relaxed)\b/i',
			'lighting'  => '/\b(bright|dim|natural|artificial|warm|cool|backlit|shadowy|golden|harsh|soft|dramatic|ambient|fluorescent)\b/i',
			'weather'   => '/\b(sunny|cloudy|rainy|snowy|foggy|stormy|clear|overcast|misty|humid|dry|windy)\b/i',
			'actions'   => '/\b(running|walking|sitting|standing|jumping|dancing|working|playing|reading|writing|cooking|eating|sleeping)\b/i',
			'positions' => '/\b(left|right|center|top|bottom|above|below|beside|behind|front|inside|outside|between|among|near|far)\b/i',
		),
		'quality_indicators'     => array(
			'specific_nouns'    => '/\b(person|man|woman|child|dog|cat|car|house|tree|flower|computer|phone|book|table|chair)\b/i',
			'action_verbs'      => '/\b(showing|displaying|featuring|holding|wearing|carrying|pointing|looking|smiling|walking)\b/i',
			'context_words'     => '/\b(in|on|at|with|during|while|beside|near|surrounded by|next to|in front of)\b/i',
			'measurement_units' => '/\b\d+\s*(years?|months?|feet|inches|meters|centimeters|pounds|kilograms|degrees)\b/i',
		),
		'problematic_patterns'   => array(
			'redundant_phrases'    => '/\b(image of|picture of|photo of|graphic of|screenshot of|illustration of)\b/i',
			'vague_terms'          => '/\b(thing|stuff|something|someone|here|there|this|that|it|they)\b/i',
			'filler_words'         => '/\b(very|really|quite|pretty|somewhat|kind of|sort of|a bit|rather)\b/i',
			'excessive_adjectives' => '/\b(\w+ly\s+\w+ly|\w+ing\s+\w+ing)\b/i',
		),
	);

	/**
	 * Analyze content comprehensiveness
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @param array  $context  Image context.
	 * @return array Content analysis result.
	 */
	public function analyze_content( $alt_text, $context = array() ) {
		return array(
			'descriptive_analysis'  => $this->analyze_descriptive_content( $alt_text ),
			'readability_analysis'  => $this->analyze_readability( $alt_text ),
			'specificity_analysis'  => $this->analyze_specificity( $alt_text ),
			'context_analysis'      => $this->analyze_context_appropriateness( $alt_text, $context ),
			'language_analysis'     => $this->analyze_language_quality( $alt_text ),
			'content_gaps'          => $this->identify_content_gaps( $alt_text, $context ),
			'improvement_potential' => $this->assess_improvement_potential( $alt_text ),
		);
	}

	/**
	 * Analyze descriptive content
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return array Descriptive content analysis.
	 */
	private function analyze_descriptive_content( $alt_text ) {
		$found_categories        = array();
		$category_counts         = array();
		$total_descriptive_words = 0;

		foreach ( $this->analysis_patterns['descriptive_categories'] as $category => $pattern ) {
			$matches = preg_match_all( $pattern, $alt_text );
			if ( $matches > 0 ) {
				$found_categories[]           = $category;
				$category_counts[ $category ] = $matches;
				$total_descriptive_words     += $matches;
			}
		}

		$variety_score = count( $found_categories );
		$density_score = $total_descriptive_words / max( 1, str_word_count( $alt_text ) );

		return array(
			'categories_found'      => $found_categories,
			'category_counts'       => $category_counts,
			'total_descriptive'     => $total_descriptive_words,
			'variety_score'         => $variety_score,
			'density_score'         => round( $density_score, 3 ),
			'descriptiveness_level' => $this->calculate_descriptiveness_level( $variety_score, $total_descriptive_words ),
		);
	}

	/**
	 * Analyze readability
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return array Readability analysis.
	 */
	private function analyze_readability( $alt_text ) {
		$word_count     = str_word_count( $alt_text );
		$sentence_count = max( 1, preg_match_all( '/[.!?]+/', $alt_text ) );
		$syllable_count = $this->estimate_syllable_count( $alt_text );
		$complex_words  = $this->count_complex_words( $alt_text );

		// Flesch Reading Ease calculation.
		$avg_sentence_length    = $word_count / $sentence_count;
		$avg_syllables_per_word = $syllable_count / max( 1, $word_count );
		$flesch_score           = 206.835 - ( 1.015 * $avg_sentence_length ) - ( 84.6 * $avg_syllables_per_word );
		$flesch_score           = max( 0, min( 100, $flesch_score ) );

		return array(
			'flesch_score'           => round( $flesch_score, 1 ),
			'reading_level'          => $this->flesch_to_reading_level( $flesch_score ),
			'avg_sentence_length'    => round( $avg_sentence_length, 1 ),
			'avg_syllables_per_word' => round( $avg_syllables_per_word, 2 ),
			'complex_word_count'     => $complex_words,
			'complexity_ratio'       => round( $complex_words / max( 1, $word_count ), 3 ),
			'readability_issues'     => $this->identify_readability_issues( $alt_text ),
		);
	}

	/**
	 * Analyze specificity
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return array Specificity analysis.
	 */
	private function analyze_specificity( $alt_text ) {
		$specific_elements = 0;
		$vague_elements    = 0;
		$specific_details  = array();
		$vague_details     = array();

		// Count specific quality indicators.
		foreach ( $this->analysis_patterns['quality_indicators'] as $type => $pattern ) {
			$matches = preg_match_all( $pattern, $alt_text, $found );
			if ( $matches > 0 ) {
				$specific_elements        += $matches;
				$specific_details[ $type ] = $found[0];
			}
		}

		// Count problematic/vague patterns.
		foreach ( $this->analysis_patterns['problematic_patterns'] as $type => $pattern ) {
			$matches = preg_match_all( $pattern, $alt_text, $found );
			if ( $matches > 0 ) {
				$vague_elements        += $matches;
				$vague_details[ $type ] = $found[0];
			}
		}

		$total_words       = str_word_count( $alt_text );
		$specificity_ratio = $specific_elements / max( 1, $total_words );
		$vagueness_ratio   = $vague_elements / max( 1, $total_words );

		return array(
			'specific_elements' => $specific_elements,
			'vague_elements'    => $vague_elements,
			'specificity_ratio' => round( $specificity_ratio, 3 ),
			'vagueness_ratio'   => round( $vagueness_ratio, 3 ),
			'specificity_score' => $this->calculate_specificity_score( $specificity_ratio, $vagueness_ratio ),
			'specific_details'  => $specific_details,
			'vague_details'     => $vague_details,
		);
	}

	/**
	 * Analyze context appropriateness
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @param array  $context  Image context.
	 * @return array Context analysis.
	 */
	private function analyze_context_appropriateness( $alt_text, $context ) {
		$context_type         = $context['type'] ?? 'general';
		$context_requirements = $this->get_context_requirements( $context_type );
		$met_requirements     = array();
		$missing_requirements = array();

		foreach ( $context_requirements as $requirement => $pattern ) {
			if ( preg_match( $pattern, $alt_text ) ) {
				$met_requirements[] = $requirement;
			} else {
				$missing_requirements[] = $requirement;
			}
		}

		$appropriateness_score = count( $met_requirements ) / max( 1, count( $context_requirements ) );

		return array(
			'context_type'          => $context_type,
			'requirements_met'      => $met_requirements,
			'requirements_missing'  => $missing_requirements,
			'appropriateness_score' => round( $appropriateness_score, 3 ),
			'context_suggestions'   => $this->get_context_suggestions( $context_type, $missing_requirements ),
		);
	}

	/**
	 * Analyze language quality
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return array Language quality analysis.
	 */
	private function analyze_language_quality( $alt_text ) {
		$grammar_score     = $this->assess_grammar_quality( $alt_text );
		$flow_score        = $this->assess_natural_flow( $alt_text );
		$tone_analysis     = $this->analyze_tone( $alt_text );
		$inclusivity_score = $this->assess_inclusivity( $alt_text );

		return array(
			'grammar_score'     => $grammar_score,
			'flow_score'        => $flow_score,
			'tone_analysis'     => $tone_analysis,
			'inclusivity_score' => $inclusivity_score,
			'language_issues'   => $this->identify_language_issues( $alt_text ),
			'style_suggestions' => $this->get_style_suggestions( $alt_text ),
		);
	}

	/**
	 * Identify content gaps
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @param array  $context  Image context.
	 * @return array Content gaps identified.
	 */
	private function identify_content_gaps( $alt_text, $context ) {
		$gaps = array();

		// Check for missing essential elements.
		$essential_elements = $this->get_essential_elements( $context );
		foreach ( $essential_elements as $element => $importance ) {
			if ( ! $this->has_element( $alt_text, $element ) ) {
				$gaps[] = array(
					'type'       => 'missing_element',
					'element'    => $element,
					'importance' => $importance,
					'suggestion' => $this->get_element_suggestion( $element ),
				);
			}
		}

		// Check for insufficient detail.
		if ( $this->is_insufficient_detail( $alt_text, $context ) ) {
			$gaps[] = array(
				'type'       => 'insufficient_detail',
				'suggestion' => __( 'Add more specific descriptive details', 'alt-audit' ),
			);
		}

		// Check for missing context.
		if ( $this->is_missing_context( $alt_text, $context ) ) {
			$gaps[] = array(
				'type'       => 'missing_context',
				'suggestion' => __( 'Explain how the image relates to surrounding content', 'alt-audit' ),
			);
		}

		return $gaps;
	}

	/**
	 * Assess improvement potential
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return array Improvement potential assessment.
	 */
	private function assess_improvement_potential( $alt_text ) {
		$current_score          = $this->calculate_overall_content_score( $alt_text );
		$potential_improvements = array();

		// Quick wins.
		if ( preg_match( '/\b(image|picture|photo) of\b/i', $alt_text ) ) {
			$potential_improvements['remove_redundant'] = array(
				'impact'      => 'medium',
				'effort'      => 'low',
				'description' => __( 'Remove redundant phrases', 'alt-audit' ),
			);
		}

		// Content enhancements.
		$descriptive_analysis = $this->analyze_descriptive_content( $alt_text );
		if ( $descriptive_analysis['variety_score'] < 3 ) {
			$potential_improvements['add_descriptors'] = array(
				'impact'      => 'high',
				'effort'      => 'medium',
				'description' => __( 'Add more varied descriptive elements', 'alt-audit' ),
			);
		}

		// Structure improvements.
		if ( ! preg_match( '/^[A-Z]/', $alt_text ) ) {
			$potential_improvements['fix_capitalization'] = array(
				'impact'      => 'low',
				'effort'      => 'low',
				'description' => __( 'Fix capitalization', 'alt-audit' ),
			);
		}

		$potential_score = $this->estimate_potential_score( $alt_text, $potential_improvements );

		return array(
			'current_score'    => $current_score,
			'potential_score'  => $potential_score,
			'improvement_gap'  => $potential_score - $current_score,
			'improvements'     => $potential_improvements,
			'priority_actions' => $this->prioritize_improvements( $potential_improvements ),
		);
	}

	/**
	 * Calculate descriptiveness level
	 *
	 * @param int $variety_score      Number of descriptive categories.
	 * @param int $total_descriptive  Total descriptive words.
	 * @return string Descriptiveness level.
	 */
	private function calculate_descriptiveness_level( $variety_score, $total_descriptive ) {
		if ( $variety_score >= 4 && $total_descriptive >= 5 ) {
			return 'excellent';
		} elseif ( $variety_score >= 3 && $total_descriptive >= 3 ) {
			return 'good';
		} elseif ( $variety_score >= 2 && $total_descriptive >= 2 ) {
			return 'adequate';
		} elseif ( $variety_score >= 1 || $total_descriptive >= 1 ) {
			return 'minimal';
		} else {
			return 'insufficient';
		}
	}

	/**
	 * Estimate syllable count
	 *
	 * @param string $text Text to analyze.
	 * @return int Estimated syllable count.
	 */
	private function estimate_syllable_count( $text ) {
		$words           = str_word_count( $text, 1 );
		$total_syllables = 0;

		foreach ( $words as $word ) {
			$word = strtolower( preg_replace( '/[^a-zA-Z]/', '', $word ) );
			if ( empty( $word ) ) {
				continue;
			}

			// Count vowel groups.
			$vowel_groups = preg_match_all( '/[aeiouy]+/', $word );

			// Subtract silent e.
			if ( substr( $word, -1 ) === 'e' ) {
				--$vowel_groups;
			}

			// Ensure at least 1 syllable.
			$syllables        = max( 1, $vowel_groups );
			$total_syllables += $syllables;
		}

		return max( 1, $total_syllables );
	}

	/**
	 * Count complex words (3+ syllables)
	 *
	 * @param string $text Text to analyze.
	 * @return int Number of complex words.
	 */
	private function count_complex_words( $text ) {
		$words         = str_word_count( $text, 1 );
		$complex_count = 0;

		foreach ( $words as $word ) {
			$word_clean = strtolower( preg_replace( '/[^a-zA-Z]/', '', $word ) );
			if ( strlen( $word_clean ) >= 7 ) { // Approximation for 3+ syllables.
				++$complex_count;
			}
		}

		return $complex_count;
	}

	/**
	 * Convert Flesch score to reading level
	 *
	 * @param float $flesch_score Flesch reading ease score.
	 * @return string Reading level description.
	 */
	private function flesch_to_reading_level( $flesch_score ) {
		if ( $flesch_score >= 90 ) {
			return __( 'Very Easy (5th grade)', 'alt-audit' );
		} elseif ( $flesch_score >= 80 ) {
			return __( 'Easy (6th grade)', 'alt-audit' );
		} elseif ( $flesch_score >= 70 ) {
			return __( 'Fairly Easy (7th grade)', 'alt-audit' );
		} elseif ( $flesch_score >= 60 ) {
			return __( 'Standard (8th-9th grade)', 'alt-audit' );
		} elseif ( $flesch_score >= 50 ) {
			return __( 'Fairly Difficult (10th-12th grade)', 'alt-audit' );
		} elseif ( $flesch_score >= 30 ) {
			return __( 'Difficult (College level)', 'alt-audit' );
		} else {
			return __( 'Very Difficult (Graduate level)', 'alt-audit' );
		}
	}

	/**
	 * Identify readability issues
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return array Readability issues found.
	 */
	private function identify_readability_issues( $alt_text ) {
		$issues = array();

		// Long sentences.
		$sentences = preg_split( '/[.!?]+/', $alt_text );
		foreach ( $sentences as $sentence ) {
			if ( str_word_count( $sentence ) > 20 ) {
				$issues[] = __( 'Some sentences are too long', 'alt-audit' );
				break;
			}
		}

		// Complex words.
		if ( $this->count_complex_words( $alt_text ) > 2 ) {
			$issues[] = __( 'Contains too many complex words', 'alt-audit' );
		}

		// Technical jargon.
		if ( preg_match( '/\b(API|HTML|CSS|JavaScript|jQuery|AJAX|SQL)\b/i', $alt_text ) ) {
			$issues[] = __( 'Contains technical jargon', 'alt-audit' );
		}

		return $issues;
	}

	/**
	 * Calculate specificity score
	 *
	 * @param float $specificity_ratio Ratio of specific elements.
	 * @param float $vagueness_ratio   Ratio of vague elements.
	 * @return int Specificity score (0-100).
	 */
	private function calculate_specificity_score( $specificity_ratio, $vagueness_ratio ) {
		$base_score = $specificity_ratio * 100;
		$penalty    = $vagueness_ratio * 50;

		return max( 0, min( 100, round( $base_score - $penalty ) ) );
	}

	/**
	 * Get context requirements
	 *
	 * @param string $context_type Context type.
	 * @return array Context requirements patterns.
	 */
	private function get_context_requirements( $context_type ) {
		$requirements = array(
			'product'    => array(
				'product_name'   => '/\b[A-Z][a-z]+(\s+[A-Z][a-z]+)*\b/',
				'key_features'   => '/\b(wireless|bluetooth|waterproof|portable|premium)\b/i',
				'visual_details' => '/\b(color|size|material|texture)\b/i',
			),
			'article'    => array(
				'content_relation' => '/\b(showing|illustrating|demonstrating|example)\b/i',
				'context_info'     => '/\b(in|during|while|at|with)\b/i',
			),
			'navigation' => array(
				'purpose'     => '/\b(home|menu|search|back|next|contact)\b/i',
				'destination' => '/\b(page|section|site|area)\b/i',
			),
		);

		return $requirements[ $context_type ] ?? array();
	}

	/**
	 * Get context suggestions
	 *
	 * @param string $context_type        Context type.
	 * @param array  $missing_requirements Missing requirements.
	 * @return array Context suggestions.
	 */
	private function get_context_suggestions( $context_type, $missing_requirements ) {
		$suggestions = array();

		foreach ( $missing_requirements as $requirement ) {
			switch ( $requirement ) {
				case 'product_name':
					$suggestions[] = __( 'Include the product name or brand', 'alt-audit' );
					break;
				case 'key_features':
					$suggestions[] = __( 'Mention key product features or benefits', 'alt-audit' );
					break;
				case 'visual_details':
					$suggestions[] = __( 'Describe color, size, or material', 'alt-audit' );
					break;
				case 'content_relation':
					$suggestions[] = __( 'Explain how the image relates to the content', 'alt-audit' );
					break;
				case 'purpose':
					$suggestions[] = __( 'Clarify the navigation purpose', 'alt-audit' );
					break;
			}
		}

		return $suggestions;
	}

	/**
	 * Assess grammar quality
	 *
	 * @param string $alt_text Alt text to assess.
	 * @return int Grammar score (0-100).
	 */
	private function assess_grammar_quality( $alt_text ) {
		$score = 100;

		// Basic grammar checks.
		if ( ! preg_match( '/^[A-Z]/', $alt_text ) ) {
			$score -= 10;
		}

		if ( preg_match( '/[.!?]{2,}/', $alt_text ) ) {
			$score -= 15;
		}

		if ( preg_match( '/\s{2,}/', $alt_text ) ) {
			$score -= 5;
		}

		return max( 0, $score );
	}

	/**
	 * Assess natural flow
	 *
	 * @param string $alt_text Alt text to assess.
	 * @return int Flow score (0-100).
	 */
	private function assess_natural_flow( $alt_text ) {
		$score = 50;

		// Check for connecting words.
		if ( preg_match( '/\b(with|in|on|while|during|wearing|holding)\b/i', $alt_text ) ) {
			$score += 25;
		}

		// Check for logical progression.
		if ( preg_match( '/\b(showing|displaying|featuring)\b/i', $alt_text ) ) {
			$score += 15;
		}

		// Penalty for choppy structure.
		if ( str_word_count( $alt_text ) > 5 && ! preg_match( '/[,;]/', $alt_text ) ) {
			$score -= 10;
		}

		return max( 0, min( 100, $score ) );
	}

	/**
	 * Analyze tone
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return array Tone analysis.
	 */
	private function analyze_tone( $alt_text ) {
		$tone_indicators = array(
			'formal'      => preg_match( '/\b(demonstrates|illustrates|exhibits|portrays)\b/i', $alt_text ),
			'casual'      => preg_match( '/\b(shows|has|with|wearing)\b/i', $alt_text ),
			'descriptive' => preg_match( '/\b(bright|colorful|detailed|intricate)\b/i', $alt_text ),
			'objective'   => ! preg_match( '/\b(amazing|beautiful|stunning|incredible)\b/i', $alt_text ),
		);

		$dominant_tone = 'neutral';
		$max_score     = 0;

		foreach ( $tone_indicators as $tone => $score ) {
			if ( $score > $max_score ) {
				$max_score     = $score;
				$dominant_tone = $tone;
			}
		}

		return array(
			'dominant_tone'   => $dominant_tone,
			'tone_scores'     => $tone_indicators,
			'appropriateness' => $this->assess_tone_appropriateness( $dominant_tone ),
		);
	}

	/**
	 * Assess inclusivity
	 *
	 * @param string $alt_text Alt text to assess.
	 * @return int Inclusivity score (0-100).
	 */
	private function assess_inclusivity( $alt_text ) {
		$score = 100;

		// Check for potentially exclusive language.
		$exclusive_terms = array( 'normal', 'weird', 'strange', 'abnormal', 'typical', 'unusual' );
		foreach ( $exclusive_terms as $term ) {
			if ( stripos( $alt_text, $term ) !== false ) {
				$score -= 20;
			}
		}

		// Check for assumptions.
		if ( preg_match( '/\b(obviously|clearly|naturally|of course)\b/i', $alt_text ) ) {
			$score -= 15;
		}

		return max( 0, $score );
	}

	/**
	 * Identify language issues
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return array Language issues found.
	 */
	private function identify_language_issues( $alt_text ) {
		$issues = array();

		if ( preg_match( '/\b(very|really|quite|extremely)\b/i', $alt_text ) ) {
			$issues[] = __( 'Contains unnecessary intensifiers', 'alt-audit' );
		}

		if ( preg_match( '/\b(thing|stuff|something)\b/i', $alt_text ) ) {
			$issues[] = __( 'Uses vague terms instead of specific descriptions', 'alt-audit' );
		}

		return $issues;
	}

	/**
	 * Get style suggestions
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return array Style suggestions.
	 */
	private function get_style_suggestions( $alt_text ) {
		$suggestions = array();

		if ( ! preg_match( '/\b(showing|displaying|featuring|with|wearing)\b/i', $alt_text ) ) {
			$suggestions[] = __( 'Consider adding connecting words for better flow', 'alt-audit' );
		}

		if ( str_word_count( $alt_text ) < 6 ) {
			$suggestions[] = __( 'Consider adding more descriptive detail', 'alt-audit' );
		}

		return $suggestions;
	}

	/**
	 * Get essential elements for context
	 *
	 * @param array $context Image context.
	 * @return array Essential elements with importance levels.
	 */
	private function get_essential_elements( $context ) {
		$context_type = $context['type'] ?? 'general';

		$elements = array(
			'product'    => array(
				'subject'  => 'high',
				'color'    => 'medium',
				'material' => 'medium',
				'brand'    => 'low',
			),
			'article'    => array(
				'subject' => 'high',
				'action'  => 'medium',
				'context' => 'medium',
			),
			'navigation' => array(
				'purpose'     => 'high',
				'destination' => 'medium',
			),
		);

		return $elements[ $context_type ] ?? array( 'subject' => 'high' );
	}

	/**
	 * Check if alt text has specific element
	 *
	 * @param string $alt_text Alt text to check.
	 * @param string $element  Element to look for.
	 * @return bool True if element is present.
	 */
	private function has_element( $alt_text, $element ) {
		$patterns = array(
			'subject' => '/\b(person|man|woman|child|people|group|team|individual)\b/i',
			'color'   => '/\b(red|blue|green|yellow|orange|purple|black|white|gray|brown|pink)\b/i',
			'action'  => '/\b(sitting|standing|walking|running|working|playing|holding|wearing)\b/i',
			'context' => '/\b(in|on|at|with|during|while|beside|near)\b/i',
			'purpose' => '/\b(home|menu|search|contact|about|navigation|link)\b/i',
		);

		$pattern = $patterns[ $element ] ?? '/\b' . preg_quote( $element, '/' ) . '\b/i';
		return preg_match( $pattern, $alt_text );
	}

	/**
	 * Get element suggestion
	 *
	 * @param string $element Element type.
	 * @return string Suggestion for the element.
	 */
	private function get_element_suggestion( $element ) {
		$suggestions = array(
			'subject' => __( 'Identify who or what is the main subject', 'alt-audit' ),
			'color'   => __( 'Include color information', 'alt-audit' ),
			'action'  => __( 'Describe what action is taking place', 'alt-audit' ),
			'context' => __( 'Explain the setting or context', 'alt-audit' ),
			'purpose' => __( 'Clarify the purpose or function', 'alt-audit' ),
		);

		return $suggestions[ $element ] ?? sprintf( /* translators: Placeholder for dynamic content */ __( 'Consider adding %s information', 'alt-audit' ), $element );
	}

	/**
	 * Check for insufficient detail
	 *
	 * @param string $alt_text Alt text to check.
	 * @param array  $context  Image context (reserved for future use).
	 * @return bool True if insufficient detail.
	 *
	 * phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.FoundAfterLastUsed -- Parameter reserved for future context-aware analysis.
	 */
	private function is_insufficient_detail( $alt_text, $context ) {
		$word_count           = str_word_count( $alt_text );
		$descriptive_analysis = $this->analyze_descriptive_content( $alt_text );

		return $word_count < 4 || $descriptive_analysis['variety_score'] < 2;
	}

	/**
	 * Check for missing context
	 *
	 * @param string $alt_text Alt text to check.
	 * @param array  $context  Image context.
	 * @return bool True if missing context.
	 */
	private function is_missing_context( $alt_text, $context ) {
		$context_type = $context['type'] ?? 'general';

		if ( 'general' !== $context_type ) {
			$requirements = $this->get_context_requirements( $context_type );
			foreach ( $requirements as $pattern ) {
				if ( preg_match( $pattern, $alt_text ) ) {
					// Found at least one context indicator.
					return false;
				}
			}
			// No context indicators found.
			return true;
		}

		// General context doesn't require specific indicators.
		return false;
	}

	/**
	 * Calculate overall content score
	 *
	 * @param string $alt_text Alt text to score.
	 * @return int Overall content score (0-100).
	 */
	private function calculate_overall_content_score( $alt_text ) {
		$descriptive = $this->analyze_descriptive_content( $alt_text );
		$readability = $this->analyze_readability( $alt_text );
		$specificity = $this->analyze_specificity( $alt_text );

		$score  = 0;
		$score += $descriptive['variety_score'] * 10; // Up to 50 points.
		$score += ( $readability['flesch_score'] / 100 ) * 25; // Up to 25 points.
		$score += ( $specificity['specificity_score'] / 100 ) * 25; // Up to 25 points.
		return min( 100, round( $score ) );
	}

	/**
	 * Estimate potential score with improvements
	 *
	 * @param string $alt_text    Alt text to analyze.
	 * @param array  $improvements Potential improvements.
	 * @return int Estimated potential score.
	 */
	private function estimate_potential_score( $alt_text, $improvements ) {
		$current_score  = $this->calculate_overall_content_score( $alt_text );
		$potential_gain = 0;

		foreach ( $improvements as $improvement ) {
			switch ( $improvement['impact'] ) {
				case 'high':
					$potential_gain += 15;
					break;
				case 'medium':
					$potential_gain += 8;
					break;
				case 'low':
					$potential_gain += 3;
					break;
			}
		}

		return min( 100, $current_score + $potential_gain );
	}

	/**
	 * Prioritize improvements
	 *
	 * @param array $improvements Improvements to prioritize.
	 * @return array Prioritized improvements.
	 */
	private function prioritize_improvements( $improvements ) {
		$priority_order = array();

		// High impact, low effort first.
		foreach ( $improvements as $key => $improvement ) {
			if ( 'high' === $improvement['impact'] && 'low' === $improvement['effort'] ) {
				$priority_order[] = array_merge(
					$improvement,
					array(
						'key'      => $key,
						'priority' => 1,
					)
				);
			}
		}

		// High impact, medium effort.
		foreach ( $improvements as $key => $improvement ) {
			if ( 'high' === $improvement['impact'] && 'medium' === $improvement['effort'] ) {
				$priority_order[] = array_merge(
					$improvement,
					array(
						'key'      => $key,
						'priority' => 2,
					)
				);
			}
		}

		// Medium impact, low effort.
		foreach ( $improvements as $key => $improvement ) {
			if ( 'medium' === $improvement['impact'] && 'low' === $improvement['effort'] ) {
				$priority_order[] = array_merge(
					$improvement,
					array(
						'key'      => $key,
						'priority' => 3,
					)
				);
			}
		}

		return $priority_order;
	}

	/**
	 * Assess tone appropriateness
	 *
	 * @param string $tone Dominant tone.
	 * @return string Appropriateness assessment.
	 */
	private function assess_tone_appropriateness( $tone ) {
		switch ( $tone ) {
			case 'formal':
				return __( 'Appropriate for professional contexts', 'alt-audit' );
			case 'casual':
				return __( 'Good for general web content', 'alt-audit' );
			case 'descriptive':
				return __( 'Excellent for detailed imagery', 'alt-audit' );
			case 'objective':
				return __( 'Good for accessibility', 'alt-audit' );
			default:
				return __( 'Neutral tone is generally appropriate', 'alt-audit' );
		}
	}
}
