<?php
/**
 * Quality Result Model
 *
 * @package AltAudit
 * @since 1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Quality Result Model class
 *
 * Represents a quality assessment result with all associated data.
 * Provides structure and validation for quality result data.
 *
 * @since 1.0.0
 */
class Altaudit82ai_Quality_Result_Model {

	/**
	 * Result data
	 *
	 * @var array
	 */
	private $data;

	/**
	 * Default result structure
	 *
	 * @var array
	 */
	private $default_structure = array(
		'score'               => 0,
		'accessibility_score' => 0,
		'status'              => 'unknown',
		'level'               => 'unknown',
		'priority'            => 'low',
		'length'              => 0,
		'word_count'          => 0,
		'descriptive_words'   => 0,
		'structure_score'     => 0,
		'issues'              => array(),
		'suggestions'         => array(),
		'achievements'        => array(),
		'wcag_compliance'     => array(
			'level_a'   => false,
			'level_aa'  => false,
			'level_aaa' => false,
			'issues'    => array(),
		),
		'best_practices'      => array(),
		'action_required'     => false,
		'auto_fixable'        => false,
		'message'             => '',
		'checker'             => '',
		'assessed_at'         => '',
	);

	/**
	 * Constructor
	 *
	 * @param array $data Initial result data.
	 */
	public function __construct( $data = array() ) {
		$this->data = wp_parse_args( $data, $this->default_structure );
		$this->validate_data();
	}

	/**
	 * Get result data
	 *
	 * @param string $key Optional specific data key.
	 * @return mixed Full data array or specific value.
	 */
	public function get( $key = null ) {
		if ( null === $key ) {
			return $this->data;
		}

		return $this->data[ $key ] ?? null;
	}

	/**
	 * Set result data
	 *
	 * @param string $key   Data key.
	 * @param mixed  $value Data value.
	 * @return void
	 */
	public function set( $key, $value ) {
		$this->data[ $key ] = $value;
		$this->validate_data();
	}

	/**
	 * Update multiple data values
	 *
	 * @param array $data Data to update.
	 * @return void
	 */
	public function update( $data ) {
		$this->data = wp_parse_args( $data, $this->data );
		$this->validate_data();
	}

	/**
	 * Get the overall quality score
	 *
	 * @return int Quality score (0-100).
	 */
	public function get_score() {
		return intval( $this->data['score'] );
	}

	/**
	 * Get the accessibility score
	 *
	 * @return int Accessibility score (0-100).
	 */
	public function get_accessibility_score() {
		return intval( $this->data['accessibility_score'] );
	}

	/**
	 * Get the quality status
	 *
	 * @return string Quality status.
	 */
	public function get_status() {
		return $this->data['status'];
	}

	/**
	 * Get quality level
	 *
	 * @return string Quality level.
	 */
	public function get_level() {
		return $this->data['level'];
	}

	/**
	 * Get priority level
	 *
	 * @return string Priority level.
	 */
	public function get_priority() {
		return $this->data['priority'];
	}

	/**
	 * Get issues array
	 *
	 * @return array Issues found.
	 */
	public function get_issues() {
		return (array) $this->data['issues'];
	}

	/**
	 * Get suggestions array
	 *
	 * @return array Improvement suggestions.
	 */
	public function get_suggestions() {
		return (array) $this->data['suggestions'];
	}

	/**
	 * Get WCAG compliance data
	 *
	 * @return array WCAG compliance status.
	 */
	public function get_wcag_compliance() {
		return (array) $this->data['wcag_compliance'];
	}

	/**
	 * Check if action is required
	 *
	 * @return bool True if action is required.
	 */
	public function requires_action() {
		return (bool) $this->data['action_required'];
	}

	/**
	 * Check if auto-fixable
	 *
	 * @return bool True if auto-fixable.
	 */
	public function is_auto_fixable() {
		return (bool) $this->data['auto_fixable'];
	}

	/**
	 * Add an issue
	 *
	 * @param string $issue Issue description.
	 * @return void
	 */
	public function add_issue( $issue ) {
		if ( ! in_array( $issue, $this->data['issues'], true ) ) {
			$this->data['issues'][] = $issue;
		}
	}

	/**
	 * Add a suggestion
	 *
	 * @param string $suggestion Suggestion description.
	 * @return void
	 */
	public function add_suggestion( $suggestion ) {
		if ( ! in_array( $suggestion, $this->data['suggestions'], true ) ) {
			$this->data['suggestions'][] = $suggestion;
		}
	}

	/**
	 * Add an achievement
	 *
	 * @param string $achievement Achievement description.
	 * @return void
	 */
	public function add_achievement( $achievement ) {
		if ( ! in_array( $achievement, $this->data['achievements'], true ) ) {
			$this->data['achievements'][] = $achievement;
		}
	}

	/**
	 * Set WCAG compliance level
	 *
	 * @param string $level WCAG level (a, aa, aaa).
	 * @param bool   $compliant Whether compliant.
	 * @return void
	 */
	public function set_wcag_compliance( $level, $compliant ) {
		$level_key = 'level_' . strtolower( $level );
		if ( isset( $this->data['wcag_compliance'][ $level_key ] ) ) {
			$this->data['wcag_compliance'][ $level_key ] = (bool) $compliant;
		}
	}

	/**
	 * Add WCAG compliance issue
	 *
	 * @param string $issue WCAG compliance issue.
	 * @return void
	 */
	public function add_wcag_issue( $issue ) {
		if ( ! in_array( $issue, $this->data['wcag_compliance']['issues'], true ) ) {
			$this->data['wcag_compliance']['issues'][] = $issue;
		}
	}

	/**
	 * Check if result meets quality threshold
	 *
	 * @param int $threshold Quality threshold (default: 70).
	 * @return bool True if meets threshold.
	 */
	public function meets_threshold( $threshold = 70 ) {
		return $this->get_accessibility_score() >= $threshold;
	}

	/**
	 * Get result summary
	 *
	 * @return array Result summary.
	 */
	public function get_summary() {
		return array(
			'score'           => $this->get_score(),
			'status'          => $this->get_status(),
			'level'           => $this->get_level(),
			'priority'        => $this->get_priority(),
			'issues_count'    => count( $this->get_issues() ),
			'has_suggestions' => ! empty( $this->get_suggestions() ),
			'requires_action' => $this->requires_action(),
			'wcag_compliant'  => $this->is_wcag_compliant(),
		);
	}

	/**
	 * Check overall WCAG compliance
	 *
	 * @param string $level Minimum WCAG level to check (default: 'aa').
	 * @return bool True if compliant.
	 */
	public function is_wcag_compliant( $level = 'aa' ) {
		$wcag      = $this->get_wcag_compliance();
		$level_key = 'level_' . strtolower( $level );

		return $wcag[ $level_key ] ?? false;
	}

	/**
	 * Convert to array for JSON serialization
	 *
	 * @return array Result data as array.
	 */
	public function to_array() {
		return $this->data;
	}

	/**
	 * Convert to JSON string
	 *
	 * @return string JSON representation.
	 */
	public function to_json() {
		return wp_json_encode( $this->data );
	}

	/**
	 * Create from JSON string
	 *
	 * @param string $json JSON string.
	 * @return Altaudit82ai_Quality_Result_Model|null Result model or null on failure.
	 */
	public static function from_json( $json ) {
		$data = json_decode( $json, true );
		if ( JSON_ERROR_NONE === json_last_error() && is_array( $data ) ) {
			return new self( $data );
		}

		return null;
	}

	/**
	 * Validate result data
	 *
	 * @return void
	 */
	private function validate_data() {
		// Ensure score is within valid range.
		$this->data['score']               = max( 0, min( 100, intval( $this->data['score'] ) ) );
		$this->data['accessibility_score'] = max( 0, min( 100, intval( $this->data['accessibility_score'] ) ) );

		// Ensure arrays are actually arrays.
		$array_fields = array( 'issues', 'suggestions', 'achievements', 'best_practices' );
		foreach ( $array_fields as $field ) {
			if ( ! is_array( $this->data[ $field ] ) ) {
				$this->data[ $field ] = array();
			}
		}

		// Ensure WCAG compliance has correct structure.
		if ( ! is_array( $this->data['wcag_compliance'] ) ) {
			$this->data['wcag_compliance'] = $this->default_structure['wcag_compliance'];
		} else {
			$this->data['wcag_compliance'] = wp_parse_args(
				$this->data['wcag_compliance'],
				$this->default_structure['wcag_compliance']
			);
		}

		// Ensure boolean fields are actually boolean.
		$boolean_fields = array( 'action_required', 'auto_fixable' );
		foreach ( $boolean_fields as $field ) {
			$this->data[ $field ] = (bool) $this->data[ $field ];
		}

		// Set assessed_at if not set.
		if ( empty( $this->data['assessed_at'] ) ) {
			$this->data['assessed_at'] = current_time( 'mysql' );
		}
	}
}
