<?php
/**
 * Quality Criteria Model
 *
 * @package AltAudit
 * @since 1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Quality Criteria Model class
 *
 * Manages quality criteria definitions and scoring rules.
 * Provides data structure for quality assessment standards.
 *
 * @since 1.0.0
 */
class Altaudit82ai_Quality_Criteria_Model {

	/**
	 * Quality status definitions
	 *
	 * @var array
	 */
	private $quality_statuses = array(
		'missing'    => array(
			'label'       => 'Missing',
			'description' => 'Alt text is completely missing',
			'priority'    => 'critical',
			'min_score'   => 0,
			'max_score'   => 0,
			'color'       => '#dc3545',
		),
		'decorative' => array(
			'label'       => 'Decorative',
			'description' => 'Image appears to be decorative',
			'priority'    => 'low',
			'min_score'   => 0,
			'max_score'   => 30,
			'color'       => '#6c757d',
		),
		'weak'       => array(
			'label'       => 'Weak',
			'description' => 'Alt text is present but could be improved',
			'priority'    => 'medium',
			'min_score'   => 31,
			'max_score'   => 54,
			'color'       => '#fd7e14',
		),
		'good'       => array(
			'label'       => 'Good',
			'description' => 'Alt text is good and provides useful description',
			'priority'    => 'maintain',
			'min_score'   => 55,
			'max_score'   => 79,
			'color'       => '#198754',
		),
		'excellent'  => array(
			'label'       => 'Excellent',
			'description' => 'Alt text is excellent and highly accessible',
			'priority'    => 'maintain',
			'min_score'   => 80,
			'max_score'   => 100,
			'color'       => '#0d6efd',
		),
	);

	/**
	 * Scoring criteria
	 *
	 * @var array
	 */
	private $scoring_criteria = array(
		'length'          => array(
			'optimal_min'    => 30,
			'optimal_max'    => 150,
			'acceptable_min' => 15,
			'acceptable_max' => 200,
			'max_points'     => 30,
		),
		'word_count'      => array(
			'optimal_min'    => 5,
			'optimal_max'    => 20,
			'acceptable_min' => 3,
			'acceptable_max' => 30,
			'max_points'     => 25,
		),
		'descriptiveness' => array(
			'min_descriptive_words'   => 2,
			'good_descriptive_words'  => 1,
			'basic_descriptive_words' => 1,
			'max_points'              => 20,
		),
		'structure'       => array(
			'proper_capitalization'    => 5,
			'no_file_extensions'       => 5,
			'no_redundant_phrases'     => 5,
			'proper_punctuation'       => 5,
			'no_excessive_punctuation' => 5,
			'max_points'               => 25,
		),
	);

	/**
	 * WCAG compliance rules
	 *
	 * @var array
	 */
	private $wcag_rules = array(
		'1.1.1' => array(
			'level'       => 'A',
			'title'       => 'Non-text Content',
			'description' => 'All non-text content has a text alternative',
			'check'       => 'alt_text_exists',
		),
		'2.4.4' => array(
			'level'       => 'AA',
			'title'       => 'Link Purpose (In Context)',
			'description' => 'Link purpose can be determined from link text alone',
			'check'       => 'link_context',
		),
		'1.4.5' => array(
			'level'       => 'AA',
			'title'       => 'Images of Text',
			'description' => 'If technologies allow visual presentation, text is used instead of images',
			'check'       => 'text_image_check',
		),
	);

	/**
	 * Get quality status definitions
	 *
	 * @return array Quality status definitions.
	 */
	public function get_quality_statuses() {
		return $this->quality_statuses;
	}

	/**
	 * Get quality status by name
	 *
	 * @param string $status Status name.
	 * @return array|null Status definition or null if not found.
	 */
	public function get_quality_status( $status ) {
		return $this->quality_statuses[ $status ] ?? null;
	}

	/**
	 * Get status by score
	 *
	 * @param int $score Quality score.
	 * @return string Status name.
	 */
	public function get_status_by_score( $score ) {
		foreach ( $this->quality_statuses as $status => $definition ) {
			if ( $score >= $definition['min_score'] && $score <= $definition['max_score'] ) {
				return $status;
			}
		}

		return 'unknown';
	}

	/**
	 * Get scoring criteria
	 *
	 * @return array Scoring criteria.
	 */
	public function get_scoring_criteria() {
		return $this->scoring_criteria;
	}

	/**
	 * Get specific scoring criteria
	 *
	 * @param string $criteria Criteria name.
	 * @return array|null Criteria definition or null if not found.
	 */
	public function get_criteria( $criteria ) {
		return $this->scoring_criteria[ $criteria ] ?? null;
	}

	/**
	 * Get WCAG compliance rules
	 *
	 * @return array WCAG rules.
	 */
	public function get_wcag_rules() {
		return $this->wcag_rules;
	}

	/**
	 * Get specific WCAG rule
	 *
	 * @param string $rule_id Rule ID (e.g., '1.1.1').
	 * @return array|null Rule definition or null if not found.
	 */
	public function get_wcag_rule( $rule_id ) {
		return $this->wcag_rules[ $rule_id ] ?? null;
	}

	/**
	 * Get descriptive word patterns
	 *
	 * @return array Descriptive word patterns.
	 */
	public function get_descriptive_patterns() {
		// phpcs:disable Generic.Files.LineLength.TooLong -- Regex patterns cannot be split.
		return array(
			'colors'           => array(
				'pattern' => '/\b(red|blue|green|yellow|orange|purple|black|white|gray|grey|brown|pink|'
					. 'violet|indigo|cyan|magenta|maroon|navy|olive|teal|silver|gold|beige|cream|tan|'
					. 'burgundy|crimson|scarlet|turquoise|coral|lavender|ivory|charcoal)\b/i',
				'weight'  => 1.0,
			),
			'sizes'            => array(
				'pattern' => '/\b(large|small|big|tiny|huge|massive|little|medium|tall|short|wide|'
					. 'narrow|long|thick|thin|spacious|compact|miniature|giant|oversized)\b/i',
				'weight'  => 1.0,
			),
			'shapes'           => array(
				'pattern' => '/\b(round|square|circular|rectangular|triangular|oval|curved|straight|'
					. 'angular|spherical|flat|arched|domed|cylindrical)\b/i',
				'weight'  => 1.0,
			),
			'visual_qualities' => array(
				'pattern' => '/\b(bright|dark|light|shadowy|sunny|cloudy|clear|blurry|sharp|soft|'
					. 'vivid|muted|illuminated|glowing|colorful|monochrome|vibrant|faded|dramatic|scenic)\b/i',
				'weight'  => 1.2,
			),
			'emotions'         => array(
				'pattern' => '/\b(happy|sad|smiling|frowning|serious|excited|calm|angry|peaceful|'
					. 'joyful|content|cheerful|relaxed|focused|thoughtful)\b/i',
				'weight'  => 1.3,
			),
			'materials'        => array(
				'pattern' => '/\b(wooden|metal|glass|plastic|fabric|leather|stone|brick|concrete|'
					. 'ceramic|steel|iron|copper|brass|marble|granite|velvet|silk|cotton|wool|paper|cardboard)\b/i',
				'weight'  => 1.1,
			),
			'textures'         => array(
				'pattern' => '/\b(smooth|rough|glossy|matte|textured|polished|bumpy|silky|coarse|'
					. 'fine|shiny|rustic|sleek|elegant|modern|vintage|antique|ornate|decorative|patterned)\b/i',
				'weight'  => 1.2,
			),
			'lighting'         => array(
				'pattern' => '/\b(bright|dim|natural|artificial|warm|cool|backlit|spotlight|ambient|'
					. 'sunset|sunrise|night|evening|daytime|dawn|dusk|twilight|moonlit|candlelit)\b/i',
				'weight'  => 1.3,
			),
			'objects'          => array(
				'pattern' => '/\b(building|house|car|tree|flower|animal|person|people|man|woman|child|'
					. 'street|road|bridge|mountain|river|lake|ocean|beach|forest|garden|park|city|town|'
					. 'village|desk|chair|table|book|phone|computer|window|door|wall|ceiling|floor|roof|stairs)\b/i',
				'weight'  => 1.0,
			),
			'scene_elements'   => array(
				'pattern' => '/\b(background|foreground|center|corner|edge|side|top|bottom|left|right|'
					. 'front|back|interior|exterior|outdoor|indoor|landscape|portrait|aerial|closeup|panoramic)\b/i',
				'weight'  => 1.1,
			),
			'actions'          => array(
				'pattern' => '/\b(standing|sitting|walking|running|holding|showing|displaying|featuring|'
					. 'depicting|wearing|carrying|pointing|looking|facing|crossing|moving|resting|hanging|floating|flying)\b/i',
				'weight'  => 1.2,
			),
		);
		// phpcs:enable Generic.Files.LineLength.TooLong
	}

	/**
	 * Get redundant phrases to avoid
	 *
	 * @return array Redundant phrases.
	 */
	public function get_redundant_phrases() {
		return array(
			'image of',
			'picture of',
			'photo of',
			'graphic of',
			'screenshot of',
			'illustration of',
			'drawing of',
			'diagram of',
			'chart of',
			'graph of',
		);
	}

	/**
	 * Get file extension patterns
	 *
	 * @return string File extension regex pattern.
	 */
	public function get_file_extension_pattern() {
		return '/\.(jpg|jpeg|png|gif|webp|svg|bmp|tiff|ico)$/i';
	}

	/**
	 * Get quality thresholds
	 *
	 * @return array Quality thresholds.
	 */
	public function get_quality_thresholds() {
		return array(
			'minimum_acceptable' => 40,
			'good_quality'       => 55,
			'excellent_quality'  => 80,
			'perfect_score'      => 100,
		);
	}

	/**
	 * Get context-specific criteria
	 *
	 * @param string $context_type Context type (e.g., 'product', 'article', 'navigation').
	 * @return array Context-specific criteria.
	 */
	public function get_context_criteria( $context_type ) {
		$context_criteria = array(
			'product'    => array(
				'required_elements'  => array( 'product_name', 'key_features', 'visual_appearance' ),
				'length_preference'  => array(
					'min' => 60,
					'max' => 120,
				),
				'descriptive_weight' => 1.2,
			),
			'article'    => array(
				'required_elements'  => array( 'relevance_to_content', 'visual_context' ),
				'length_preference'  => array(
					'min' => 40,
					'max' => 100,
				),
				'descriptive_weight' => 1.0,
			),
			'navigation' => array(
				'required_elements'  => array( 'navigation_purpose', 'destination' ),
				'length_preference'  => array(
					'min' => 20,
					'max' => 60,
				),
				'descriptive_weight' => 0.8,
			),
			'decorative' => array(
				'required_elements'  => array(),
				'length_preference'  => array(
					'min' => 0,
					'max' => 0,
				),
				'descriptive_weight' => 0.0,
			),
		);

		return $context_criteria[ $context_type ] ?? $context_criteria['article'];
	}

	/**
	 * Get accessibility guidelines
	 *
	 * @return array Accessibility guidelines.
	 */
	public function get_accessibility_guidelines() {
		return array(
			'general'        => array(
				'Be concise but descriptive',
				'Focus on the most important visual information',
				'Consider the context in which the image appears',
				'Avoid redundant phrases like "image of"',
				'Start with the most important information',
			),
			'wcag_2_2'       => array(
				'Provide text alternatives for non-text content (1.1.1)',
				'Ensure alt text is meaningful and contextual',
				'Keep descriptions under 150 characters when possible',
				'Use empty alt="" for purely decorative images',
			),
			'best_practices' => array(
				'Write for your audience',
				'Be accurate and truthful',
				'Avoid subjective interpretations',
				'Include relevant details that serve the content purpose',
				'Test with screen readers when possible',
			),
		);
	}
}
