<?php
/**
 * Quality Controller
 *
 * @package AltAudit
 * @since 1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Quality Controller class
 *
 * Handles HTTP requests and coordinates quality assessment operations.
 * Acts as the main entry point for quality-related functionality.
 *
 * @since 1.0.0
 */
class Altaudit82ai_Quality_Controller {


	/**
	 * Quality service
	 *
	 * @var Altaudit82ai_Quality_Service
	 */
	private $quality_service;

	/**
	 * Container instance
	 *
	 * @var Altaudit82ai_Container
	 */
	private $container;

	/**
	 * Constructor
	 *
	 * @param Altaudit82ai_Container $container Container instance.
	 */
	public function __construct( ?Altaudit82ai_Container $container = null ) {
		$this->container = $container;
	}

	/**
	 * Get quality service instance (lazy loading)
	 *
	 * @return Altaudit82ai_Quality_Service|null
	 */
	private function get_quality_service() {
		if ( null === $this->quality_service && $this->container ) {
			$this->quality_service = $this->container->get( 'quality_service' );
		}
		return $this->quality_service;
	}

	/**
	 * Initialize controller hooks
	 *
	 * @return void
	 */
	public function init() {
		add_action( 'wp_ajax_altaudit82ai_check_quality', array( $this, 'handle_quality_check' ) );
		add_action( 'wp_ajax_altaudit82ai_batch_quality', array( $this, 'handle_batch_quality' ) );
		add_action( 'wp_ajax_altaudit82ai_quality_suggestions', array( $this, 'handle_quality_suggestions' ) );
		add_action( 'wp_ajax_altaudit82ai_quality_stats', array( $this, 'handle_quality_stats' ) );
	}

	/**
	 * Handle single quality check request
	 *
	 * @return void
	 */
	public function handle_quality_check() {
		// Verify nonce.
		if ( ! check_ajax_referer( 'altaudit82ai_nonce', 'nonce', false ) ) {
			wp_send_json_error( esc_html__( 'Security check failed', 'alt-audit' ) );
		}

		// Check user permissions.
		if ( ! current_user_can( 'edit_posts' ) ) {
			wp_send_json_error( esc_html__( 'Insufficient permissions', 'alt-audit' ) );
		}

		// Validate and sanitize input.
		if ( ! isset( $_POST['alt_text'] ) ) {
			wp_send_json_error( esc_html__( 'Missing alt text parameter', 'alt-audit' ) );
		}

		$alt_text = sanitize_text_field( wp_unslash( $_POST['alt_text'] ) );
		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Custom sanitization via sanitize_context_array method.
		$context = isset( $_POST['context'] ) ? $this->sanitize_context_array( wp_unslash( $_POST['context'] ) ) : array();

		$quality_service = $this->get_quality_service();
		if ( ! $quality_service ) {
			wp_send_json_error( esc_html__( 'Quality service not available', 'alt-audit' ) );
		}

		try {
			$result = $quality_service->assess_quality( $alt_text, $context );
			wp_send_json_success( $result );
		} catch ( Exception $e ) {
			wp_send_json_error( esc_html__( 'Quality assessment failed', 'alt-audit' ) );
		}
	}

	/**
	 * Handle batch quality assessment request
	 *
	 * @return void
	 */
	public function handle_batch_quality() {
		// Verify nonce.
		if ( ! check_ajax_referer( 'altaudit82ai_nonce', 'nonce', false ) ) {
			wp_send_json_error( esc_html__( 'Security check failed', 'alt-audit' ) );
		}

		// Check user permissions.
		if ( ! current_user_can( 'edit_posts' ) ) {
			wp_send_json_error( esc_html__( 'Insufficient permissions', 'alt-audit' ) );
		}

		// Validate and sanitize input.
		if ( ! isset( $_POST['alt_texts'] ) ) {
			wp_send_json_error( esc_html__( 'Missing alt texts parameter', 'alt-audit' ) );
		}

		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Custom sanitization via sanitize_alt_texts_array method.
		$alt_texts = $this->sanitize_alt_texts_array( wp_unslash( $_POST['alt_texts'] ) );

		if ( empty( $alt_texts ) ) {
			wp_send_json_error( esc_html__( 'No alt texts provided', 'alt-audit' ) );
		}

		$quality_service = $this->get_quality_service();
		if ( ! $quality_service ) {
			wp_send_json_error( esc_html__( 'Quality service not available', 'alt-audit' ) );
		}

		try {
			$results = $quality_service->batch_assess( $alt_texts );
			wp_send_json_success( $results );
		} catch ( Exception $e ) {
			wp_send_json_error( __( 'Batch assessment failed', 'alt-audit' ) );
		}
	}

	/**
	 * Handle quality suggestions request
	 *
	 * @return void
	 */
	public function handle_quality_suggestions() {
		// Verify nonce.
		if ( ! check_ajax_referer( 'altaudit82ai_nonce', 'nonce', false ) ) {
			wp_send_json_error( esc_html__( 'Security check failed', 'alt-audit' ) );
		}

		// Check user permissions.
		if ( ! current_user_can( 'edit_posts' ) ) {
			wp_send_json_error( esc_html__( 'Insufficient permissions', 'alt-audit' ) );
		}

		// Validate and sanitize input.
		if ( ! isset( $_POST['alt_text'] ) ) {
			wp_send_json_error( esc_html__( 'Missing alt text parameter', 'alt-audit' ) );
		}

		$alt_text = sanitize_text_field( wp_unslash( $_POST['alt_text'] ) );
		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Custom sanitization via sanitize_context_array method.
		$context = isset( $_POST['context'] ) ? $this->sanitize_context_array( wp_unslash( $_POST['context'] ) ) : array();

		$quality_service = $this->get_quality_service();
		if ( ! $quality_service ) {
			wp_send_json_error( esc_html__( 'Quality service not available', 'alt-audit' ) );
		}

		try {
			$suggestions = $quality_service->get_suggestions( $alt_text, $context );
			wp_send_json_success( $suggestions );
		} catch ( Exception $e ) {
			wp_send_json_error( __( 'Failed to generate suggestions', 'alt-audit' ) );
		}
	}

	/**
	 * Handle quality statistics request
	 *
	 * @return void
	 */
	public function handle_quality_stats() {
		// Verify nonce.
		if ( ! check_ajax_referer( 'altaudit82ai_nonce', 'nonce', false ) ) {
			wp_send_json_error( __( 'Security check failed', 'alt-audit' ) );
		}

		// Check user permissions.
		if ( ! current_user_can( 'edit_posts' ) ) {
			wp_send_json_error( __( 'Insufficient permissions', 'alt-audit' ) );
		}

		$post_id = isset( $_POST['post_id'] ) ? intval( wp_unslash( $_POST['post_id'] ) ) : 0;

		$quality_service = $this->get_quality_service();
		if ( ! $quality_service ) {
			wp_send_json_error( esc_html__( 'Quality service not available', 'alt-audit' ) );
		}

		try {
			$stats = $quality_service->get_post_quality_stats( $post_id );
			wp_send_json_success( $stats );
		} catch ( Exception $e ) {
			wp_send_json_error( __( 'Failed to generate statistics', 'alt-audit' ) );
		}
	}

	/**
	 * Check alt text quality (public API)
	 *
	 * @param string $alt_text Alt text to check.
	 * @param array  $context  Additional context.
	 * @return array Quality assessment result.
	 */
	public function check_quality( $alt_text, $context = array() ) {
		$quality_service = $this->get_quality_service();
		if ( ! $quality_service ) {
			return array(
				'error' => __( 'Quality service not available', 'alt-audit' ),
			);
		}

		return $quality_service->assess_quality( $alt_text, $context );
	}

	/**
	 * Get quality statistics for a post
	 *
	 * @param int $post_id Post ID.
	 * @return array Quality statistics.
	 */
	public function get_post_quality_stats( $post_id ) {
		$quality_service = $this->get_quality_service();
		if ( ! $quality_service ) {
			return array(
				'error' => __( 'Quality service not available', 'alt-audit' ),
			);
		}

		return $quality_service->get_post_quality_stats( $post_id );
	}

	/**
	 * Compare two alt texts
	 *
	 * @param string $alt_text_1 First alt text.
	 * @param string $alt_text_2 Second alt text.
	 * @return array Comparison result.
	 */
	public function compare_alt_texts( $alt_text_1, $alt_text_2 ) {
		$quality_service = $this->get_quality_service();
		if ( ! $quality_service ) {
			return array(
				'error' => __( 'Quality service not available', 'alt-audit' ),
			);
		}

		return $quality_service->compare_alt_texts( $alt_text_1, $alt_text_2 );
	}

	/**
	 * Check if alt text meets quality threshold
	 *
	 * @param string $alt_text Alt text to check.
	 * @return bool True if meets threshold.
	 */
	public function meets_threshold( $alt_text ) {
		$quality_service = $this->get_quality_service();
		if ( ! $quality_service ) {
			return false;
		}

		return $quality_service->meets_threshold( $alt_text );
	}

	/**
	 * Sanitize context array
	 *
	 * @param array $context Context array to sanitize.
	 * @return array Sanitized context.
	 */
	private function sanitize_context_array( $context ) {
		if ( ! is_array( $context ) ) {
			return array();
		}

		$sanitized    = array();
		$allowed_keys = array( 'image_id', 'post_id', 'type', 'src', 'caption', 'description' );

		foreach ( $context as $key => $value ) {
			if ( in_array( $key, $allowed_keys, true ) ) {
				if ( in_array( $key, array( 'image_id', 'post_id' ), true ) ) {
					$sanitized[ $key ] = intval( $value );
				} else {
					$sanitized[ $key ] = sanitize_text_field( $value );
				}
			}
		}

		return $sanitized;
	}

	/**
	 * Sanitize alt texts array
	 *
	 * @param array $alt_texts Alt texts array to sanitize.
	 * @return array Sanitized alt texts.
	 */
	private function sanitize_alt_texts_array( $alt_texts ) {
		if ( ! is_array( $alt_texts ) ) {
			return array();
		}

		$sanitized = array();
		foreach ( $alt_texts as $alt_text ) {
			if ( is_string( $alt_text ) ) {
				$sanitized[] = sanitize_text_field( $alt_text );
			}
		}

		return $sanitized;
	}
}
